/* vim:set ts=4 sts=4 sw=4 noet fenc=utf-8:

   Copyright 2009 senju@users.sourceforge.jp

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
 */

package jp.sourceforge.rabbitBTS.controllers;

import java.util.Date;

import javax.servlet.http.HttpServletRequest;

import jp.sourceforge.rabbitBTS.Sht;
import jp.sourceforge.rabbitBTS.exceptions.HasNotValidIdException;
import jp.sourceforge.rabbitBTS.exceptions.RabbitBTSException;
import jp.sourceforge.rabbitBTS.models.Account;
import jp.sourceforge.rabbitBTS.services.AccountService;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.validation.BindingResult;
import org.springframework.validation.Validator;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

@Controller
public final class RegisterController extends BaseController implements
		IPublicController {
	private static final String REDIRECT_HOME_HTML = "redirect:/home/";

	@Autowired
	private AccountService accountService;

	@Autowired
	private Validator validator;

	/**
	 * 新規登録画面を表示する。
	 * <p>
	 * ユーザーがgoogleにログインしていない場合、googleのログイン画面にリダイレクトする。
	 * ユーザーがアカウントを持っている場合、ユーザーホームにリダイレクトする。
	 * ユーザーがgoogleにログイン済みでかつアカウントを持っていない時だけ、画面を表示する。
	 * 
	 * @param req
	 * @param map
	 * @return
	 */
	@RequestMapping(method = RequestMethod.GET)
	public String index(HttpServletRequest req, ModelMap map) {
		String ret = null;
		try {
			final Account account = this.accountService.fetchAccount();
			if (account == null) {
				// 正常系
				map.addAttribute(new Account());
				ret = null;
			} else {
				// すでに登録済みの場合ホームへリダイレクト
				Sht.log(this).trace("既に登録済み");
				ret = REDIRECT_HOME_HTML;
			}
		} catch (final HasNotValidIdException e) {
			// googleにログインしてない場合
			Sht.log(this).trace("ログインが必要");
			ret = "redirect:" + Sht.loginUrl(req);
		}
		return ret;
	}

	/**
	 * 新規登録を行う。
	 * 
	 * <p>
	 * 正常系
	 * <ol>
	 * <li>バリデート
	 * <li>Accountを登録
	 * <li>ユーザーホームにリダイレクト
	 * <ol>
	 * <p>
	 * 異常系
	 * <ol>
	 * <li>バリデートに失敗した場合、エラーメッセージ
	 * <li>Account登録に失敗した場合(二重登録等)、ロールバック、エラーメッセージ
	 * </ol>
	 * 
	 * @param account
	 * @param result
	 * @return
	 */
	@RequestMapping(method = RequestMethod.POST)
	public String index(Account account, BindingResult result, ModelMap map) {
		if (!this.isCsrfSafe(result)) {
			return null;
		}

		account.setLastAccess(new Date());

		account.setEmail(Sht.user().getEmail());
		this.validator.validate(account, result);
		if (result.hasErrors()) {
			Sht.log(this).trace("ユーザ登録画面にてバリデートに失敗");
			return null;
		}

		try {
			// nickName重複チェック
			if (this.accountService.getAccountByNickName(account.getNickName()) != null) {
				result.rejectValue("nickName", "Account.nickName[duplicate]");
				Sht.log(this).trace("nickName重複チェックエラー");
				return null;
			}

			this.accountService.registAccount(account);
		} catch (final RabbitBTSException e) {
			Sht.log(this).warn("ユーザー登録で例外発生", e);
			// TODO:resultにセット
			map.addAttribute("errorMessage", "登録に失敗しました。再度やりなおしてください。");
			return null;
		}
		return REDIRECT_HOME_HTML;
	}
}
