#include "TestBoard.h"
#include "../HalfGammon/Board.h"
#include <QTest>
#include <QDebug>
#include <time.h>
#include <boost/random.hpp>

#define		HG_NP		12		//	n[tM |Cg
#define		HG_NS		8		//	n[tM ΐ
#define		NOT_YET		0x8000	//	萔e[u̗vfl vZ

TestBoard::TestBoard(QObject *parent)
	: QObject(parent)
{

}

TestBoard::~TestBoard()
{

}
void TestBoard::testBoard()
{
	Board bd("G B2  W3    B3W3    B3  W2 G");	//	
	qDebug() << bd.position();
	QCOMPARE( bd.position(), QString("G B2. W3. . B3W3. . B3. W2 G") );

	bd.setPosition("G B1  W3B1  B3W3    B3  W2 G");
	QCOMPARE( bd.position(), QString("G B1. W3B1. B3W3. . B3. W2 G") );
	QVERIFY( !bd.doesBlackWin() );
	QVERIFY( !bd.doesWhiteWin() );
	QVERIFY( !bd.canBlackBearOff() );
	QVERIFY( !bd.canWhiteBearOff() );
	QCOMPARE( bd.nBlackGoal(), 0 );
	QCOMPARE( bd.nWhiteGoal(), 0 );
	QCOMPARE( bd.blackTailIndex(), POINT_START - 1 );
	QCOMPARE( bd.whiteTailIndex(), POINT_START - 1 );

	bd.setPosition("G     W3      W3    B3  W2 G");
	QVERIFY( !bd.doesBlackWin() );
	QVERIFY( !bd.doesWhiteWin() );
	QVERIFY( bd.canBlackBearOff() );
	QVERIFY( !bd.canWhiteBearOff() );
	QCOMPARE( bd.nBlackGoal(), 5 );
	QCOMPARE( bd.nWhiteGoal(), 0 );
	QCOMPARE( bd.blackTailIndex(), POINT_GOAL + 3 );
	QCOMPARE( bd.whiteTailIndex(), POINT_START - 1 );

	bd.setPosition("G . . W3. . . . . B3. . .  G");
	QVERIFY( !bd.canBlackBearOff() );
	QVERIFY( bd.canWhiteBearOff() );
}
void TestBoard::testMove()
{
	Board bd("G B2  W3    B3W3    B3  W2 G");	//	
	bd.moveBlack(Move(12, 1));		//	12|Cg̍ЂƂړ
	QCOMPARE( bd.position(), QString("G B1B1W3. . B3W3. . B3. W2 G") );
	bd.moveWhite(Move(reverseIX(10), 2));		//	10|Cg̔12|Cg̍qbg
	QCOMPARE( bd.position(), QString("B1W1B1W2. . B3W3. . B3. W2 G") );
}
//	\擾eXg
void TestBoard::testGetMoves()
{
	Board bd("G B2  W3    B3W3    B3  W2 G");	//	
	QList<Moves> mvsList = bd.whiteMovesList(3, 3);	//	3-3 ɂ\胊Xg擾
	QCOMPARE( mvsList.size(), 2 );
	mvsList = bd.whiteMovesList(2, 2);
	QCOMPARE( mvsList.size(), 2 );
	mvsList = bd.whiteMovesList(1, 1);
	QCOMPARE( mvsList.size(), 2 );
	mvsList = bd.whiteMovesList(1, 2);	//	1-2 ɂ\胊Xg擾
	QCOMPARE( mvsList.size(), 4 );
	mvsList = bd.whiteMovesList(1, 3);
	QCOMPARE( mvsList.size(), 6 );
	mvsList = bd.whiteMovesList(2, 3);
	QCOMPARE( mvsList.size(), 4 );

	bd.setPosition("G B2  W3                   G");
	Moves mvs = bd.whiteMoves(2);
	QVERIFY( mvs.isEmpty() );

	bd.setPosition("G     W3            B2B2B2W2");
	mvsList = bd.whiteMovesList3(1, 2);
	QVERIFY( mvsList.isEmpty() );
	bd.setPosition("G     W3            B1B2B2W2");
	mvsList = bd.whiteMovesList3(1, 2);
	QVERIFY( mvsList.isEmpty() );
	mvsList = bd.whiteMovesList3(2, 3);
	QCOMPARE( mvsList.size(), 1 );
	bd.setPosition("G     W3            . B2B2W2");
	mvsList = bd.whiteMovesList3(2, 3);
	QCOMPARE( mvsList.size(), 1 );
	bd.setPosition("G     W3            . B1B2W2");
	mvsList = bd.whiteMovesList3(2, 3);
	QCOMPARE( mvsList.size(), 1 );

	bd.setPosition("G . . W3            . B1B2W1");
	mvsList = bd.whiteMovesList3(2, 3);
	QCOMPARE( mvsList.size(), 3 );		//	RŏoꍇAS[߂Qړo邪AQŏoꍇ́ARňړoȂ

	bd.setPosition("G . . . . . . . . W1B2. . W1");
	mvsList = bd.whiteMovesList3(1, 2);
	QCOMPARE( mvsList.size(), 2 );		//	1 or 2 ŏoāAЂƂc̖ڂŐi߂
	bd.setPosition("G . . . . . . . B2W1B2. . W1");
	mvsList = bd.whiteMovesList3(1, 2);
	QCOMPARE( mvsList.size(), 1 );		//	QŏoƂPgȂ̂ŁAPŏoāAQi


	bd.setPosition("G . . B3          B1. . W2 G");
	mvsList = bd.whiteMovesList3(2, 3);
	QCOMPARE( mvsList.size(), 3 );		//	PiQʂj or Q
	bd.setPosition("G . . B3            . . W2 G");
	mvsList = bd.whiteMovesList3(2, 3);
	QCOMPARE( mvsList.size(), 2 );		//	PiPʂj or Q

	bd.setPosition("G . . B3        B2B2W1W1.  G");		//	P肵ړoȂꍇ
	mvsList = bd.whiteMovesList3(1, 2);
	QCOMPARE( mvsList.size(), 1 );

	bd.setPosition("G . . B3W1. . . B2. . W1.  G");		//	̐΂QƁAPgȂꍇ
	mvsList = bd.whiteMovesList3(1, 2);
	QCOMPARE( mvsList.size(), 2 );

	bd.setPosition("G . . B3W1        B1. . .  G");		//	Pڂ̈ړŁAxAIt\ɂȂꍇ
	mvsList = bd.whiteMovesList3(1, 3);
	QCOMPARE( mvsList.size(), 1 );
	mvs = mvsList[0];
	QCOMPARE( mvs.size(), 2 );

	bd.setPosition("G B1. B3W1        B1. . .  G");		//	Sďオꍇ́AP̂ݕԂ
	mvsList = bd.whiteMovesList3(2, 3);
	QCOMPARE( mvsList.size(), 1 );

	bd.setPosition("G . . B3. . . B2. . . . W1 G");		//	Е̖ڂgȂꍇ́A傫ق̖ڂgȂȂƂ
	mvsList = bd.whiteMovesList3(2, 3);
	QCOMPARE( mvsList.size(), 1 );
	mvs = mvsList[0];
	QCOMPARE( mvs.size(), 1 );
	QCOMPARE( (int)mvs[0].m_d, 3 );		//	傫̖

	bd.setPosition("G B2  W3    B3W3    B3  W2 G");	//	
	mvsList = bd.whiteMovesList3(1, 1);
	QCOMPARE( mvsList.size(), 2 );
	mvsList = bd.whiteMovesList3(2, 2);
	QCOMPARE( mvsList.size(), 2 );
	mvsList = bd.whiteMovesList3(3, 3);
	QCOMPARE( mvsList.size(), 2 );

	bd.setPosition("G B2W1W1    B3W3      B3W2 G");
	mvsList = bd.whiteMovesList3(1, 1);
	QCOMPARE( mvsList.size(), 1 );
	mvs = mvsList[0];
	QCOMPARE( mvs.size(), 1 );
	QCOMPARE( (int)mvs[0].m_src, 3 );
	bd.setPosition("G B2  W2    B3W3      B3W2 G");
	mvsList = bd.whiteMovesList3(1, 1);
	QCOMPARE( mvsList.size(), 1 );
	mvs = mvsList[0];
	QCOMPARE( mvs.size(), 2 );
	QCOMPARE( (int)mvs[0].m_src, 3 );

	bd.setPosition("B1W4B4B1. . . . . . B2. . W1");		//	SPR#0205
	mvsList = bd.whiteMovesList3(1, 2);
	QCOMPARE( mvsList.size(), 1 );
	mvs = mvsList[0];
	QCOMPARE( mvs.size(), 1 );
	QCOMPARE( (int)mvs[0].m_d, 2 );		//	K傫ڂg

	bd.setPosition("G . . W3B2. B2W5. . B3B1.  G");		//	SPR#0218
	mvsList = bd.whiteMovesList3(1, 3);
	QCOMPARE( mvsList.size(), 1 );

}

void TestBoard::benchRandomGame()
{
#ifdef _DEBUG
	const int N_LOOP = 100;
#else
	const int N_LOOP = 1000;
#endif
	int gameCount = 0;
	int turnCount = 0;
	int mvsCount = 0;
	QBENCHMARK_ONCE {
		for(int i = 0; i < N_LOOP; ++i) {
			++gameCount;
			Board bd("G B2  W3    B3W3    B3  W2 G");		//	
			bool blackTurn = true;
			for(bool init = true;;) {
				//qDebug() << bd.position();
				if( blackTurn )
					bd.swapBW();
				int d1, d2;
				do {
					d1 = (qrand() % 3) + 1;
					d2 = (qrand() % 3) + 1;
				} while( init && d1 == d2 );
				init = false;
				QList<Moves> mvsList = bd.whiteMovesList(d1, d2);
				if( !mvsList.isEmpty() ) {
					mvsCount += mvsList.size();
					++turnCount;
					QList<Move> mvs = mvsList[qrand() % mvsList.size()];
					foreach(Move mv, mvs)
						bd.moveWhite(mv);
				}
				if( blackTurn )
					bd.swapBW();
				if( bd.doesBlackWin() || bd.doesWhiteWin() ) break;
				blackTurn = !blackTurn;
			}
		}
	}
	qDebug() << "gameCount = " << gameCount;
	qDebug() << "ave turn count = " << (double)turnCount / gameCount;
	qDebug() << "ave mvs count = " << (double)mvsCount / turnCount;
}
void TestBoard::benchRandomGame3()
{
#ifdef _DEBUG
	const int N_LOOP = 100;
#else
	const int N_LOOP = 1000;
#endif
	int gameCount = 0;
	int turnCount = 0;
	int mvsCount = 0;
	QBENCHMARK_ONCE {
		for(int i = 0; i < N_LOOP; ++i) {
			++gameCount;
			Board bd("G B2  W3    B3W3    B3  W2 G");		//	
			bool blackTurn = true;
			for(bool init = true;;) {
				//qDebug() << bd.position();
				if( blackTurn )
					bd.swapBW();
				int d1, d2;
				do {
					d1 = (qrand() % 3) + 1;
					d2 = (qrand() % 3) + 1;
				} while( init && d1 == d2 );
				init = false;
				QString text = bd.position();
				Board bd2(bd);
				QList<Moves> mvsList = bd.whiteMovesList3(d1, d2);
				if( bd.nBlack() != N_PIECE || bd.nWhite() != N_PIECE )
					mvsList = bd2.whiteMovesList3(d1, d2);
				Q_ASSERT( bd.nBlack() == N_PIECE );
				Q_ASSERT( bd.nWhite() == N_PIECE );
				QString pos = bd.position();
				if( !mvsList.isEmpty() ) {
					mvsCount += mvsList.size();
					++turnCount;
					QList<Move> mvs = mvsList[qrand() % mvsList.size()];
					foreach(Move mv, mvs)
						bd.moveWhite(mv);
				}
				if( blackTurn )
					bd.swapBW();
				if( bd.doesBlackWin() || bd.doesWhiteWin() ) break;
				blackTurn = !blackTurn;
			}
		}
	}
	qDebug() << "gameCount = " << gameCount;
	qDebug() << "ave turn count = " << (double)turnCount / gameCount;
	qDebug() << "ave mvs count = " << (double)mvsCount / turnCount;
}
//	]֐eXg
void TestBoard::testEval5()
{
	//	Ԃ̉l 400 ƂĂ
	Board bd("G >2  <3    >3<3    >3  <2 G");	//	
	QVERIFY( bd.eval5() > 0 );		//	Ԃ̕L
	bd.setPosition("G >2<1<1<1  >3<3    >3  <2 G");	//	肩qbg\
	QVERIFY( bd.eval5() < 400 );
	bd.setPosition("G >2  <3    >3<3  >1>1>1<2 G");	//	qbgo
	QVERIFY( bd.eval5() > 400 );
	Board bd1, bd2;
	//               G 121110 9 8 7 6 5 4 3 2 1 G
	bd1.setPosition(">1<1<2<2<2    <1        >7 G");	//	12ubg̏ꍇ
	bd2.setPosition(">1<2<2<2      <2        >7 G");	//	Ci[SăubNĂꍇ
	QVERIFY( bd1.eval5() < bd2.eval5() );
	//               G 121110 9 8 7 6 5 4 3 2 1 G
	//               >1    <3<3<2            >7 G
	bd1.setPosition(">1  <2<2<2<2            >7 G");	//	vCOɐi߂
	bd2.setPosition(">1    <4<4              >7 G");	//	ubNς
	QVERIFY( bd1.eval5() > bd2.eval5() );

	bd1.setPosition("G B2B2. . W2W2W2W2. . B2B2 G");	//	sbvAɂ
	QVERIFY( bd1.eval5() > 0 );

	bd1.setPosition("G B2B2. . . W2W2W2W2. B2B2 G");	//	ɂAsbvIɂ͍L
	QVERIFY( bd1.eval5() > 0 );

	bd1.setPosition("G B2B2. . . . W2W2W2W2B2B2 G");	//	ɂAsbvIɂ͍ƗL
	QVERIFY( bd1.eval5() > 0 );

	//               B1W2. B1. . . . . . B2. B4 G
	bd1.setPosition("G W2. B1. . . . . . B2. B5 G");	//	o[̐΂SړiɂƂėLj
	bd2.setPosition("G W2. . . . B1. . B1B2. B4 G");	//	obNM
	QVERIFY( bd1.eval5() > bd2.eval5() );
}
void TestBoard::testEval6()
{
	int ev1, ev2;
	//	Ԃ̉l 400 ƂĂ
	Board bd("G >2  <3    >3<3    >3  <2 G");	//	
	QVERIFY( bd.eval6() > 0 );		//	Ԃ̕L
	bd.setPosition("G >2<1<1<1  >3<3    >3  <2 G");	//	肩qbg\
	QVERIFY( bd.eval6() < 400 );
	bd.setPosition("G >2  <3    >3<3  >1>1>1<2 G");	//	qbgo
	QVERIFY( bd.eval6() > 400 );
	Board bd1, bd2;
	//               G 121110 9 8 7 6 5 4 3 2 1 G
	bd1.setPosition(">1<1<2<2<2    <1        >7 G");	//	12ubg̏ꍇ
	bd2.setPosition(">1<2<2<2      <2        >7 G");	//	Ci[SăubNĂꍇ
	QVERIFY( bd1.eval6() < bd2.eval6() );
	//               G 121110 9 8 7 6 5 4 3 2 1 G
	//               >1    <3<3<2            >7 G
	bd1.setPosition(">1  <2<2<2<2            >7 G");	//	vCOɐi߂
	bd2.setPosition(">1    <4<4              >7 G");	//	ubNς
	QVERIFY( bd1.eval6() > bd2.eval6() );

	bd1.setPosition("G B2B2. . W2W2W2W2. . B2B2 G");	//	sbvAɂ
	QVERIFY( bd1.eval6() > 0 );

	bd1.setPosition("G B2B2. . . W2W2W2W2. B2B2 G");	//	ɂAsbvIɂ͍L
	QVERIFY( bd1.eval6() > 0 );

	bd1.setPosition("G B2B2. . . . W2W2W2W2B2B2 G");	//	ɂAsbvIɂ͍ƗL
	QVERIFY( bd1.eval6() > 0 );

	//               G W4. W2. . . . . . W1. B2W1
	bd1.setPosition("G W4. W2W1. . W1. . . . B2 G");	//	obNM
	bd2.setPosition("G W5. W2. . . . . . W1. B2 G");	//	o[̐΂Sړ
	QVERIFY( (ev1 = bd1.eval6()) > (ev2 = bd2.eval6()) );
#if 1
	//               B1W2. B1. . . . . . B2. B4 G
	bd1.setPosition("G W2. B1. . . . . . B2. B5 G");	//	o[̐΂SړiɂƂėLj
	bd2.setPosition("G W2. . . . B1. . B1B2. B4 G");	//	obNM
	QVERIFY( bd1.eval6() > bd2.eval6() );
#endif

	//	SPR#0187	 G W1.W4. W2         B2B2B3W1	 1-1
	bd1.setPosition("G W6. . . . . . . . . B2B5W1");	//	Ԃ󂯂Ȃꍇ
	bd2.setPosition("G W6. . . . . . . . B2. B3W1");	//	3-3 PA[āA2|Cg󂯂ꍇ
	QVERIFY( (ev1 = bd1.eval6()) < (ev2 = bd2.eval6()) );

	//	SPR#0200	 B1W2. W2W2W1B2W1. . B3. B2 G	 2-2
	bd1.setPosition("G W2B1W2W2W1. W1B1. B4. B2 G");	//	1pnt ɐς܂Ȃ
	bd2.setPosition("G W2B1W2W2W1B2W1. . . . B5 G");	//	1pnt ɐςށiɗLj
	QVERIFY( (ev1 = bd1.eval6()) < (ev2 = bd2.eval6()) );

	//	SPR#0206	 G W3B1W2B1. . . . B1. B1B4 G	 1-1
	//				 G W4W1. W1. . . . W1B2W1B3 G	 1-1
	bd1.setPosition("G W4W1. W1. . . . W1. . B5W1");	//	ubgȂ
	bd2.setPosition("G W4W1. W1. . . . W1. B2B1W1");	//	ubgiɗLj
	QVERIFY( (ev1 = bd1.eval6()) < (ev2 = bd2.eval6()) );

	//	SPR#0207	 B1W2W3W2. . . . . B1B3B1B2 G  2-2
	bd1.setPosition("B1W4W1. . . . . . B1B3B1B2 G");	//	2-* Ńqbgi5/9)
	bd2.setPosition("B1W3. W1. . . . . B1B3B1B2 G");	//	3-*, 1-2 Ńqbg(7/9)
	QVERIFY( (ev1 = eval6black(bd1)) < (ev2 = eval6black(bd2)) );

	//	SPR#0213	 G . W2W5. . B3. . B2B3. W1 G	 1-1
	bd1.setPosition("G . W2W5. . . B3. B2B2B1W1 G");	//	
	bd2.setPosition("G . W2W5. . B1. B2B2B3. W1 G");	//	CN vC
	QVERIFY( (ev1 = eval6(bd1)) > (ev2 = eval6(bd2)) );

	//	SPR#0219	 G W2B1W4. . . . . . . B3B4 G	 1-1
	bd1.setPosition("B1W2W4. . . . . . . . B3B4 G");	//	1pnt 󂯂Ȃ
	bd2.setPosition("B1. W2W2. . . . . . . B3B4 G");	//	1pnt 󂯂
	QVERIFY( (ev1 = eval6black(bd1)) < (ev2 = eval6black(bd2)) );

	//	SPR#0220	 G . . W7. . . . . . . B3B4W1	 3-2
	bd1.setPosition("G . . W7. . . . W1. . B3B4 G");	//	Ōړ
	bd2.setPosition("G W1. W6. . . . . . W1B3B4 G");	//	ŌړȂ
	QVERIFY( (ev1 = eval6black(bd1)) < (ev2 = eval6black(bd2)) );

	//	SPR#0222	 G W5W2. B1W1. . . . . B4B3 G	 1-1
	bd1.setPosition("B1W6W2. . . . . . . . B4B3 G");	//	Ō
	bd2.setPosition("B1W4W3. . . . . . . . B4B3 G");	//	Ō
	QVERIFY( (ev1 = eval6black(bd1)) < (ev2 = eval6black(bd2)) );

	//	SPR#0223	 B1B2W2W5. . W1. . . B4B1.  G	 1-1
	bd1.setPosition("G B3W2W5. . W1. . . B1B4.  G");	//
	bd2.setPosition("G B3W2W5. . W1. . . B2B2B1 G");	//	Ci[ꂢ
	QVERIFY( (ev1 = eval6(bd1)) > (ev2 = eval6(bd2)) );
}
void TestBoard::testEval5Ply1()
{
	Board bd("G W4. W2. . . . . . W1. B2W1");
	//bd.swapBW();
	double eval = 0;
	Moves mvs = bd.whiteMovesEval5PlyD(1, 3, 3, eval);
	foreach(Move mv, mvs)
		bd.moveWhite(mv);
	//bd.swapBW();
	QCOMPARE( bd.white(10), 0 );	//	10|Cg̍ړ
}
void TestBoard::testEval6Ply1()
{
	Board bd("B1W2. B1. . . . . . B2. B4 G");
	bd.swapBW();
	double eval = 0;
	Moves mvs = bd.whiteMovesEvalXPlyD(1, 3, 3, eval, ::eval6);
	foreach(Move mv, mvs)
		bd.moveWhite(mv);
	bd.swapBW();
	QCOMPARE( bd.black(10), 0 );	//	10|Cg̍ړ
}
void statsRandomRollOut(const Board &bd0, int N_LOOP)
{
	int blackPointSum = 0;
	int blackPointSum2 = 0;	//	a
	for(int i = 0; i < N_LOOP; ++i) {
		Board bd(bd0);
		bd.swapBW();		//	ɔԂƂď
		bool blackTurn = true;
		int score = 0;
		for(;;) {
			int d1 = (qrand() % 3) + 1;
			int d2 = (qrand() % 3) + 1;
			Moves mvs = bd.whiteMovesRandom(d1, d2);
			foreach(Move mv, mvs)
				bd.moveWhite(mv);
#if 0
			if( blackTurn ) {
				bd.swapBW();
				qDebug() << bd.position();
				bd.swapBW();
			} else
				qDebug() << bd.position();
#endif
			if( bd.doesWhiteWin() ) {
				if( bd.doesBlackBackGammonLose() )
					score = 3;
				else if( bd.doesBlackGammonLose() )
					score = 2;
				else
					score = 1;
				break;
			}
			blackTurn = !blackTurn;
			bd.swapBW();
		}
		if( blackTurn ) {
			//qDebug() << "black win";
		} else {
			score = -score;
			//qDebug() << "white win";
		}
		blackPointSum += score;
		blackPointSum2 += score * score;
	}
	const double SD = sqrt((blackPointSum2 - (double)blackPointSum * blackPointSum/N_LOOP) / N_LOOP);
	qDebug() << "Random Dice: N = " << N_LOOP << "expect = " << (double)blackPointSum / N_LOOP << "SD = " << SD;
}
int g_dice1 = 1;
int g_dice2 = 1;
void getSeqDice(int &d1, int &d2)
{
	if( ++g_dice1 > 3 ) {
		g_dice1 = 1;
		if( ++g_dice2 > 3 )
			g_dice2 = 1;
	}
	d1 = g_dice1;
	d2 = g_dice2;
}
void makeNotSameSeq()
{
	static int gd1 = 1;
	static int gd2 = 1;
	if( g_dice2 == gd1 && g_dice2 == gd2 )
		getSeqDice(gd1, gd2);
	else {
		gd1 = g_dice1;
		gd2 = g_dice2;
	}
}
int g_seqIndex = -1;
QVector<int> g_vDice;
void getSeqDice2(int &d1, int &d2)
{
	if( g_seqIndex < 0 ) {
		for(int i = 0; i < 9; ++i)
			g_vDice << i;
		g_seqIndex = 0;
	} else {
		if( ++g_seqIndex >= g_vDice.size() ) {
			if( !std::next_permutation(g_vDice.begin(), g_vDice.end()) ) {
				g_vDice.clear();
				for(int i = 0; i < 9; ++i)
					g_vDice << i;
			}
			g_seqIndex = 0;
		}
	}
	d1 = g_vDice[g_seqIndex] % 3 + 1;
	d2 = g_vDice[g_seqIndex] / 3 + 1;
}
void getSeqDice3(int &d1, int &d2)
{
	static QVector<int> v;
	if( v.isEmpty() ) {
		QVector<int> t;
		for(int i = 0; i < 9; ++i)
			t << i;
		for(int i = 0; i < 8; ++i) {
			int r = qrand() % t.size();
			v << t[r];
			t.remove(r);
		}
		v << t[0];
	}
	Q_ASSERT( v[v.size() - 1] >= 0 );
	Q_ASSERT( v[v.size() - 1] < 9 );
	d1 = v[v.size() - 1] % 3 + 1;
	d2 = v[v.size() - 1] / 3 + 1;
	v.pop_back();
}
void statsSeqRollOut(const Board &bd0, int N_LOOP)
{
	int blackPointSum = 0;
	int blackPointSum2 = 0;	//	a
	for(int i = 0; i < N_LOOP; ++i) {
		//	_CX~񐔂ƃ_CX̎vĂꍇւ̑Ώ
		//makeNotSameSeq();		ꂶ_
		//g_dice1 = i % 3 + 1;
		//g_dice2 = i / 3 % 3 + 1;
		Board bd(bd0);
		bd.swapBW();		//	ɔԂƂď
		bool blackTurn = true;
		int score = 0;
		for(;;) {
			int d1, d2;
			getSeqDice3(d1, d2);
			Moves mvs = bd.whiteMovesRandom(d1, d2);
			foreach(Move mv, mvs)
				bd.moveWhite(mv);
#if 0
			if( blackTurn ) {
				bd.swapBW();
				qDebug() << d1 << d2 << bd.position();
				bd.swapBW();
			} else
				qDebug() << d1 << d2 << bd.position();
#endif
			if( bd.doesWhiteWin() ) {
				if( bd.doesBlackBackGammonLose() )
					score = 3;
				else if( bd.doesBlackGammonLose() )
					score = 2;
				else
					score = 1;
				break;
			}
			blackTurn = !blackTurn;
			bd.swapBW();
		}
		if( blackTurn ) {
			//qDebug() << "black win";
		} else {
			score = -score;
			//qDebug() << "white win";
		}
		blackPointSum += score;
		blackPointSum2 += score * score;
	}
	const double SD = sqrt((blackPointSum2 - (double)blackPointSum * blackPointSum/N_LOOP) / N_LOOP);
	qDebug() << "Seq Dice: N = " << N_LOOP << "expect = " << (double)blackPointSum / N_LOOP << "SD = " << SD;
}
void TestBoard::statsRunning()
{
	{
		Board bd("G           W4B4           G");	//	
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G           W5B5           G");	//	
		qDebug() << bd.blackExpectedValue() << bd.position();
	}

#if 0
	{
		Board bd("G     W4            B4     G");	//	ҒliF+0.406226j킩ĂǖʁF
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G     W4              B4B4 G");	//	
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G   W4                B4   G");	//	
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G   W4                B3B2 G");	//	
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G   W4                B2B4 G");	//	
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G   W4                B1B6 G");	//	
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G   W4                  B8 G");	//	
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
#endif
#if 0
	{
		Board bd("G W4                    B4 G");	//	ҒliF+0.555556j킩ĂǖʁF
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G   W4                B4   G");	//	ҒliF+0.373876j킩ĂǖʁF
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G     W4            B4     G");	//	ҒliF+0.406226j킩ĂǖʁF
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G       W4        B4       G");	//	ҒliF+0.340931j킩ĂǖʁF
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G         W4    B4         G");	//	
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G           W4B4           G");	//	
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G           W4B3           G");	//	sbvF-6
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G           W4B3        B1 G");	//	sbvF-5
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G           W4B3      B1   G");	//	sbvF-4
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G           W4B3    B1     G");	//	sbvF-3
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G           W4B3  B1       G");	//	sbvF-2
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
	{
		Board bd("G           W4B3B1         G");	//	sbvF-1
		qDebug() << bd.blackExpectedValue() << bd.position();
	}
#endif
}
//	jOim[R^Ngj
//	TCRɗ͎gpA(1, 1) -> (3, 3) ɐ̂Ƃ
void TestBoard::statsRunningRollout()
{
	qsrand((int)time(0));
#ifdef _DEBUG
	const int N_LOOP = 9*10;
#else
	const int N_LOOP = 9*1000;
#endif
	//	Board bd("G W4                    B4 G");	//	ҒliF+0.555556j킩ĂǖʁF
	//	Board bd("G   W4                B4   G");	//	ҒliF+0.373876j킩ĂǖʁF
	//	Board bd("G     W4            B4     G");	//	ҒliF+0.406226j킩ĂǖʁF
	//	Board bd("G       W4        B4       G");	//	ҒliF+0.340931j킩ĂǖʁF
	//	Board bd("G         W4    B4         G");	//	
	//	Board bd("G           W4B4           G");	//	

	//	Board bd("G           W4B3           G");	//	sbvF-6
	//	Board bd("G           W4B3        B1 G");	//	sbvF-5
	//	Board bd("G           W4B3      B1   G");	//	sbvF-4
	//	Board bd("G           W4B3    B1     G");	//	sbvF-3
	//	Board bd("G           W4B3  B1       G");	//	sbvF-2
		Board bd("G           W4B3B1         G");	//	sbvF-1
	qDebug() << bd.position();
	//statsRandomRollOut(bd, N_LOOP);
	//statsSeqRollOut(bd, N_LOOP);
#if 1
	statsRandomRollOut(bd, 10);
	statsRandomRollOut(bd, 100);
	statsRandomRollOut(bd, 1000);
#ifndef	_DEBUG
	statsRandomRollOut(bd, 10000);
#endif
#endif
	statsSeqRollOut(bd, 10);
	statsSeqRollOut(bd, 100);
	statsSeqRollOut(bd, 1000);
#ifndef	_DEBUG
	statsSeqRollOut(bd, 10000);
#endif
}
void TestBoard::statsEndGameRollout()
{
	qsrand((int)time(0));
	double eval;
	int blackPoint = 0;
#ifdef _DEBUG
	const int N_LOOP = 90;
#else
	const int N_LOOP = 900;
#endif
	QBENCHMARK {
	blackPoint = 0;
	for(int i = 0; i < N_LOOP; ++i) {
		//Board bd("G       W4        B4       G");
		Board bd("G W4                  B2B4 G");
		//Board bd("G W4                    B4 G");
		//Board bd("G W1                    B4 G");
		bool blackTurn = true;
		for(;;) {
			int d1 = (qrand() % 3) + 1;
			int d2 = (qrand() % 3) + 1;
			if( blackTurn ) {
				bd.swapBW();
				QList<Move> mvs = bd.whiteMovesEvalXPlyD(3, d1, d2, eval, ::eval6);
				foreach(Move mv, mvs)
					bd.moveWhite(mv);
				bd.swapBW();
				if( bd.doesBlackWin() ) {
					if( bd.doesWhiteBackGammonLose() )
						blackPoint += 3;
					else if( bd.doesWhiteGammonLose() )
						blackPoint += 2;
					else
						++blackPoint;
					break;
				}
			} else {
				QList<Move> mvs = bd.whiteMovesEvalXPlyD(3, d1, d2, eval, ::eval6);
				foreach(Move mv, mvs)
					bd.moveWhite(mv);
				if( bd.doesWhiteWin() ) {
					if( bd.doesBlackBackGammonLose() )
						blackPoint -= 3;
					else if( bd.doesBlackGammonLose() )
						blackPoint -= 2;
					else
						--blackPoint;
					break;
				}
			}
			blackTurn = !blackTurn;
		}
	}
	}
	qDebug() << "BlackPoint/N = " << blackPoint << "/" << N_LOOP << "=" << (double)blackPoint / N_LOOP;
}
//	ŏI with Variance Reduction
void TestBoard::statsEndGameRolloutVR()
{
	qsrand((int)time(0));
	double eval;
	double blackPoint = 0;
#ifdef _DEBUG
	const int N_LOOP = 90;
#else
	const int N_LOOP = 900;
#endif
	QBENCHMARK {
	blackPoint = 0;
	for(int i = 0; i < N_LOOP; ++i) {
		//Board bd("G W1                    B4 G");		//	]ڂoΏAłȂΕ
		//Board bd("G       W4        B4       G");
		Board bd("G W4                  B2B4 G");
		//Board bd("G W4                    B4 G");
		//Board bd("G W1                    B4 G");
		bool blackTurn = true;
		for(;;) {
			int d1 = (qrand() % 3) + 1;
			int d2 = (qrand() % 3) + 1;
			if( blackTurn ) {
				bd.swapBW();
				QList<Move> mvs = bd.whiteMovesEvalXPlyD(3, d1, d2, eval, ::eval6);
				if( evalCallCount() == 0 ) {
					blackPoint += eval / NORMAL_WIN;
					break;
				}
				foreach(Move mv, mvs)
					bd.moveWhite(mv);
				bd.swapBW();
				if( bd.doesBlackWin() ) {
					if( bd.doesWhiteBackGammonLose() )
						blackPoint += 3;
					else if( bd.doesWhiteGammonLose() )
						blackPoint += 2;
					else
						++blackPoint;
					break;
				}
			} else {
				QList<Move> mvs = bd.whiteMovesEvalXPlyD(3, d1, d2, eval, ::eval6);
				if( evalCallCount() == 0 ) {
					blackPoint -= eval / NORMAL_WIN;
					break;
				}
				foreach(Move mv, mvs)
					bd.moveWhite(mv);
				if( bd.doesWhiteWin() ) {
					if( bd.doesBlackBackGammonLose() )
						blackPoint -= 3;
					else if( bd.doesBlackGammonLose() )
						blackPoint -= 2;
					else
						--blackPoint;
					break;
				}
			}
			blackTurn = !blackTurn;
		}
	}
	}
	qDebug() << "BlackPoint/N = " << blackPoint << "/" << N_LOOP << "=" << (double)blackPoint / N_LOOP;
}
void TestBoard::statsEndGameRolloutVR2()
{
	qsrand((int)time(0));
	double eval;
	double blackPoint = 0;
#ifdef _DEBUG
	const int N_LOOP = 10;
#else
	const int N_LOOP = 100;
#endif
	QBENCHMARK {
	blackPoint = 0;
	for(int i = 0; i < N_LOOP; ++i) {
		for(int dd1 = 1; dd1 <= 3; ++dd1) {
			for(int dd2 = 1; dd2 <= 3; ++dd2 ) {
				int d1 = dd1;
				int d2 = dd2;
				//Board bd("G W1                    B4 G");
				//Board bd("G       W4        B4       G");
				Board bd("G W4                  B2B4 G");
				//Board bd("G W4                    B4 G");
				//Board bd("G W1                    B4 G");
				bool blackTurn = true;
				for(;;) {
					if( blackTurn ) {
						bd.swapBW();
						const int rdepth = bd.canWhiteBearOff() ? 4 : 3;
						Moves mvs = bd.whiteMovesEvalXPlyD(rdepth, d1, d2, eval, ::eval6);
						bd.swapBW();
						if( evalCallCount() == 0 ) {
							Q_ASSERT( terminalNodeCount() != 0 );
							blackPoint += eval / NORMAL_WIN;
							break;
						}
						foreach(Move mv, mvs)
							bd.moveBlack(mv);
						if( bd.doesBlackWin() ) {
							if( bd.doesWhiteBackGammonLose() )
								blackPoint += 3;
							else if( bd.doesWhiteGammonLose() )
								blackPoint += 2;
							else
								blackPoint += 1;
							break;
						}
					} else {
						const int rdepth = bd.canWhiteBearOff() ? 4 : 3;
						QList<Move> mvs = bd.whiteMovesEvalXPlyD(rdepth, d1, d2, eval, ::eval6);
						if( evalCallCount() == 0 ) {
							Q_ASSERT( terminalNodeCount() != 0 );
							blackPoint -= eval / NORMAL_WIN;
							break;
						}
						foreach(Move mv, mvs)
							bd.moveWhite(mv);
						if( bd.doesWhiteWin() ) {
							if( bd.doesBlackBackGammonLose() )
								blackPoint -= 3;
							else if( bd.doesBlackGammonLose() )
								blackPoint -= 2;
							else
								blackPoint -= 1;
							break;
						}
					}
					blackTurn = !blackTurn;
					d1 = (qrand() % 3) + 1;
					d2 = (qrand() % 3) + 1;
				}
			}
		}
	}
	}
	qDebug() << "BlackPoint/(N*9) = " << blackPoint << "/ (" << N_LOOP << "* 9) =" << (double)blackPoint / (N_LOOP * 9);
}
void TestBoard::statsWhiteEndGameRolloutVR2()
{
	qsrand((int)time(0));
	double eval;
	double blackPoint = 0;
#ifdef _DEBUG
	const int N_LOOP = 10;
#else
	const int N_LOOP = 100;
#endif
	blackPoint = 0;
	for(int i = 0; i < N_LOOP; ++i) {
		for(int dd1 = 1; dd1 <= 3; ++dd1) {
			for(int dd2 = 1; dd2 <= 3; ++dd2 ) {
				int d1 = dd1;
				int d2 = dd2;
				Board bd("G W2W2W2              B1W2 G");	//	1pnt ubN
				//Board bd("G W1                    B4 G");
				//Board bd("G       W4        B4       G");
				//Board bd("G W4                  B2B4 G");
				//Board bd("G W4                    B4 G");
				//Board bd("G W1                    B4 G");
				bool blackTurn = false;
				for(;;) {
					if( blackTurn ) {
						bd.swapBW();
						const int rdepth = bd.canWhiteBearOff() ? 4 : 3;
						Moves mvs = bd.whiteMovesEvalXPlyD(rdepth, d1, d2, eval, ::eval6);
						bd.swapBW();
						if( !mvs.isEmpty() ) {
							if( evalCallCount() == 0 ) {
								Q_ASSERT( terminalNodeCount() != 0 );
								blackPoint += eval / NORMAL_WIN;
								break;
							}
							foreach(Move mv, mvs)
								bd.moveBlack(mv);
							//qDebug() << bd.position();
							if( bd.doesBlackWin() ) {
								if( bd.doesWhiteBackGammonLose() )
									blackPoint += 3;
								else if( bd.doesWhiteGammonLose() )
									blackPoint += 2;
								else
									blackPoint += 1;
								break;
							}
						}
					} else {
						const int rdepth = bd.canWhiteBearOff() ? 4 : 3;
						QList<Move> mvs = bd.whiteMovesEvalXPlyD(rdepth, d1, d2, eval, ::eval6);
						if( !mvs.isEmpty() ) {
							if( evalCallCount() == 0 ) {
								Q_ASSERT( terminalNodeCount() != 0 );
								blackPoint -= eval / NORMAL_WIN;
								break;
							}
							foreach(Move mv, mvs)
								bd.moveWhite(mv);
							//qDebug() << bd.position();
							if( bd.doesWhiteWin() ) {
								if( bd.doesBlackBackGammonLose() )
									blackPoint -= 3;
								else if( bd.doesBlackGammonLose() )
									blackPoint -= 2;
								else
									blackPoint -= 1;
								break;
							}
						}
					}
					blackTurn = !blackTurn;
					d1 = (qrand() % 3) + 1;
					d2 = (qrand() % 3) + 1;
				}
			}
		}
		qDebug() << "WhitePoint/(N*9) = " << -blackPoint << "/ (" << (i+1) << "* 9) ="
					<< (double)-blackPoint / ((i+1) * 9);
	}
	//qDebug() << "BlackPoint/(N*9) = " << blackPoint << "/ (" << N_LOOP << "* 9) =" << (double)blackPoint / (N_LOOP * 9);
}
//	_ vs eval5+1ǂ
void TestBoard::statsRollout_Random_Eval5Ply1()
{
	qsrand((int)time(0));
	double eval;
	int randomWin = 0;
	int eval5Win = 0;
	int eval5Point = 0;
#ifdef _DEBUG
	const int N_LOOP = 100;
#else
	const int N_LOOP = 500;
#endif
	for(int i = 0; i < N_LOOP; ++i) {
		Board bd("G >2  <3    >3<3    >3  <2 G");		//	
		bool randomTurn = !(i & 1);
		for(;;) {
			int d1 = (qrand() % 3) + 1;
			int d2 = (qrand() % 3) + 1;
			if( randomTurn ) {
				bd.swapBW();
				QList<Move> mvs = bd.whiteMovesRandom(d1, d2);
				foreach(Move mv, mvs)
					bd.moveWhite(mv);
				bd.swapBW();
				if( bd.doesBlackWin() ) {
					++randomWin;
					if( bd.doesWhiteBackGammonLose() )
						eval5Point -= 3;
					else if( bd.doesWhiteGammonLose() )
						eval5Point -= 2;
					else
						--eval5Point;
					break;
				}
			} else {
				QList<Move> mvs = bd.whiteMovesEval5PlyD(1, d1, d2, eval);	//	1ǂ݁{eval5()
				foreach(Move mv, mvs)
					bd.moveWhite(mv);
				if( bd.doesWhiteWin() ) {
					++eval5Win;
					if( bd.doesBlackBackGammonLose() )
						eval5Point += 3;
					else if( bd.doesBlackGammonLose() )
						eval5Point += 2;
					else
						++eval5Point;
					break;
				}
			}
			randomTurn = !randomTurn;
		}
	}
	qDebug() << "randomWin = " << randomWin << "(" << randomWin * 100 / (double)N_LOOP << "%)";
	qDebug() << "eval5 ply-1 = " << eval5Win << "(" << eval5Win * 100 / (double)N_LOOP << "%)"
				<< "Point = " << eval5Point;
}
//	_ vs eval6+1ǂ
void TestBoard::statsRollout_Random_Eval6Ply1()
{
	qsrand((int)time(0));
	double eval;
	int randomWin = 0;
	int eval6Win = 0;
	int eval6Point = 0;
#ifdef _DEBUG
	const int N_LOOP = 100;
#else
	const int N_LOOP = 500;
#endif
	for(int i = 0; i < N_LOOP; ++i) {
		Board bd("G >2  <3    >3<3    >3  <2 G");		//	
		bool randomTurn = !(i & 1);
		for(;;) {
			int d1 = (qrand() % 3) + 1;
			int d2 = (qrand() % 3) + 1;
			if( randomTurn ) {
				bd.swapBW();
				QList<Move> mvs = bd.whiteMovesRandom(d1, d2);
				foreach(Move mv, mvs)
					bd.moveWhite(mv);
				bd.swapBW();
				if( bd.doesBlackWin() ) {
					++randomWin;
					if( bd.doesWhiteBackGammonLose() )
						eval6Point -= 3;
					else if( bd.doesWhiteGammonLose() )
						eval6Point -= 2;
					else
						--eval6Point;
					break;
				}
			} else {
				QList<Move> mvs = bd.whiteMovesEvalXPlyD(1, d1, d2, eval, ::eval6);	//	1ǂ݁{eval6()
				foreach(Move mv, mvs)
					bd.moveWhite(mv);
				if( bd.doesWhiteWin() ) {
					++eval6Win;
					if( bd.doesBlackBackGammonLose() )
						eval6Point += 3;
					else if( bd.doesBlackGammonLose() )
						eval6Point += 2;
					else
						++eval6Point;
					break;
				}
			}
			randomTurn = !randomTurn;
		}
	}
	qDebug() << "randomWin = " << randomWin << "(" << randomWin * 100 / (double)N_LOOP << "%)";
	qDebug() << "eval6 ply-1 = " << eval6Win << "(" << eval6Win * 100 / (double)N_LOOP << "%)"
				<< "Point = " << eval6Point;
}
//	_ vs eval5+3ǂ
void TestBoard::statsRollout_Random_Eval5Ply3()
{
	///qsrand((int)time(0));
	double eval;
	int randomWin = 0;
	int eval5Win = 0;
#ifdef _DEBUG
	const int N_LOOP = 10;
#else
	const int N_LOOP = 20;
#endif
	for(int i = 0; i < N_LOOP; ++i) {
		Board bd("G >2  <3    >3<3    >3  <2 G");		//	
		bool randomTurn = !(i & 1);
		for(;;) {
			int d1 = (qrand() % 3) + 1;
			int d2 = (qrand() % 3) + 1;
			if( randomTurn ) {
				bd.swapBW();
				QList<Move> mvs = bd.whiteMovesRandom(d1, d2);
				foreach(Move mv, mvs)
					bd.moveWhite(mv);
				bd.swapBW();
				if( bd.doesBlackWin() ) {
					++randomWin;
					break;
				}
			} else {
				QList<Move> mvs = bd.whiteMovesEval5PlyD(3, d1, d2, eval);	//	3ǂ݁{eval5()
				foreach(Move mv, mvs)
					bd.moveWhite(mv);
				if( bd.doesWhiteWin() ) {
					++eval5Win;
					break;
				}
			}
			randomTurn = !randomTurn;
		}
	}
	qDebug() << "eval5 ply-3 = " << eval5Win << "(" << eval5Win * 100 / (double)N_LOOP << "%)";
	qDebug() << "randomWin = " << randomWin << "(" << randomWin * 100 / (double)N_LOOP << "%)";
}
//	eval5+1ǂvs eval5+3ǂ
void TestBoard::statsRollout_Eval5Ply1_Eval5Ply3()
{
	///qsrand((int)time(0));
	double eval;
	int eval5Ply1Win = 0;
	int eval5Ply3Win = 0;
#ifdef _DEBUG
	const int N_LOOP = 10;
#else
	const int N_LOOP = 20;
#endif
	for(int i = 0; i < N_LOOP; ++i) {
		Board bd("G >2  <3    >3<3    >3  <2 G");		//	
		bool randomTurn = !(i & 1);
		for(;;) {
			int d1 = (qrand() % 3) + 1;
			int d2 = (qrand() % 3) + 1;
			if( randomTurn ) {
				bd.swapBW();
				QList<Move> mvs = bd.whiteMovesEval5PlyD(1, d1, d2, eval);	//	1ǂ݁{eval5()
				foreach(Move mv, mvs)
					bd.moveWhite(mv);
				bd.swapBW();
				if( bd.doesBlackWin() ) {
					++eval5Ply1Win;
					break;
				}
			} else {
				QList<Move> mvs = bd.whiteMovesEval5PlyD(3, d1, d2, eval);	//	3ǂ݁{eval5()
				foreach(Move mv, mvs)
					bd.moveWhite(mv);
				if( bd.doesWhiteWin() ) {
					++eval5Ply3Win;
					break;
				}
			}
			randomTurn = !randomTurn;
		}
	}
	qDebug() << "eval5 ply-1 = " << eval5Ply1Win << "(" << eval5Ply1Win * 100 / (double)N_LOOP << "%)";
	qDebug() << "eval5 ply-3 = " << eval5Ply3Win << "(" << eval5Ply3Win * 100 / (double)N_LOOP << "%)";
}
//	eval5+1ǂvs eval6+1ǂ
void TestBoard::statsRollout_Eval5Ply1_Eval6Ply1()
{
	qsrand((int)time(0));
	double eval;
	int eval5Ply1Win = 0;
	int eval6Ply1Win = 0;
	int eval6Point = 0;
#ifdef _DEBUG
	const int N_LOOP = 100;
#else
	const int N_LOOP = 500;
#endif
	for(int i = 0; i < N_LOOP; ++i) {
		Board bd("G >2  <3    >3<3    >3  <2 G");		//	
		bool randomTurn = !(i & 1);
		for(;;) {
			int d1 = (qrand() % 3) + 1;
			int d2 = (qrand() % 3) + 1;
			if( randomTurn ) {
				bd.swapBW();
				QList<Move> mvs = bd.whiteMovesEvalXPlyD(1, d1, d2, eval, ::eval5);	//	1ǂ݁{eval5()
				foreach(Move mv, mvs)
					bd.moveWhite(mv);
				bd.swapBW();
				if( bd.doesBlackWin() ) {
					++eval5Ply1Win;
					if( bd.doesWhiteBackGammonLose() )
						eval6Point -= 3;
					else if( bd.doesWhiteGammonLose() )
						eval6Point -= 2;
					else
						--eval6Point;
					break;
				}
			} else {
				QList<Move> mvs = bd.whiteMovesEvalXPlyD(1, d1, d2, eval, ::eval6);	//	1ǂ݁{eval6()
				foreach(Move mv, mvs)
					bd.moveWhite(mv);
				if( bd.doesWhiteWin() ) {
					++eval6Ply1Win;
					if( bd.doesBlackBackGammonLose() )
						eval6Point += 3;
					else if( bd.doesBlackGammonLose() )
						eval6Point += 2;
					else
						++eval6Point;
					break;
				}
			}
			randomTurn = !randomTurn;
		}
	}
	qDebug() << "eval5 ply-1 = " << eval5Ply1Win << "(" << eval5Ply1Win * 100 / (double)N_LOOP << "%)";
	qDebug() << "eval6 ply-1 = " << eval6Ply1Win << "(" << eval6Ply1Win * 100 / (double)N_LOOP << "%)"
				<< "Point = " << eval6Point;
}
void TestBoard::benchEval5Ply1()
{
	double eval;
	QBENCHMARK {
		Board bd("G >2  <3    >3<3    >3  <2 G");		//	
		int d1 = 1;		//(qrand() % 3) + 1;
		int d2 = 3;		//(qrand() % 3) + 1;
		QList<Move> mvs = bd.whiteMovesEval5PlyD(1, d1, d2, eval);	//	1ǂ݁{eval5()
	}
	qDebug() << "terminalNodeCount = " << terminalNodeCount();
}
void TestBoard::benchEval5Ply2()
{
	double eval;
	QBENCHMARK {
		Board bd("G >2  <3    >3<3    >3  <2 G");		//	
		int d1 = 2;	//(qrand() % 3) + 1;
		int d2 = 3;	//(qrand() % 3) + 1;
		QList<Move> mvs = bd.whiteMovesEval5PlyD(2, d1, d2, eval);	//	2ǂ݁{eval5()
	}
	qDebug() << "terminalNodeCount = " << terminalNodeCount();
}
void TestBoard::benchEval5Ply3()
{
	double eval;
	QBENCHMARK_ONCE {
		Board bd("G >2  <3    >3<3    >3  <2 G");		//	
		int d1 = 2;	//(qrand() % 3) + 1;
		int d2 = 3;	//(qrand() % 3) + 1;
		QList<Move> mvs = bd.whiteMovesEval5PlyD(3, d1, d2, eval);	//	3ǂ݁{eval5()
	}
	qDebug() << "terminalNodeCount = " << terminalNodeCount();
}
void TestBoard::benchEval5Ply4()
{
	double eval;
	QBENCHMARK_ONCE {
		Board bd("G >2  <3    >3<3    >3  <2 G");		//	
		int d1 = 2;	//(qrand() % 3) + 1;
		int d2 = 3;	//(qrand() % 3) + 1;
		QList<Move> mvs = bd.whiteMovesEval5PlyD(4, d1, d2, eval);	//	4ǂ݁{eval5()
	}
	qDebug() << "terminalNodeCount = " << terminalNodeCount();
}
//	\擾eXg
void TestBoard::benchWhiteMovesList()
{
	QList<Moves> mvsList;
	Board bd("G >2  <3    >3<3    >3  <2 G");
	QBENCHMARK {
		//mvsList = bd.whiteMovesList(1, 2);
		//mvsList = bd.whiteMovesList(1, 3);
		//mvsList = bd.whiteMovesList(2, 3);
		mvsList = bd.whiteMovesList(1, 1);
		mvsList = bd.whiteMovesList(2, 2);
		mvsList = bd.whiteMovesList(3, 3);
	}
}
void TestBoard::benchWhiteMovesList3()
{
	QList<Moves> mvsList;
	Board bd("G >2  <3    >3<3    >3  <2 G");
	QBENCHMARK {
		//mvsList = bd.whiteMovesList3(1, 2);
		//mvsList = bd.whiteMovesList3(1, 3);
		//mvsList = bd.whiteMovesList3(2, 3);
		mvsList = bd.whiteMovesList3(1, 1);
		mvsList = bd.whiteMovesList3(2, 2);
		mvsList = bd.whiteMovesList3(3, 3);
	}
}
//	eXg
void TestBoard::statsDice()
{
	{
		qsrand((int)time(0));
		QVector<int> cnt(9, 0);
		for(int i = 0; i < 9*1000; ++i) {
			int d1 = qrand() % 3;
			int d2 = qrand() % 3;
			cnt[d1 * 3 + d2] += 1;
		}
		qDebug() << cnt;
	}
	{
		// uZkcCX^[v( Seed=ݎ ) 
		// uȐ̈lv( 1`6 ) 𐶐
		boost::mt19937            gen( static_cast<unsigned long>(time(0)) );
		boost::uniform_smallint<> dst( 0, 2 );
		boost::variate_generator<
			boost::mt19937&, boost::uniform_smallint<>
		> rand( gen, dst );

		//for( int i=0; i<10; ++i )
		//	qDebug() << rand();
		QVector<int> cnt(9, 0);
		for(int i = 0; i < 9*1000; ++i) {
			int d1 = rand();
			int d2 = rand();
			cnt[d1 * 3 + d2] += 1;
		}
		qDebug() << cnt;
	}
}
int NC(int p, int n)
{
	Q_ASSERT( p > 0 );
	if( !n || p == 1 ) return 1;
	if( n == 1 ) return p;
	int sum = 0;
	for(int i = 0; i <= n; ++i)
		 sum += NC(p-1, n-i);
	return sum;
}
//	NC2(p, n) for p > 0, n >= 0 ̃e[u
QVector< QVector<int> > g_NC2;
int NC2(int p, int n)
{
	Q_ASSERT( p > 0 );
	Q_ASSERT( n >= 0 );
	while( g_NC2.size() < p )
		g_NC2 << QVector<int>();
	QVector<int> &v = g_NC2[p-1];
	while( v.size() <= n )
		v << 0;
	if( v[n] != 0 )
		return v[n];
	if( p == 1 )
		return v[n] = 1;
	int sum = 0;
	for(int i = 0; i <= n; ++i)
		 sum += NC2(p-1, n-i);
	return v[n] = sum;
}
//	ix ŌŁAn ̐΂uꍇ̐߂
int nPermutationBlack(int ix, int n)
{
	if( !ix )
		return 1;
	else {
		int sum = 0;
		for(int i = 1; i <= n; ++i)
			sum += NC(ix, n - i);
		return sum;
	}
}
//	ix ŌŁAnStone ̐΂uꍇ̐߂
//	AŌɂ̂ 1`n Ƃ
int nPermutationBlack(int ix, int nStone, int n)
{
	if( !ix )
		return 1;
	else {
		int sum = 0;
		for(int i = 1; i <= n; ++i)
			sum += NC(ix, nStone - i);
		return sum;
	}
}
//	̍Ō bt ̏ꍇ̏Ԑ
template<int NP, int NS>
int nPermutation(int bt)
{
	const int nPoint = NP;
	const int nStone = NS;
	int sum = 0;
	for(int wt = 1; wt <= nPoint - bt; ++wt)
		sum += nPermutationBlack(wt, nStone);
	return sum * nPermutationBlack(bt, nStone);
}
//	̍Ō bt ŁAb[wt]  n ȉ̏ꍇ̏Ԑ
template<int NP, int NS>
int nPermutation(int bt, int n)
{
	const int nPoint = NP;
	const int nStone = NS;
	int sum = 0;
	for(int wt = 1; wt <= nPoint - bt; ++wt)
		sum += nPermutationBlack(wt, nStone);
	return sum * nPermutationBlack(bt, nStone, n);
}
int nPermutation33(int bt)
{
	return nPermutation<3, 3>(bt);
}
int nPermutation33(int bt, int n)
{
	return nPermutation<3, 3>(bt, n);
}
//	ix ` 0  n ̐΂uꍇ̐𐔂
//	 ix ʒuɂ͂Pȏ΂ûƂ
int nPermutation1Color(int ix, int n)
{
	if( !ix )
		return 1;
	else {
		int sum = 0;
		for(int i = 1; i <= n; ++i)
			sum += NC(ix, n - i);
		return sum;
	}
}
//	jOԂ̏ԐԂ
template<int NP, int NS>
int nPermutationRunning()
{
	int sum = 0;
	for(int bt = 1; bt < NP; ++bt) {
		int n = nPermutation1Color(bt, NS);	//	Ō bt ̎̏Ԑ
		for(int wt = 1; wt <= NP - bt; ++wt) {
			sum += n * nPermutation1Color(wt, NS);
		}
	}
	return sum;
}
//	{n, 0,...0}  #0 ƂĐ
int seqNumber(const QVector<int> &v,
				int ix,		//	v[ix`0] Q
				int n)		//	index <= ix ̐ΐ
{
	Q_ASSERT( ix >= 0 );
	if( !n ) return 0;		//	0̐΂̒u͂Ђƒʂ肵Ȃ
	if( !ix ) return 0;		//	Ō͎̌IɌ܂

	int sum = 0;	//	ix ̌ [0, v[ix]) ̏ꍇ̐𐔂
	for(int i = 0; i < v[ix]; ++i) {
		sum += NC2(ix, n - i);
	}
	return seqNumber(v, ix - 1, n - v[ix]) + sum;
}
template<int NP, int NS>
int seqNumber(const QVector<int> &b, const QVector<int> &w)
{
	const int nPoint = NP;
	const int nStone = NS;
	int bt = nPoint;
	while( --bt > 1 && !b[bt] ) {}
	int sum = 0;
	//	̍Ō 1pnt 傫ꍇ́A菬ꍇ̏ԐvX
	for(int ix = 1; ix < bt; ++ix)
		sum += nPermutation33(ix);
	if( b[bt] > 1 ) {
		int wsum = 0;	//	̍Ō bt ̏ꍇ́ȀԐ
		for(int wt = 1; wt <= nPoint - bt; ++wt) {	//	wtF̍Ō
			for(int n = 1; n <= nStone; ++n) {		//	nFŌ̐ΐ
				wsum += NC(wt, nStone - n);			//	̍Ō wt ̏ꍇ̏Ԑ
			}
		}
		sum += wsum * (b[bt] - 1);
		//for(int n = 1; n < b[bt]; ++n) {
		//	sum += NC(bt + 1, nStone - 1) * wsum;
		//}
	}
	return seqNumber(w, nPoint - bt, nStone) + sum;		//	̏Ԑ
	//return seqNumber(b, bt - 1, nStone - b[bt]) + sum;
}
int seqNumber33(const QVector<int> &b, const QVector<int> &w)
{
	return seqNumber<3, 3>(b, w);
}
int seqNumber42(const QVector<int> &b, const QVector<int> &w)
{
	return seqNumber<4, 2>(b, w);
}
int seqNumberRev(const QVector<int> &v,
				int ix,		//	v[ix`0] Q
				int n)		//	index <= ix ̐ΐ
{
	QVector<int> t(v.size());
	for(int i = 0; i < v.size(); ++i)
		t[v.size() - i - 1] = v[i];
	return seqNumber(t, ix, n);
}
int g_count = 0;
#if 0
const int nPoint = 6;		//	1/4 M
const int nPiece = 4;		//	o̐ΐ
#endif
#if 1
int nPoint = 4;		//	1/6 M
int nPiece = 3;		//	o̐ΐ
#endif
#if 0
const int nPoint = 3;		//	1/8 M
const int nPiece = 2;		//	o̐ΐ
#endif
void printNC(const QVector<int> &v0,
				int p,		//	cӏ
				int n)		//	ΐ
{
	QVector<int> v(v0);
#if 0
	if( !n ) {
		for(int i = 0; i < p; ++i)
			v << 0;
		qDebug() << ++g_count << v;
		return;
	}
#endif
	if( p == 1 ) {
		v << n;
		qDebug() << ++g_count << v << "->" << seqNumberRev(v, nPoint-1, nPiece);
		return;
	}
#if 0
	if( n == 1 ) {
		for(int i = 0; i < p; ++i) {
			v = v0;
			for(int k = 0; k < p; ++k) {
				v << (k == i ? 1 : 0);
			}
			qDebug() << ++g_count << v << "->" << seqNumberRev(v, nPoint-1, nPiece);
		}
		return;
	}
#endif
	for(int i = 0; i <= n; ++i) {
		v = v0;
		v << i;
		printNC(v, p - 1, n - i);
	}
}
template<typename T>
QVector<T> reverse(const QVector<T> v)
{
	QVector<T> t(v.size());
	for(int i = 0; i < v.size(); ++i)
		t[v.size() - i - 1] = v[i];
	return t;
}
//	1/6 M̑SԐ\i΂̂݁j
//		G 4 3 2 1 G		ΐ͑oR
//	1/8 M̑SԐ\i΂̂݁j
//		G 3 2 1 G		ΐ͑oRƂ
template<int NP, int NS>
void printPermutationBlack(QVector<int> v, int ix, int n)
{
	const int nPoint = NP;
	const int nStone = NS;
	if( !ix ) {
		v[ix] = n;
		qDebug() << reverse(v) << seqNumber(v, nPoint - 1, nStone);
	} else {
		for(int i = 0; i <= n; ++i) {
			v[ix] = i;
			printPermutationBlack<NP, NS>(v, ix - 1, n - i);
		}
	}
}
template<int NP, int NS>
void printPermutationBlack()
{
	QVector<int> v(NP + 2, 0);
	for(int tix = 1; tix < NP; ++tix) {
		for(int n = 1; n <= NS; ++n) {
			v[tix] = n;
			printPermutationBlack<NP, NS>(v, tix - 1, 3 - n);
		}
	}
}
template<int NP, int NS>
void printPermutation(QVector<int> b, QVector<int> w, int ix, int n)
{
	const int nPoint = NP;
	const int nStone = NS;
	if( !ix ) {
		w[ix] = n;
		//qDebug() << w << reverse(b) /*<< seqNumber(b, nPoint - 1, nStone)*/;
		QVector<int> t(b.size(), 0);
		for(int i = 0; i < b.size(); ++i)
			t[i] = w[i] ? -w[i] : b[b.size() - 1 - i];
		//qDebug() << t;
		QString text;
		foreach(int n, t)
			text += QString("%1 ").arg(n, 2);
		qDebug() << text << seqNumber33(b, w);
	} else {
		for(int i = 0; i <= n; ++i) {
			w[ix] = i;
			printPermutation<NP, NS>(b, w, ix - 1, n - i);
		}
	}
}
template<int NP, int NS>
void printPermutation(int bt, QVector<int> b, int ix, int n)
{
	const int nPoint = NP;
	const int nStone = NS;
	if( !ix ) {
		b[ix] = n;
		//qDebug() << reverse(b) << seqNumber(b, nPoint - 1, nStone);
		QVector<int> w(nPoint + 2, 0);
		for(int wt = 1; wt <= nPoint - bt; ++wt) {
			for(int n = 1; n <= nStone; ++n) {
				w[wt] = n;
				printPermutation<NP, NS>(b, w, wt - 1, nStone - n);
			}
		}
	} else {
		for(int i = 0; i <= n; ++i) {
			b[ix] = i;
			printPermutation<NP, NS>(bt, b, ix - 1, n - i);
		}
	}
}
template<int NP, int NS>
void printPermutation()
{
	const int nPoint = NP;
	const int nStone = NS;
	QVector<int> b(nPoint + 2, 0);
	for(int bt = 1; bt < nPoint; ++bt) {
		for(int n = 1; n <= nStone; ++n) {
			b[bt] = n;
			printPermutation<NP, NS>(bt, b, bt - 1, nStone - n);
		}
	}
}
	
void printRunningNC(const QVector<int> &v0,
				int p,		//	cӏ
				int n)		//	ΐ
{
	QVector<int> v(v0);
	if( !n ) {
		for(int i = 0; i < p; ++i)
			v << 0;
		qDebug() << ++g_count << v;
		return;
	}
	if( p == 1 ) {
		if( n < nPiece ) {	//	Iǂ̓JEgȂ
			v << n;
			qDebug() << ++g_count << v;
		}
		return;
	}
	if( n == 1 ) {
		for(int i = 0; i < p; ++i) {
			v = v0;
			for(int k = 0; k < p; ++k) {
				v << (k == i ? 1 : 0);
			}
			qDebug() << ++g_count << v;
		}
		return;
	}
	for(int i = 0; i <= n; ++i) {
		v = v0;
		v << i;
		printRunningNC(v, p - 1, n - i);
	}
}
void printRunningSub(const QVector<int> &v0,
				int tail,	//	Ō|Cg
				int p,		//	cӏ
				int n)		//	ΐ
{
	QVector<int> v(v0);
	if( !n ) {
		for(int i = 0; i < p; ++i)
			v << 0;
		//qDebug() << ++g_count << v;
		v[tail] -= 1;
		printRunningNC(v, nPoint - tail + 1, nPiece);
		return;
	}
	if( p == 1 ) {
		v << -n;
		//qDebug() << ++g_count << v;
		v[tail] -= 1;
		printRunningNC(v, nPoint - tail + 1, nPiece);
		return;
	}
	if( n == 1 ) {
		for(int i = 0; i < p; ++i) {
			v = v0;
			for(int k = 0; k < p; ++k) {
				v << (k == i ? -1 : 0);
			}
			//qDebug() << ++g_count << v;
			v[tail] -= 1;
			printRunningNC(v, nPoint - tail + 1, nPiece);
		}
		return;
	}
	for(int i = 0; i <= n; ++i) {
		v = v0;
		v << -i;
		printRunningSub(v, tail, p - 1, n - i);
	}
}
//	NC(p, n)	1 <= p <= 12, 0 <= n <= 8 ߂
QVector<int> g_NC;
void buildNC(int nPoint, int nStone)
{
	g_NC.clear();
	for(int n = 0; n <= nStone; ++n)
		g_NC << 1;
	for(int p = 2; p <= nPoint; ++p) {
		int ix = g_NC.size() - (nStone + 1);
		int sum = 1;
		g_NC << sum;
		for(int n = 1; n <= nStone; ++n) {
			sum += g_NC[++ix];
			g_NC << sum;
		}
	}
}
int tblNC(int p, int n)
{
	Q_ASSERT( p >= 1 );
	return g_NC[(p - 1) * 9 + n];
}
//	{n, 0,...0}  #0 ƂĐ
int seqNumber(cchar *v, int ix, int n)
{
	Q_ASSERT( ix >= 0 );
	if( !n ) return 0;		//	0̐΂̒u͂Ђƒʂ肵Ȃ
	if( !ix ) return 0;		//	Ō͎̌IɌ܂

	int sum = 0;	//	ix ̌ [0, v[ix]) ̏ꍇ̐𐔂
	for(int i = 0; i < v[ix]; ++i) {
		sum += tblNC(ix, n - i);
	}
	return seqNumber(v, ix - 1, n - v[ix]) + sum;
}
void TestBoard::testNC()
{
	QCOMPARE( NC(2, 8), 9 );
	QCOMPARE( NC(4, 8), 165 );
	qDebug() << "NC(14, 8) = " << NC(14, 8);

	{
		int sum = 0;
		for(int tix = 11; tix >= 1; --tix) {	//	tix : Ō|Cg
			sum += NC(13-tix, 8) * NC(tix+1, 7);
		}
		qDebug() << "sum = " << sum;
	}
	{
		int sum = 0;
		for(int i = 1; i < 12; ++i) {	//	i : Ō|Cg
			int n = NC(i+1, 7);		//	+1 for GOAL
			for(int k = 1; k <= 12 - i; ++k) {
				sum += NC(k + 1, 7) * n;
			}
		}
		qDebug() << "sum = " << sum;
	}

	g_count = 0;
	printNC(QVector<int>(), 4, 3);
#if 0
	for(int tail = 1; tail < nPoint; ++tail) {	//	|Cg͍E 1nPoint Ƃ
		printRunningSub(QVector<int>(), tail, tail + 1, nPiece - 1);
	}
#endif
	buildNC(14, 8);
	int ix = 0;
	for(int p = 1; p <= 14; ++p) {
		QVector<int> t;
		for(int i = 0; i <= 8; ++i) {
			QCOMPARE( g_NC[ix], NC(p, i) );
			t << g_NC[ix++];
		}
		qDebug() << t;
	}
}
void TestBoard::testNC2()
{
	QCOMPARE( NC2(1, 8), 1 );
	QCOMPARE( NC2(2, 8), 9 );
	QCOMPARE( NC2(4, 8), 165 );
}
void TestBoard::benchLoop()
{
	QVector<char> v(30*1000*1000);
	int sum = 0;
	QBENCHMARK {
		sum = 0;
		foreach(char c, v)
			sum += c;
	}
	qDebug() << "sum = " << sum;
}
void TestBoard::testPermutation()
{
	QCOMPARE( (nPermutationRunning<2, 2>()), 4 );
	QCOMPARE( (nPermutationRunning<3, 2>()), 16 );
	QCOMPARE( (nPermutationRunning<3, 3>()), 45 );
	QCOMPARE( (nPermutationRunning<12, 8>()), 30169816 );
}
void TestBoard::testPermutation6()
{
	printPermutationBlack<4, 3>();
	//printPermutation6();
}
void TestBoard::testPermutation8()
{
	//	̍Ōwӏ̏ꍇ̍̏Ԑ
	QCOMPARE( nPermutationBlack(1, 3), 3 );
	QCOMPARE( nPermutationBlack(2, 3), 6 );
	//	̍Ōwӏ̏ꍇ́ȀԐi1/8MA|CgFRAΐFRj
	QCOMPARE( nPermutation33(1), 27 );
	QCOMPARE( nPermutation33(1, 1), 9 );
	QCOMPARE( nPermutation33(1, 2), 18 );
	QCOMPARE( nPermutation33(1, 3), 27 );
	//printPermutationBlack8();
	printPermutation<3, 3>();
	printPermutation<4, 2>();
	//QVector<char> v;
	//v << 0 << 1 << 0x41;
	//qDebug() << v;
}
//	萔e[u
//	So[ɂꍇ pips  13*8 = 104Aŏ_CXl 21 Ȃ̂ŁA104/3. = 34.6667
//	Ēl [0, +35000]	x 10^(-3)
QVector<ushort> g_vNumMove;
//	萔e[u
ushort numMoveBlack(const Board &bd);
ushort numMoveBlack(const Board &bd0, int d1, int d2)
{
	ushort minNM = 0xffff;
	QList<Moves> mvsList = bd0.blackMovesList3(d1, d2);
	foreach(Moves mvs, mvsList) {
		Board bd(bd0);
		foreach(Move mv, mvs)
			bd.moveBlack(mv);
		if( bd.doesBlackWin() )
			return 1000;
		ushort nm = numMoveBlack(bd) + 1000;
		minNM = qMin(minNM, nm);
	}
	return minNM;
}
ushort numMoveBlack(const Board &bd)
{
	int sn = seqNumber(bd.blackPtr(), HG_NP + 1, HG_NS) - 1;	//	sn : 0 org
	if( g_vNumMove[sn] != NOT_YET )
		return g_vNumMove[sn];
	uint t = 0;
	t += numMoveBlack(bd, 3, 3);
	t += numMoveBlack(bd, 2, 2);
	t += numMoveBlack(bd, 3, 2) * 2;
	t += numMoveBlack(bd, 1, 1);
	t += numMoveBlack(bd, 2, 1) * 2;
	t += numMoveBlack(bd, 2, 1) * 2;
	ushort nm = (ushort)(t / 9);
	return g_vNumMove[sn] = nm;
}
void TestBoard::buildNumMoveTable()
{
	buildNC(HG_NP + 2, HG_NS);		//	for 1/2 gammon
	g_vNumMove = QVector<ushort>(203490, NOT_YET);
	Board bd("G . . . . . . . . . . . B1 G");
	::numMoveBlack(bd);
	for(int p = 1; p <= 8; ++p) {
		for(int n = 1; n <= HG_NS; ++n) {
			int pips = n * p;
			bd.setBlack(p, n);
			bd.setBlack(0, HG_NS - n);
			ushort nm = numMoveBlack(bd);
			qDebug() << bd.position() << QString("%1").arg(QString::number((double)nm/1000, 'f', 4), 7)
						<< "pips / 5.333 = " << pips * 3 / 16.0;
		}
		bd.setBlack(p, 0);
	}
	for(int p = 1; p <= 8; ++p) {
		for(int n = 1; n <= HG_NS; ++n) {
			int pips = n * p;
			bd.setBlack(p, n);
			bd.setBlack(0, HG_NS - n);
			ushort nm = numMoveBlack(bd);
			qDebug() << bd.position() << QString("%1").arg(QString::number((double)nm/1000, 'f', 4), 7)
						<< "pips / 5.333 = " << pips * 3 / 16.0;
		}
		bd.setBlack(p, 0);
	}
	int nny = 0;
	foreach(ushort v, g_vNumMove)
		if( v == NOT_YET ) ++nny;
	qDebug() << "notYet/vNumMove.size() = " << nny << "/" << g_vNumMove.size()
				<< "=" << (double)nny / g_vNumMove.size();
}
void buildBoard(Board &bd, int ix, int n)
{
	if( !ix ) {
		if( n == HG_NS ) return;	//	オԂ͖
		bd.setBlack(0, n);
		int pips = bd.blackPips();
		ushort nm = numMoveBlack(bd);
		qDebug() << bd.position() << QString("%1").arg(QString::number((double)nm/1000, 'f', 4), 7)
					<< pips << "/ 5.333 = " << pips * 3 / 16.0;
	} else {
		for(int i = 0; i <= n; ++i) {
			bd.setBlack(ix, i);
			buildBoard(bd, ix - 1, n - i);
		}
	}
}
//	xAItԂSĐA萔vZ
void TestBoard::buildBoardBearOff()
{
	buildNC(HG_NP + 2, HG_NS);		//	for 1/2 gammon
	g_vNumMove = QVector<ushort>(203490, NOT_YET);
	Board bd;
	buildBoard(bd, 3, 8);	//	xAIt
}
//	p ӏ n ̐΂up^[
template<typename T>
bool init_NC(QVector<T> &v, int p, int n)
{
	if( n < 0 || p <= 0 ) return false;
	v.clear();
	v << n;
	for(int i = 1; i < p; ++i)
		v << 0;
	return true;
}
template<typename T>
bool next_NC(QVector<T> &v)
{
	if( v.size() == 1 ) return false;
	//	{n0 n1, ...}  {n0-1, n1+1, ...}
	//	{0, n1, n2,...}  {n1-1, 0, n2+1, ...}
	//	{0, ... 0, ni, nj,...}  {ni-1, 0, ... 0, nj +1, ...}
	//	{0, ...n}  return false;
	if( v[0] > 0 ) {
		--v[0];
		++v[1];
	} else {
		int i = 1;
		for(;;) {
			if( i == v.size() - 1 ) return false;
			if( v[i] != 0 ) break;
			++i;
		}
		++v[i+1];
		v[0] = v[i] - 1;
		v[i] = 0;
	}
	return true;
}
//	΁ior ΁ĵ݂̒ʂԍvZ
void TestBoard::testSeqNumber()
{
	QVector<int> v;
	v << 1;
	QCOMPARE( seqNumber(v, 0, 1), 0 );
	v[0] = 2;
	v << 0;		//	{2, 0}
	QCOMPARE( seqNumber(v, 1, 2), 0 );
	v[0] = v[1] = 1;		//	{1, 1} 
	QCOMPARE( seqNumber(v, 1, 2), 1 );
	v << 0;		//	{1, 1, 0}
	QCOMPARE( seqNumber(v, 2, 2), 1 );
	v[0] = v[1] = 0;
	v[2] = 2;		//	{2, 0, 0} {1, 1, 0} {0, 2, 0} {1, 0, 1} {0, 1, 1} {0, 0, 2}
	QCOMPARE( seqNumber(v, 2, 2), 5 );
	v[0] = v[1] = v[2] = 0;
	v << 0 << 0 << 0 << 0 << 0 << 0 << 0 << 0 << 0 << 0 << 8;
	QCOMPARE( seqNumber(v, 13, 8), 203490 - 1 );
	{
		QVector<int> v;
		init_NC(v, 3, 3);
		int i = 0;
		while( next_NC(v) ) {
			QCOMPARE( seqNumber(v, 2, 3), ++i );
		}
	}
}

//	gݍ킹p^[
template<typename T>
bool init_C(QVector<T> &v, int x, int y)
{
	if( x <= 0 || x < y || y <= 0 ) return false;
	v.clear();
	for(int i = 0; i < y; ++i)
		v << i;
	return true;
}
template<typename T>
bool next_C(QVector<T> &v, int x)
{
	if( v.isEmpty() ) return false;
	int y = v.size();
	if( ++v[--y] >= x ) {
		for(;;) {
			if( !y ) return false;
			int t = v[--y];
			for(int i = y; i < v.size(); ++i)
				v[i] = ++t;
			if( t < x ) break;
		}
	}
	return true;
}
template<int NP, int NS>
qint64 countPosition()
{
	QVector<int> b, w;
	qint64 sum = 0;
	init_NC(b, NP + 2, NS);
	while( next_NC(b) ) {
		int s = 0;	//	󔒃|Cg
		for(int i = 1; i <= NP; ++i)
			if( !b[i] ) ++s;
		sum += NC2(s + 2, NS) - 1;
#if 0
		init_NC(w, s + 2, NS);
		while( next_NC(w) )
			++sum;
#endif
	}
	return sum;
}

//	G 3 2 1 G	΂2+2
void TestBoard::printNanoGammon()
{
	QVector<int> b, w;
	const int nPoint = 2;
	const int nStone = 2;
	int sn = 0;
	init_NC(b, nPoint + 2, nStone);
	while( next_NC(b) ) {
		int s = 0;	//	󔒃|Cg
		for(int i = 1; i <= nPoint; ++i)
			if( !b[i] ) ++s;
		init_NC(w, s + 2, nStone);
		while( next_NC(w) ) {
			int k = 0;
			QString text = !b[nPoint + 1] ? "G " : QString("%1 ").arg(b[nPoint + 1]);
			for(int i = nPoint; i != 0; --i) {
				if( b[i] != 0 )
					text += QString("%1 ").arg(b[i], 2);
				else
					text += QString("%1 ").arg(-w[++k], 2);
			}
			text += !w[s + 1] ? " G" : QString("%1").arg(-w[s + 1]);
			qDebug() << QString("%1:").arg(++sn, 3) << text;
		}
	}
	qDebug() << countPosition<2, 2>();
#if 0
	QVector<int> v;
	qDebug() << "C:";
	bool b = init_C(v, 5, 3);
	while( b ) {
		qDebug() << v;
		b = next_C(v, 5);
	}
	qDebug() << "NC:";
	b = init_NC(v, 3, 2);
	while( b ) {
		qDebug() << v;
		b = next_NC(v);
	}
	qDebug() << "NC(3, 2) = " << NC(3, 2);
	b = init_NC(v, 3, 3);
	while( b ) {
		qDebug() << v;
		b = next_NC(v);
	}
	qDebug() << "NC(3, 3) = " << NC(3, 3);
	const int nPoint = 3;
	const int nStone = 2;
	for(int B = 0; B <= nStone; ++B) {	//	S/G ȊOō΂̂ӏ
	}
#endif
}
void TestBoard::benchCountPosition()
{
	qint64 c = 0;
	QBENCHMARK_ONCE {
		c = countPosition<8, 5>();
	}
	qDebug() << c;
}
int P(int x, int y)
{
	if( x < y ) return 0;	//	error
	if( !y || x == y ) return 1;
	int t = x;
	int v = x;
	while( --t > x - y )
		v *= t;
	for(t = y; t > 1; --t)
		v /= t;
	return v;
}
//	̃M̏ԐvZ
void TestBoard::countNGPosition()
{
	QCOMPARE( P(3, 2), 3 );
	QCOMPARE( P(4, 2), 6 );
	QCOMPARE( P(5, 3), 10 );

	const int nPoint = 3;
	const int nStone = 1;
	int sum = 0;
	for(int B = 0; B <= nStone; ++B) {	//	S/G ȊOō΂̂ӏ
		int n = P(nPoint, B);	//	ꍇ̐
		sum += n * NC(B+2, nStone - B) * NC((nPoint - B) + 2, nStone);
	}
	qDebug() << "sum = " << sum;
}
