﻿#ifndef BOARD_H
#define BOARD_H

//----------------------------------------------------------------------
//
//			File:			"Board.h"
//			Created:		26-Aug-2012
//			Author:			Nobuhide Tsuda
//			Description:	ボードクラス宣言
//
//----------------------------------------------------------------------

/*

	Copyright (C) 2012 by Nobuhide Tsuda

	HalfGammon のライセンスは CDDL 1.0 です。
	http://opensource.org/licenses/cddl-1.0
	無保証・無サポートですが、無償で利用でき、商用アプリでもソースコードを流用することが可能です。 
	ソースコードを流用した場合、流用したファイルのライセンスは CDDL 1.0 となります。
	流用部分の著作権は HalfGammon のそれのままです。
	作者は、プログラマにとって不自由極まりないのに自由だ自由だと言い張るGPL系が嫌いなので、 
	CDDL 1.0 を選択しました。このライセンスはGPL系より再利用しやすく、GPL系とは矛盾するライセンスなので、
	HalfGammon のソースをGPL系プロジェクトで使用することはできません。

*/

#include <QList>
#include <QByteArray>

class Board;

typedef quint64 uint64;
typedef const char cchar;
typedef unsigned char uchar;

typedef int (*EVAL_FUNC)(const Board &bd);

#define		N_POSITION		1106945
#define		INIT_BOARD		"G B2  W2B1W1B2  W2 G"			//	初期状態

//	期待値
#define		NORMAL_WIN			10000
#define		GAMMONED_WIN		20000
#define		BACKGAMMON_WIN		30000

#define		PACKED		0		//	1 for 配列ではなく uint64 を使用

#if	PACKED
inline int		shift(int ix) { return ix * 4; }
inline uint64	one(int ix) { return (uint64)1 << shift(ix); }
inline uint64	mask(int ix) { return (uint64)0x0f << shift(ix); }
#endif

enum {
	N_PIECE = 5,			//	白黒の石数
	N_DICE = 3,				//	サイコロの目の種類数、値は [0, 2] とする
	MAX_DICE = N_DICE - 1,
	N_INNER = MAX_DICE,		//	インナーのポイント数
	N_POINT = N_INNER * 4,
	SIZE_S_POINTS_G = N_POINT + 2,		//	2 for Start/Goal
	POINT_START = SIZE_S_POINTS_G - 1,		//	Start→Goal でインデックスは減少
	POINT_GOAL = 0,
	IX_START = SIZE_S_POINTS_G - 1,		//	Start→Goal でインデックスは減少、黒の場合は POINT番号と一致
	IX_GOAL = 0,

	LEVEL_RANDOM = 0,		//	完全ランダム
	LEVEL_BEGINNER,			//	初級：ランダム/評価関数＋1手読み
	LEVEL_INTERMEDIATE,		//	中級：評価関数 eval5 ＋1手読み
	LEVEL_ADVANCED,			//	上級：評価関数 eval5 ＋3手読み
	LEVEL_LEARNING,			//	単純な暗記学習
};
inline int reverseIX(int ix) { return IX_START - ix; }
inline int pointToBlackIndex(int p) { return p; }				//	黒はポイント数＝インデックス
inline int blackIndexToPoint(int ix) { return ix; }
inline int pointToWhiteIndex(int p) { return reverseIX(p); }	//	白は左右反転
inline int whiteIndexToPoint(int ix) { return reverseIX(ix); }

int	terminalNodeCount();
void clearTerminalNodeCount();
int evalCallCount();

//	一つの移動
struct Move
{
	uchar	m_src;
	uchar	m_d;

public:
	Move(uchar src = 0, uchar d = 0)
		: m_src(src)
		, m_d(d)
	{
		Q_ASSERT( src <= IX_START );
		Q_ASSERT( src > IX_GOAL || !d );
		Q_ASSERT( d <= MAX_DICE );
	}
	bool	isValid() const { return m_d != 0; }
	QString	toString() const
	{
		return QString("(%1 %2)").arg((int)m_src).arg((int)m_d);
	}
};

typedef QList<Move> Moves;

class Board
{
public:
	Board();
	Board(const Board &);
	Board(cchar *);
#if		PACKED
	Board(uint64 black, uint64 white) : m_black(black), m_white(white) {}
#endif
	~Board() {};

public:
	Board	&operator=(const Board &);
	bool	operator==(const Board &) const;

public:
	QByteArray	hashKey() const;
	int		seqNumber() const;
	double	expectedValue() const;
	bool	doesBlackWin() const { return black(IX_GOAL) == N_PIECE; }
	bool	doesWhiteWin() const { return white(IX_GOAL) == N_PIECE; }
	bool	doesBlackGammonLose() const { return !nBlackGoal(); }
	bool	doesWhiteGammonLose() const { return !nWhiteGoal(); }
#if PACKED
	//bool	doesBlackGammonLose() const { return (m_black & 0xffffffffff0000) != 0; }
	//bool	doesWhiteGammonLose() const { return (m_white & 0xffffffffff0000) != 0; }
	bool	doesBlackBackGammonLose() const { return (m_black & 0xffff0000000000) != 0; }
	bool	doesWhiteBackGammonLose() const { return (m_white & 0xffff0000000000) != 0; }
	uint64	black() const { return m_black; }
	uint64	white() const { return m_white; }
	int		black(int ix) const { return (m_black >> shift(ix)) & 0x0f; }
	int		white(int ix) const { return (m_white >> shift(ix)) & 0x0f; }
#else
	//bool	doesBlackGammonLose() const { return blackTailIndex() > N_DICE; }
	//bool	doesWhiteGammonLose() const { return whiteTailIndex() > N_DICE; }
	bool	doesBlackBackGammonLose() const { return !nBlackGoal() && blackTailIndex() >= IX_START - MAX_DICE; }
	bool	doesWhiteBackGammonLose() const { return !nWhiteGoal() && whiteTailIndex() >= IX_START - MAX_DICE; }
	int		black(int ix) const { return m_black[ix]; }
	int		white(int ix) const { return m_white[ix]; }
	int		nBlack() const;
	int		nWhite() const;
	int		blackPips() const;
	int		whitePips() const;
	cchar	*blackPtr() const { return m_black; }
	cchar	*whitePtr() const { return m_white; }
#endif
	char	position(int pnt) const { return black(pointToBlackIndex(pnt))
												? black(pointToBlackIndex(pnt))
												: -white(pointToWhiteIndex(pnt)); }
	QString	position() const;
	int		nBlackGoal() const { return black(IX_GOAL); }
	int		nWhiteGoal() const { return white(IX_GOAL); }
	int		nBlackBar() const { return black(IX_START); }
	int		nWhiteBar() const { return white(IX_START); }
#if		PACKED
	bool	canBlackBearOff() const { return !(m_black & 0xffffffffff0000); }
	bool	canWhiteBearOff() const { return !(m_white & 0xffffffffff0000); }
#else
	bool	canBlackBearOff() const;
	bool	canWhiteBearOff() const;
#endif
	int		blackTailIndex() const;
	int		whiteTailIndex() const;
	bool	canBlackMoveSrcDst(int, int) const;
	bool	canWhiteMoveSrcDst(int, int) const;
	Moves	blackMoves(int d1) const;		//	指定ダイスの目で可能な着手をリストアップ
	Moves	whiteMoves(int d1) const;		//	指定ダイスの目で可能な着手をリストアップ
	Moves	blackMoves(int d1, int d2) const;		//	指定ダイスの目で可能な着手をリストアップ
	Moves	whiteMoves(int d1, int d2) const;		//	指定ダイスの目で可能な着手をリストアップ
	QList<Moves>	blackMovesList0(int d1, int d2) const;
	QList<Moves>	whiteMovesList0(int d1, int d2) const;	//	全ての可能着手を生成
	QList<Moves>	blackMovesList(int d1, int d2) const;
	QList<Moves>	whiteMovesList(int d1, int d2) const;	//	勝ちの手があれば、それのみを生成
	QList<Moves>	whiteMovesList2(int d1, int d2) const;
	QList<Moves>	whiteMovesList3(int d1, int d2) const;
	QList<Moves>	blackMovesList3(int d1, int d2) const;
	int		eval5() const;		//	ピップス数をメインにした評価関数
	int		eval6() const;		//	ピップス数をメインにした評価関数
	Moves	whiteMovesEval5PlyD(int depth, int d1, int d2, double &) const;	//	評価関数＋depth手読みにより着手を決定
	Moves	whiteMovesEvalXPlyD(int depth, int d1, int d2, double &, EVAL_FUNC) const;	//	評価関数＋depth手読みにより着手を決定
	Moves	whiteMovesLearning(int depth, int d1, int d2, double &) const;	//	評価値テーブルを使い着手を決定＆テーブルの値を更新
	Moves	whiteMovesRandom(int d1, int d2) const;		//	ランダムに着手を選ぶ
	Moves	blackMoves(int level, int d1, int d2, double &) const;
	Moves	whiteMoves(int level, int d1, int d2, double &) const;

	double	whiteExpectedValue() const;		//	白番での期待値算出
	double	blackExpectedValue() const;		//	黒番での期待値算出

public:
	void	setPosition(cchar *pat);
#if	PACKED
	void	setBlack(int ix, char v) { clearBlack(ix); m_black += (uint64)v << shift(ix); }
	void	setWhite(int ix, char v) { clearWhite(ix); m_white += (uint64)v << shift(ix); }
	void	incBlack(int ix) { m_black += one(ix); }
	void	incWhite(int ix) { m_white += one(ix); }
	void	decBlack(int ix) { m_black -= one(ix); }
	void	decWhite(int ix) { m_white -= one(ix); }
	void	clearBlack(int ix) { m_black &= ~mask(ix); }
	void	clearWhite(int ix) { m_white &= ~mask(ix); }
#else
	void	setBlack(int ix, char v) { m_black[ix] = v; }
	void	setWhite(int ix, char v) { m_white[ix] = v; }
	void	incBlack(int ix) { ++m_black[ix]; }
	void	incWhite(int ix) { ++m_white[ix]; }
	void	decBlack(int ix) { --m_black[ix]; }
	void	decWhite(int ix) { --m_white[ix]; }
	void	clearBlack(int ix) { m_black[ix] = 0; }
	void	clearWhite(int ix) { m_white[ix] = 0; }
#endif
	void	swapBW() { qSwap(m_black, m_white); }
	void	moveBlack(int src, int dst);
	void	moveWhite(int src, int dst);
	void	moveBlack(Move mv) { moveBlack(mv.m_src, (int)mv.m_src - mv.m_d); }
	void	moveWhite(Move mv) { moveWhite(mv.m_src, (int)mv.m_src - mv.m_d); }

private:
#if	PACKED
	uint64	m_black;
	uint64	m_white;
#else
	char	*m_black;
	char	*m_white;
	char	m_array[SIZE_S_POINTS_G * 2];
#endif
};
int eval5(const Board &bd);
int eval6(const Board &bd);
int eval6black(const Board &bd);	//	黒番での評価関数
double negaChanceNodeEvalX(const Board &bd, int);
void setEvalFunc(EVAL_FUNC);

int PNC(int p, int n);

//	p 箇所に n 個の石を置くパターン生成
template<typename Container>
bool init_PN(Container &v, int p, int n)
{
	if( n < 0 || p <= 0 ) return false;
	v.clear();
	v << n;
	for(int i = 1; i < p; ++i)
		v << 0;
	return true;
}
template<typename Container>
bool next_PN(Container &v, int vSize)
{
	///if( v.size() == 1 ) return false;
	//	{n0 n1, ...} → {n0-1, n1+1, ...}
	//	{0, n1, n2,...} → {n1-1, 0, n2+1, ...}
	//	{0, ... 0, ni, nj,...} → {ni-1, 0, ... 0, nj +1, ...}
	//	{0, ...n} → return false;
	if( v[0] > 0 ) {
		--v[0];
		++v[1];
	} else {
		int i = 1;
		for(;;) {
			if( i == vSize - 1 ) return false;
			//if( i == v.size() - 1 ) return false;
			if( v[i] != 0 ) break;
			++i;
		}
		++v[i+1];
		v[0] = v[i] - 1;
		v[i] = 0;
	}
	return true;
}
template<typename Container>
bool prev_PN(Container &v, int vSize)
{
	//	{n1, n2, ...} → {n1+1, n2-1, ...}
	//	{n1, 0, n2,...} → {0, n1+1, n2-1, ...}
	//	{n1, 0, ... 0, n2,,...} → {0, ...0, n1+1, nj -1, ...}
	//	{n, 0, ...0} → return false;
	if( v[1] > 0 ) {
		++v[0];
		--v[1];
	} else {
		int i = 2;
		for(;;) {	//	v[1] 以降で v[i] != 0 を探す
			if( i >= vSize ) return false;
			if( v[i] != 0 ) break;
			++i;
		}
		--v[i];
		v[i-1] = v[0] + 1;
		v[0] = 0;
	}
	return true;
}

#endif // BOARD_H
