﻿//----------------------------------------------------------------------
//
//			File:			"Board.cpp"
//			Created:		26-Aug-2012
//			Author:			Nobuhide Tsuda
//			Description:	ボードクラス実装
//
//----------------------------------------------------------------------

/*

	Copyright (C) 2012 by Nobuhide Tsuda

	HalfGammon のライセンスは CDDL 1.0 です。
	http://opensource.org/licenses/cddl-1.0
	無保証・無サポートですが、無償で利用でき、商用アプリでもソースコードを流用することが可能です。 
	ソースコードを流用した場合、流用したファイルのライセンスは CDDL 1.0 となります。
	流用部分の著作権は HalfGammon のそれのままです。
	作者は、プログラマにとって不自由極まりないのに自由だ自由だと言い張るGPL系が嫌いなので、 
	CDDL 1.0 を選択しました。このライセンスはGPL系より再利用しやすく、GPL系とは矛盾するライセンスなので、
	HalfGammon のソースをGPL系プロジェクトで使用することはできません。

*/


#include <QHash>
#include <QSet>
#include "Board.h"

int (*g_eval_func)(const Board &bd);

Board::Board()
{
#if PACKED
	m_black = m_white = 0;
#else
	m_white = m_array;
	m_black = m_array + SIZE_S_POINTS_G;
	for(int i = 0; i < sizeof(m_array); ++i)
		m_array[i] = 0;
#endif
}
Board::Board(cchar *pat)
{
#if	PACKED
#else
	m_white = m_array;
	m_black = m_array + SIZE_S_POINTS_G;
#endif
	setPosition(pat);
}
Board::Board(const Board &x)
{
#if	PACKED
	m_black = x.m_black;
	m_white = x.m_white;
#else
	if( x.m_white == x.m_array ) {
		m_white = m_array;
		m_black = m_array + SIZE_S_POINTS_G;
	} else {
		m_white = m_array + SIZE_S_POINTS_G;
		m_black = m_array;
	}
	for(int i = 0; i < sizeof(m_array); ++i)
		m_array[i] = x.m_array[i];
#endif
}
Board &Board::operator=(const Board &x)
{
	if( &x == this ) return *this;
#if	PACKED
	m_black = x.m_black;
	m_white = x.m_white;
#else
	if( x.m_white == x.m_array ) {
		m_white = m_array;
		m_black = m_array + SIZE_S_POINTS_G;
	} else {
		m_white = m_array + SIZE_S_POINTS_G;
		m_black = m_array;
	}
	for(int i = 0; i < sizeof(m_array); ++i)
		m_array[i] = x.m_array[i];
#endif
	return *this;
}
bool Board::operator==(const Board &x) const
{
#if	PACKED
	return m_black == x.m_black && m_white == x.m_white;
#else
	for(int i = 0; i < SIZE_S_POINTS_G; ++i) {
		if( m_black[i] != x.m_black[i] ||
			m_white[i] != x.m_white[i] )
		{
			return false;
		}
	}
	return true;
#endif
}
QByteArray Board::hashKey() const
{
#if PACKED
	QByteArray ba(64/8*2, 0);
	*(uint64*)ba.data() = m_black;
	*((uint64*)ba.data() + 1) = m_white;
#else
	QByteArray ba(SIZE_S_POINTS_G*2, 0);
	for(int i = 0; i < SIZE_S_POINTS_G; ++i) {
		ba[i] = black(i);
		ba[i + SIZE_S_POINTS_G] = white(i);
	}
#endif
	return ba;
}
QString Board::position() const
{
	QString text;
	int n = black(IX_START);
	if( !n )
		text = "G ";
	else
		text = QString("B%1").arg(n);
	for(int i = POINT_START - 1; i > POINT_GOAL; --i) {
		n = position(i);
		if( n > 0 )
			text += QString("B%1").arg(n);
		else if( n < 0 )
			text += QString("W%1").arg(-n);
		else
			text += ". ";
	}
	n = white(IX_START);
	if( !n )
		text += " G";
	else
		text += QString("W%1").arg(n);
	return text;
}
//	黒石：B or X or >
//	白石：W or O or <
//	           242322212019181716151413121110 9 8 7 6 5 4 3 2 1
//	初期値："S B2        W5  W3      B5W5      B3  B5        W2 S"
//	ゴールにある石数は指定しなくても、自動的に計算するものとする
//	各位置に10個以上の石がある場合は数字を2桁記述する（桁数はその分増える）
void Board::setPosition(cchar *pat)
{
	int nBlack = 0;
	int nWhite = 0;
	for(int i = POINT_START; i >= POINT_GOAL; --i) {
		char t = *pat++;
		char n = *pat++ - '0';
		if( *pat >= '0' && *pat <= '9' )
			n = n * 10 + (*pat++ - '0');
		switch( t ) {
		case 'B':
		case 'X':
		case '>':
			Q_ASSERT( n <= 15 );
			nBlack += n;
			setBlack(i, n);
			setWhite(reverseIX(i), 0);
			break;
		case 'W':
		case 'O':
		case '<':
			Q_ASSERT( n <= 15 );
			nWhite += n;
			setWhite(reverseIX(i), n);
			setBlack(i, 0);
			break;
		default:
			setBlack(i, 0);
			setWhite(reverseIX(i), 0);
		}
	}
	Q_ASSERT( nBlack <= N_PIECE );
	Q_ASSERT( nWhite <= N_PIECE );
	setBlack(IX_GOAL, N_PIECE - nBlack);
	setWhite(IX_GOAL, N_PIECE - nWhite);
}
#if	PACKED
//	最後尾の石のインデックスを返す
//	下位ビットがゴール方向
//	Ｓ 12 11 10 ９ ８ ７ ６ ５ ４ ３ ２ １ Ｇ
int tailIndex(uint64 bb)
{
	if( !(bb & 0xffffff00000000) ) {	//	上位６ポイントには石が無い
		if( !(bb & 0xffff0000) ) {		//	4～7ポイントに石が無い
			if( !(bb & 0xff00) ) {
				if( !(bb & 0xf0) ) return 0;
				return 1;
			} else {
				if( !(bb & 0xf000) ) return 2;
				return 3;
			}
		} else {	//	4～7ポイントに石がある
			if( !(bb & 0xff000000) ) {
				if( !(bb & 0xf00000) ) return 4;
				return 5;
			} else {	//	8, 9 ポイントに石がある
				if( !(bb & 0xf0000000) ) return 6;
				return 7;
			}
		}
	} else {	//	Ｓ～８ポイントに石がある
		if( !(bb & 0xff000000000000) ) {
			if( !(bb & 0xff0000000000) ) {
				if( !(bb & 0xf000000000) ) return 8;
				return 9;
			} else {
				if( !(bb & 0xf00000000000) ) return 10;
				return 11;
			}
		} else {	//	下位４ポイントに石がある
			if( !(bb & 0xf0000000000000) ) return 12;
			return 13;
		}
	}
}
int Board::blackTailIndex() const
{
	return tailIndex(black());
}
int Board::whiteTailIndex() const
{
	return tailIndex(white());
}
#else
bool Board::canBlackBearOff() const
{
	return m_black[IX_GOAL] +
				m_black[IX_GOAL + 1] + m_black[IX_GOAL + 2] + 
				m_black[IX_GOAL + 3] + m_black[IX_GOAL + 4]
			== N_PIECE;
}
bool Board::canWhiteBearOff() const
{
	return m_white[IX_GOAL] +
				m_white[IX_GOAL + 1] + m_white[IX_GOAL + 2] + 
				m_white[IX_GOAL + 3] + m_white[IX_GOAL + 4]
			== N_PIECE;
}
//	処理時間は O(N) だが、Nは少ないし、処理が軽いので良しとしておく
int Board::blackTailIndex() const
{
	for(int ix = IX_START; ix > IX_GOAL; --ix)
		if( black(ix) != 0 ) return ix;
	return IX_GOAL;
}
int Board::whiteTailIndex() const
{
	for(int ix = IX_START; ix > IX_GOAL; --ix)
		if( white(ix) != 0 ) return ix;
	return IX_GOAL;
}
#endif
int Board::nBlack() const
{
	int n = 0;
	for(int i = IX_START; i >= IX_GOAL; --i)
		n += black(i);
	return n;
}
int Board::nWhite() const
{
	int n = 0;
	for(int i = IX_START; i >= IX_GOAL; --i)
		n += white(i);
	return n;
}
bool Board::canBlackMoveSrcDst(int src, int dst) const
{
	Board bd(*this);
	bd.swapBW();
	return bd.canWhiteMoveSrcDst(src, dst);
}
bool Board::canWhiteMoveSrcDst(int src, int dst) const
{
	if( !white(src) ) return false;
	if( canWhiteBearOff() ) {
		return dst < IX_GOAL && src == whiteTailIndex()
				|| dst == IX_GOAL
				|| dst > IX_GOAL && black(reverseIX(dst)) <= 1;
	} else
		return dst > IX_GOAL && black(reverseIX(dst)) <= 1;
}
//	指定ダイスの目で可能な着手をリストアップ
Moves Board::blackMoves(int d1, int d2) const
{
	Board bd(*this);
	bd.swapBW();
	return bd.whiteMoves(d1, d2);
}
//	d1 または d2 を使用して移動出来る着手をリストアップする
Moves Board::whiteMoves(int d1, int d2) const
{
	Q_ASSERT( !doesWhiteWin() );
	Moves lst = whiteMoves(d1);
	if( d2 != 0 && d2 != d1 )
		lst << whiteMoves(d2);
	return lst;
}
//	d1 を使用して移動出来る着手をリストアップする
Moves Board::blackMoves(int d1) const
{
	Board bd(*this);
	bd.swapBW();
	return bd.whiteMoves(d1);
}
//	d1 を使用して移動出来る着手をリストアップする
Moves Board::whiteMoves(int d1) const
{
	Q_ASSERT( !doesWhiteWin() );
	Q_ASSERT( d1 != 0 );
	Moves lst;
	if( nWhiteBar() ) {	//	バーに石がある場合
		if( black(d1) <= 1 )
			lst << Move(IX_START, d1);
	} else {
		if( canWhiteBearOff() ) {
			int i = N_DICE;
			while( white(i) == 0 ) --i;		//	白石は必ず存在するはず
			if( i <= d1 )	//	最後尾の石はポイント数以上のサイの目で上がれる
				lst << Move(IX_GOAL + i, d1);
			else if( white(d1) != 0 )	//	サイの目でちょうど上がれる場合
				lst << Move(IX_GOAL + d1, d1);
		}
		//	ゴールしない着手をリストアップ
		for(int i = IX_START - 1; i > IX_GOAL + 1; --i) {
			if( white(i) != 0 ) {
				int dst = i - d1;
				if( dst > IX_GOAL && black(reverseIX(dst)) <= 1 )
					lst << Move(i, d1);
			}
		}
	}
	return lst;
}
QList<Moves> Board::blackMovesList0(int d1, int d2) const
{
	Board bd(*this);
	bd.swapBW();
	return bd.whiteMovesList0(d1, d2);
}
QList<Moves> Board::whiteMovesList0(int d1, int d2) const
{
	Q_ASSERT( !doesWhiteWin() );
	QSet<QByteArray> set;
	QList<Moves> mvsList;
	Moves lst1 = whiteMoves(d1);
	if( d2 != d1 )
		lst1 << whiteMoves(d2);
	if( lst1.isEmpty() ) return mvsList;
	foreach(Move mv, lst1) {
		Board bd(*this);
		bd.moveWhite(mv);
		Moves curMvs;
		curMvs << mv;		//	初手を追加
		if( bd.doesWhiteWin() ) {		//	１手でゲーム勝利
			if( !set.contains(bd.hashKey()) ) {
				mvsList << curMvs;
				set << bd.hashKey();
			}
			continue;
		}
		if( d1 != d2 ) {	//	ゾロ目以外の場合
			int nd = mv.m_d == d1 ? d2 : d1;	//	残りのサイコロの目
			Moves lst2 = bd.whiteMoves(nd);
			if( lst2.isEmpty() ) {
				if( !set.contains(bd.hashKey()) ) {
					mvsList << curMvs;
					set << bd.hashKey();
				}
				continue;
			}
			foreach(Move mv2, lst2) {
				Board bd2(bd);
				bd2.moveWhite(mv2);
				if( !set.contains(bd2.hashKey()) ) {
					curMvs << mv2;		//	2手目を追加
					mvsList << curMvs;
					set << bd2.hashKey();
					curMvs.pop_back();
				}
			}
		} else {	//	ゾロ目の場合、同じ目を4回使用する
			Moves lst2 = bd.whiteMoves(d1);
			if( lst2.isEmpty() ) {
				if( !set.contains(bd.hashKey()) ) {
					mvsList << curMvs;
					set << bd.hashKey();
				}
				continue;
			}
			foreach(Move mv2, lst2) {
				Board bd2(bd);
				bd2.moveWhite(mv2);
				curMvs << mv2;		//	2手目を追加
				Moves lst3;
				if( bd2.doesWhiteWin() || (lst3 = bd2.whiteMoves(d1)).isEmpty() ) {
					if( !set.contains(bd2.hashKey()) ) {
						mvsList << curMvs;
						set << bd2.hashKey();
					}
				} else {
					foreach(Move mv3, lst3) {
						Board bd3(bd2);
						bd3.moveWhite(mv3);
						curMvs << mv3;		//	3手目を追加
						Moves lst4;
						if( bd3.doesWhiteWin() || (lst4 = bd3.whiteMoves(d1)).isEmpty() ) {
							if( !set.contains(bd3.hashKey()) ) {
								mvsList << curMvs;
								set << bd3.hashKey();
							}
						} else {
							foreach(Move mv4, lst4) {
								Board bd4(bd3);
								bd4.moveWhite(mv4);
								if( !set.contains(bd4.hashKey()) ) {
									curMvs << mv4;		//	4手目を追加
									mvsList << curMvs;
									set << bd4.hashKey();
									curMvs.pop_back();
								}
							}
						}
						curMvs.pop_back();
					}
				}
				curMvs.pop_back();
			}
		}
	}
	return mvsList;
}
QList<Moves> Board::blackMovesList(int d1, int d2) const
{
	Board bd(*this);
	bd.swapBW();
	return bd.whiteMovesList(d1, d2);
}
//	全ての石をゴール出来る場合は、その手だけを返す
QList<Moves> Board::whiteMovesList(int d1, int d2) const
{
	Q_ASSERT( !doesWhiteWin() );
	QSet<QByteArray> set;
	QList<Moves> mvsList;
	Moves lst1 = whiteMoves(d1);
	if( d2 != d1 )
		lst1 << whiteMoves(d2);
	if( lst1.isEmpty() ) return mvsList;
	foreach(Move mv, lst1) {
		Board bd(*this);
		bd.moveWhite(mv);
		Moves curMvs;
		curMvs << mv;		//	初手を追加
		if( bd.doesWhiteWin() ) {		//	１手でゲーム勝利
			mvsList.clear();
			mvsList << curMvs;
			return mvsList;
		}
		if( d1 != d2 ) {	//	ゾロ目以外の場合
			int nd = mv.m_d == d1 ? d2 : d1;	//	残りのサイコロの目
			Moves lst2 = bd.whiteMoves(nd);
			if( lst2.isEmpty() ) {
				if( !set.contains(bd.hashKey()) ) {
					mvsList << curMvs;
					set << bd.hashKey();
				}
				continue;
			}
			foreach(Move mv2, lst2) {
				Board bd2(bd);
				bd2.moveWhite(mv2);
				if( !set.contains(bd2.hashKey()) ) {
					curMvs << mv2;		//	2手目を追加
					mvsList << curMvs;
					set << bd2.hashKey();
					curMvs.pop_back();
				}
			}
		} else {	//	ゾロ目の場合、同じ目を4回使用する
			Moves lst2 = bd.whiteMoves(d1);
			if( lst2.isEmpty() ) {
				if( !set.contains(bd.hashKey()) ) {
					mvsList << curMvs;
					set << bd.hashKey();
				}
				continue;
			}
			foreach(Move mv2, lst2) {
				Board bd2(bd);
				bd2.moveWhite(mv2);
				curMvs << mv2;		//	2手目を追加
				Moves lst3;
				if( bd2.doesWhiteWin() ) {
					mvsList.clear();
					mvsList << curMvs;
					return mvsList;
				}
				if( (lst3 = bd2.whiteMoves(d1)).isEmpty() ) {
					if( !set.contains(bd2.hashKey()) ) {
						mvsList << curMvs;
						set << bd2.hashKey();
					}
				} else {
					foreach(Move mv3, lst3) {
						Board bd3(bd2);
						bd3.moveWhite(mv3);
						curMvs << mv3;		//	3手目を追加
						Moves lst4;
						if( bd3.doesWhiteWin() ) {
							mvsList.clear();
							mvsList << curMvs;
							return mvsList;
						}
						if( (lst4 = bd3.whiteMoves(d1)).isEmpty() ) {
							if( !set.contains(bd3.hashKey()) ) {
								mvsList << curMvs;
								set << bd3.hashKey();
							}
						} else {
							foreach(Move mv4, lst4) {
								Board bd4(bd3);
								bd4.moveWhite(mv4);
								if( !set.contains(bd4.hashKey()) ) {
									curMvs << mv4;		//	4手目を追加
									mvsList << curMvs;
									set << bd4.hashKey();
									curMvs.pop_back();
								}
							}
						}
						curMvs.pop_back();
					}
				}
				curMvs.pop_back();
			}
		}
	}
	return mvsList;
}
//	d1 != d2 の場合に、QSet を使わない版
QList<Moves> Board::whiteMovesList3(int d1, int d2) const
{
	Q_ASSERT( !doesWhiteWin() );
	Q_ASSERT( (cchar *)m_black >= (cchar *)this );
	Q_ASSERT( (cchar *)m_black < (cchar *)this + sizeof(Board) );
	Q_ASSERT( (cchar *)m_white >= (cchar *)this );
	Q_ASSERT( (cchar *)m_white < (cchar *)this + sizeof(Board) );
	QList<Moves> mvsList;
	if( d1 != d2 ) {	//	ゾロ目以外の場合
		if( d1 < d2 ) qSwap(d1, d2);	//	make sure d1 > d2
		Moves mvs;
		if( nWhiteBar() >= 2 ) {	//	バーに２個
			if( black(d1) <= 1 )
				mvs << Move(IX_START, d1);
			if( black(d2) <= 1 )
				mvs << Move(IX_START, d2);
			if( !mvs.isEmpty() )
				mvsList << mvs;
		} else if( nWhiteBar() == 1 ) {	//	バーに１個
			//	バーに１個の場合、[1] どちらの目でも出れない [2] 片方でのみ出れる
			//	[3] 両方で出れる の３通りがある
			//char wa[SIZE_S_POINTS_G];
			if( black(d1) > 1 ) {
				if( black(d2) > 1 )	//	どちらの目でも出れない
					return mvsList;
				else {	//	d2 でのみ出られる
					mvs << Move(IX_START, d2) << Move(0, 0);	//	２つめはダミー
					++m_white[IX_START - d2];
					for(int src = IX_START - 1; src > IX_GOAL + d1; --src) {
						if( white(src) != 0 && black(reverseIX(src - d1)) <= 1 ) {
							mvs[1] = Move(src, d1);
							mvsList << mvs;
						}
					}
					--m_white[IX_START - d2];
					if( mvsList.isEmpty() ) {
						mvs.pop_back();
						mvsList << mvs;
					}
				}
			} else {
				if( black(d2) > 1 ) {	//	d1 でのみ出られる
					mvs << Move(IX_START, d1) << Move(0, 0);	//	２つめはダミー
					++m_white[IX_START - d1];
					for(int src = IX_START - 1; src > IX_GOAL + d2; --src) {
						if( white(src) != 0 && black(reverseIX(src - d2)) <= 1 ) {
							mvs[1] = Move(src, d2);
							mvsList << mvs;
						}
					}
					--m_white[IX_START - d1];
					if( mvsList.isEmpty() ) {
						mvs.pop_back();
						mvsList << mvs;
					}
				} else {	//	どちらの目でも出られる
					mvs << Move(IX_START, d1) << Move(0, 0);	//	２つめはダミー
					++m_white[IX_START - d1];
					for(int src = IX_START - 1; src > IX_GOAL + d2; --src) {
						if( white(src) != 0 && black(reverseIX(src - d2)) <= 1 ) {
							mvs[1] = Move(src, d2);
							mvsList << mvs;
						}
					}
					--m_white[IX_START - d1];
					mvs[0] = Move(IX_START, d2);
					++m_white[IX_START - d2];
					for(int src = IX_START - 1; src > IX_GOAL + d1; --src) {
						if( white(src) != 0 && black(reverseIX(src - d1)) <= 1 ) {
							mvs[1] = Move(src, d1);
							mvsList << mvs;
						}
					}
					--m_white[IX_START - d2];
				}
			}
		} else if( canWhiteBearOff() ) {
			mvs.reserve(2);
			mvs << Move(0, 0) << Move(0, 0);
			bool twoMovedAdded = false;		//	サイの目を２つ使った
			bool oneMovedAdded = false;		//	サイの目をひとつだけ使う手をリストに加えた
			//	条件：
			//		[1] 石が１個だけで、それを上げることができれば、着手は１個のみとする
			//		[2] 石が２個でも、両方上げることができれば、着手は１個のみとする
			//		[3] mv1.m_src >= mv2.m_src		遠い方を必ず先に動かす
			//		[4] mv1.m_src - mv1.m_d == mv2.m_src （同じ石を続けて移動）の場合、
			//			mv1.m_src - mv1.m_d1, mv1.m_src - mv1.m_d2 の両方に黒石が無ければ
			//			m_d1 > m_d2 とする
			int tailIndex = 0;
			for(int src1 = IX_GOAL + N_DICE; src1 > IX_GOAL; --src1) {
				if( white(src1) == 0 ) continue;
				if( !tailIndex ) tailIndex = src1;
				int dst1 = src1 - d1;	//	先に d1 で移動する場合
				if( dst1 > IX_GOAL && black(reverseIX(dst1)) <= 1
					|| src1 == d1
					|| src1 == tailIndex && d1 >= src1 )
				{
					mvs[0] = Move(src1, d1);
					--m_white[src1];	//	白石を移動。黒石はヒットされたとしても移動しない
					if( dst1 >= IX_GOAL ) ++m_white[dst1]; else ++m_white[IX_GOAL];
					if( doesWhiteWin() ) {
						if( dst1 >= IX_GOAL ) --m_white[dst1]; else --m_white[IX_GOAL];
						++m_white[src1];
						mvsList.clear();
						mvs.pop_back();
						return mvsList << mvs;
					}
					const int sz = mvsList.size();
					int src2 = src1;
					//if( canWhiteBearOff() ) {
						src2 = IX_GOAL + N_DICE;
						while( !white(src2) ) --src2;	//	最後尾を探す
						if( src2 <= d2 ) {	//	最後尾の石はポイント数以上のサイの目で上がれる
							mvs[1] = Move(src2, d2);
							if( nWhiteGoal() == N_PIECE - 1 ) {
								if( dst1 >= IX_GOAL ) --m_white[dst1]; else --m_white[IX_GOAL];
								++m_white[src1];
								mvsList.clear();
								return mvsList << mvs;
							}
							mvsList << mvs;
						} else if( white(IX_GOAL + d2) != 0 ) {		//	d2 でちょうど上がれる場合
							mvs[1] = Move(IX_GOAL + d2, d2);
							if( nWhiteGoal() == N_PIECE - 1 ) {
								if( dst1 >= IX_GOAL ) --m_white[dst1]; else --m_white[IX_GOAL];
								++m_white[src1];
								mvsList.clear();
								return mvsList << mvs;
							}
							mvsList << mvs;
						}
					//}
					for(; src2 > IX_GOAL + 1; --src2) {		//	src2 == src1 を含む
						if( white(src2) == 0 ) continue;
						int dst2 = src2 - d2;
						if( dst2 > IX_GOAL && black(reverseIX(dst2)) <= 1 ) {
							mvs[1] = Move(src2, d2);
							mvsList << mvs;
							twoMovedAdded = true;
						}
					}
					if( dst1 >= IX_GOAL ) --m_white[dst1]; else --m_white[IX_GOAL];
					++m_white[src1];
					if( !twoMovedAdded && mvsList.size() == sz ) {
						mvs.pop_back();
						mvsList << mvs;
						mvs << Move(0, 0);
						oneMovedAdded = true;
					}
				}
				dst1 = src1 - d2;		//	先に d2 で移動する場合
				if( dst1 > IX_GOAL && black(reverseIX(dst1)) <= 1 ) {
					mvs[0] = Move(src1, d2);
					--m_white[src1];	//	白石を移動。黒石はヒットされたとしても移動しない
					++m_white[dst1];
					bool removed = false;
					const int sz = mvsList.size();
					int src2 = src1 - 1;	//	src2 == src1 を含まない
					//if( canWhiteBearOff() ) {
						src2 = IX_GOAL + N_DICE;
						while( !white(src2) ) --src2;	//	最後尾を探す
						if( src2 <= d1 ) {	//	最後尾の石はポイント数以上のサイの目で上がれる
							mvs[1] = Move(src2, d1);
							if( nWhiteGoal() == N_PIECE - 1 ) {
								--m_white[dst1];
								++m_white[src1];
								mvsList.clear();
								return mvsList << mvs;
							}
							mvsList << mvs;
						} else if( white(IX_GOAL + d1) != 0 ) {		//	d1 でちょうど上がれる場合
							mvs[1] = Move(IX_GOAL + d1, d1);
							if( nWhiteGoal() == N_PIECE - 1 ) {
								--m_white[dst1];
								++m_white[src1];
								mvsList.clear();
								return mvsList << mvs;
							}
							mvsList << mvs;
						}
					//}
					for(; src2 > IX_GOAL + 1; --src2) {
						if( src2 == dst1 )	//	同じ石を続けて移動する場合
							if( !black(reverseIX(dst1)) && !black(reverseIX(src1 - d1)) ) {
								removed = true;
								continue;	//	ダイス使用順に依らない場合
							}
						if( white(src2) == 0 ) continue;
						int dst2 = src2 - d1;
						if( dst2 > IX_GOAL && black(reverseIX(dst2)) <= 1 ) {
							mvs[1] = Move(src2, d1);
							mvsList << mvs;
							twoMovedAdded = true;
						}
					}
					--m_white[dst1];
					++m_white[src1];
					if( !twoMovedAdded && !removed && mvsList.size() == sz ) {
						mvs.pop_back();
						mvsList << mvs;
						mvs << Move(0, 0);
						oneMovedAdded = true;
					}
				}
			}
			if( twoMovedAdded && oneMovedAdded ) {
				for(int i = 0; i < mvsList.size();) {
					Move mv = mvsList[i][0];	//	for Debug
					if( mvsList[i].size() == 1 )
						mvsList.removeAt(i);
					else
						++i;
				}
			}
		} else {	//	ベアオフ不可能な場合
			mvs.reserve(2);
			mvs << Move(0, 0) << Move(0, 0);
			bool twoMovedAdded = false;		//	サイの目を２つ使った
			bool oneMovedAdded = false;		//	サイの目をひとつだけ使う手をリストに加えた
			//	条件：
			//		[0] 全てゴールすることが出来る場合は、その着手のみを返す。
			//		[1] mv1.m_src >= mv2.m_src		遠い方を必ず先に動かす
			//		[2] mv1.m_src - mv1.m_d == mv2.m_src （同じ石を続けて移動）の場合、
			//			mv1.m_src - mv1.m_d1, mv1.m_src - mv1.m_d2 の両方に黒石が無ければ
			//			m_d1 > m_d2 とする
			for(int src1 = IX_START - 1; src1 > IX_GOAL + 1; --src1) {
				if( white(src1) == 0 ) continue;
				int dst1 = src1 - d1;	//	先に d1 で移動する場合
				if( dst1 > IX_GOAL && black(reverseIX(dst1)) <= 1 ) {
					mvs[0] = Move(src1, d1);
					--m_white[src1];	//	白石を移動。黒石はヒットされたとしても移動しない
					++m_white[dst1];
					const int sz = mvsList.size();
					int src2 = src1;
					if( canWhiteBearOff() ) {
						src2 = IX_GOAL + N_DICE;
						while( !white(src2) ) --src2;	//	最後尾を探す
						if( src2 <= d2 ) {	//	最後尾の石はポイント数以上のサイの目で上がれる
							mvs[1] = Move(src2, d2);
							if( nWhiteGoal() == N_PIECE - 1 ) {
								--m_white[dst1];
								++m_white[src1];
								mvsList.clear();
								return mvsList << mvs;
							}
							mvsList << mvs;
						} else if( white(IX_GOAL + d2) != 0 ) {		//	d2 でちょうど上がれる場合
							mvs[1] = Move(IX_GOAL + d2, d2);
							if( nWhiteGoal() == N_PIECE - 1 ) {
								--m_white[dst1];
								++m_white[src1];
								mvsList.clear();
								return mvsList << mvs;
							}
							mvsList << mvs;
						}
					}
					for(; src2 > IX_GOAL + 1; --src2) {		//	src2 == src1 を含む
						if( white(src2) == 0 ) continue;
						int dst2 = src2 - d2;
						if( dst2 > IX_GOAL && black(reverseIX(dst2)) <= 1 ) {
							mvs[1] = Move(src2, d2);
							mvsList << mvs;
							twoMovedAdded = true;
						}
					}
					--m_white[dst1];
					++m_white[src1];
					if( !twoMovedAdded && mvsList.size() == sz ) {
						mvs.pop_back();
						mvsList << mvs;
						mvs << Move(0, 0);
						oneMovedAdded = true;
					}
				}
				dst1 = src1 - d2;		//	先に d2 で移動する場合
				if( dst1 > IX_GOAL && black(reverseIX(dst1)) <= 1 ) {
					mvs[0] = Move(src1, d2);
					--m_white[src1];	//	白石を移動。黒石はヒットされたとしても移動しない
					++m_white[dst1];
					bool removed = false;
					const int sz = mvsList.size();
					int src2 = src1;	//	src2 == src1 を含む
					//int src2 = src1 - 1;	//	src2 == src1 を含まない
					if( canWhiteBearOff() ) {
						src2 = IX_GOAL + N_DICE;
						while( !white(src2) ) --src2;	//	最後尾を探す
						if( src2 <= d1 ) {	//	最後尾の石はポイント数以上のサイの目で上がれる
							mvs[1] = Move(src2, d1);
							if( nWhiteGoal() == N_PIECE - 1 ) {
								--m_white[dst1];
								++m_white[src1];
								mvsList.clear();
								return mvsList << mvs;
							}
							mvsList << mvs;
						} else if( white(IX_GOAL + d1) != 0 ) {		//	d1 でちょうど上がれる場合
							mvs[1] = Move(IX_GOAL + d1, d1);
							if( nWhiteGoal() == N_PIECE - 1 ) {
								--m_white[dst1];
								++m_white[src1];
								mvsList.clear();
								return mvsList << mvs;
							}
							mvsList << mvs;
						}
					}
					for(; src2 > IX_GOAL + 1; --src2) {
						if( src2 == dst1 )	//	同じ石を続けて移動する場合
							if( !black(reverseIX(dst1)) && !black(reverseIX(src1 - d1)) ) {
								removed = true;
								continue;	//	ダイス使用順に依らない場合
							}
						if( white(src2) == 0 ) continue;
						int dst2 = src2 - d1;
						if( dst2 > IX_GOAL && black(reverseIX(dst2)) <= 1 ) {
							mvs[1] = Move(src2, d1);
							mvsList << mvs;
							twoMovedAdded = true;
						}
					}
					--m_white[dst1];
					++m_white[src1];
					if( !twoMovedAdded && !removed && mvsList.size() == sz ) {
						mvs.pop_back();
						mvsList << mvs;
						mvs << Move(0, 0);
						oneMovedAdded = true;
					}
				}
			}
			if( twoMovedAdded && oneMovedAdded ) {
				for(int i = 0; i < mvsList.size();) {
					Move mv = mvsList[i][0];	//	for Debug
					if( mvsList[i].size() == 1 )
						mvsList.removeAt(i);
					else
						++i;
				}
			}
		}
	} else {	//	ゾロ目の場合、同じ目を4回使用する
		int nDice = 4;		//	残りの目の数
		Moves mvs;
		if( nWhiteBar() != 0 ) {	//	バーに石がある場合
			if( black(IX_GOAL + d1) > 1 )	//	移動不可
				return mvsList;
			Move mv(IX_START, d1);
			if( nWhiteBar() >= 4 ) {
				mvs << mv << mv << mv << mv;
				return mvsList << mvs;
			}
			nDice -= nWhiteBar();
			for(int i = 0; i < nWhiteBar(); ++i)
				mvs << mv;
			m_white[IX_START - d1] += nWhiteBar();
		}
		int nUsedDice = mvs.size();		//	使ったダイスの目の数
		//	残りのダイスの目の数（nDice）だけ、手を生成
		//if( canWhiteBearOff() ) {
		//} else {
			const bool cbf1 = canWhiteBearOff();
			int tailIndex1 = 0;
			const int sz = mvsList.size();
			for(int src1 = IX_START - 1; src1 > IX_GOAL; --src1) {
				if( !white(src1) ) continue;
				if( !tailIndex1 ) tailIndex1 = src1;
				int dst1 = src1 - d1;
				//if( dst1 <= IX_GOAL || black(reverseIX(dst1)) > 1 ) continue;
				if( !(dst1 > IX_GOAL && black(reverseIX(dst1)) <= 1
						|| cbf1 && (src1 == tailIndex1 && d1 >= src1 || src1 == d1)) )
					continue;
				mvs << Move(src1, d1);
				if( nDice == 1 ) {
					mvsList << mvs;
					nUsedDice = 4;
					mvs.pop_back();
					continue;
				}
				--m_white[src1];
				++m_white[dst1 = qMax(dst1, (int)IX_GOAL)];
				const bool cbf2 = canWhiteBearOff();
				int tailIndex2 = !m_white[tailIndex1] ? 0 : tailIndex1;
				const int sz = mvsList.size();
				for(int src2 = src1; src2 > IX_GOAL; --src2) {
					if( !white(src2) ) continue;
					if( !tailIndex2 ) tailIndex2 = src2;
					int dst2 = src2 - d1;
					if( !(dst2 > IX_GOAL && black(reverseIX(dst2)) <= 1
							|| cbf2 && (src2 == tailIndex2 && d1 >= src2 || src2 == d1)) )
						continue;
					mvs << Move(src2, d1);
					if( nDice == 2 ) {
						mvsList << mvs;
						nUsedDice = 4;
						mvs.pop_back();
						continue;
					}
					--m_white[src2];
					++m_white[dst2 = qMax(dst2, (int)IX_GOAL)];
					const bool cbf3 = canWhiteBearOff();
					int tailIndex3 = !m_white[tailIndex2] ? 0 : tailIndex2;
					const int sz = mvsList.size();
					for(int src3 = src2; src3 > IX_GOAL; --src3) {
						if( !white(src3) ) continue;
						if( !tailIndex3 ) tailIndex3 = src3;
						int dst3 = src3 - d1;
						if( !(dst3 > IX_GOAL && black(reverseIX(dst3)) <= 1
								|| cbf3 && (src3 == tailIndex3 && d1 >= src3 || src3 == d1)) )
							continue;
						mvs << Move(src3, d1);
						if( nDice == 3 ) {
							mvsList << mvs;
							nUsedDice = 4;
							mvs.pop_back();
							continue;
						}
						--m_white[src3];
						++m_white[dst3 = qMax(dst3, (int)IX_GOAL)];
						const bool cbf4 = canWhiteBearOff();
						int tailIndex4 = !m_white[tailIndex3] ? 0 : tailIndex3;
						const int sz = mvsList.size();
						for(int src4 = src3; src4 > IX_GOAL; --src4) {
							if( !white(src4) ) continue;
							if( !tailIndex4 ) tailIndex4 = src4;
							int dst4 = src4 - d1;
							if( !(dst4 > IX_GOAL && black(reverseIX(dst4)) <= 1
									|| cbf4 && (src4 == tailIndex4 && d1 >= src4 || src4 == d1)) )
								continue;
							mvs << Move(src4, d1);
							mvsList << mvs;
							nUsedDice = 4;
							mvs.pop_back();
						}
						if( mvsList.size() == sz ) {
							mvsList << mvs;
							nUsedDice = qMax(nUsedDice, mvs.size());
						}
						--m_white[dst3];
						++m_white[src3];
						mvs.pop_back();
					}
					if( mvsList.size() == sz ) {
						mvsList << mvs;
						nUsedDice = qMax(nUsedDice, mvs.size());
					}
					--m_white[dst2];
					++m_white[src2];
					mvs.pop_back();
				}
				if( mvsList.size() == sz ) {
					mvsList << mvs;
					nUsedDice = qMax(nUsedDice, mvs.size());
				}
				--m_white[dst1];
				++m_white[src1];
				mvs.pop_back();
			}
			if( !mvs.isEmpty() && mvsList.size() == sz ) {
				mvsList << mvs;
				nUsedDice = qMax(nUsedDice, mvs.size());
			}
		//}
		m_white[IX_START - d1] -= nWhiteBar();
		for(int i = 0; i < mvsList.size(); ) {
			if( mvsList[i].size() < nUsedDice )
				mvsList.removeAt(i);
			else
				++i;
		}
	}
	return mvsList;
}
QList<Moves> Board::whiteMovesList2(int d1, int d2) const
{
	Q_ASSERT( !doesWhiteWin() );
	QList<Moves> mvsList;
	Moves mvs;
	Move mv;
	Board bd(*this);
	if( d1 != d2 ) {	//	ゾロ目でない場合
	} else {	//	ゾロ目の場合
		int n = 4;	//	残り目の数
		if( bd.white(IX_START) != 0 ) {	//	バーに石がある場合
			while( n != 0 && bd.white(IX_START) != 0 ) {
				if( bd.black(IX_GOAL + d1) > 1 )	//	移動不可
					break;
				mvs << (mv = Move(IX_START, d1));
				bd.moveWhite(mv);
				--n;
			}
			if( bd.white(IX_START) != 0 || !n ) {
				if( !mvs.isEmpty() )
					mvsList << mvs;
				return mvsList;
			}
		}
		//	残りn個のダイス目について処理
		for(int src = IX_START - 1; src > IX_GOAL; --src) {
			if( !white(src) ) continue;
			//	白石を見つけた場合は、それを移動する場合と移動しない場合の両方を処理
		}
	}
	return mvsList;
}
void Board::moveBlack(int src, int dst)
{
	swapBW();
	moveWhite(src, dst);
	swapBW();
}
void Board::moveWhite(int src, int dst)
{
	Q_ASSERT( src <= IX_START );
	Q_ASSERT( src > IX_GOAL );
	Q_ASSERT( src > dst );
	Q_ASSERT( white(src) > 0 );
#if 1
	decWhite(src);
	Q_ASSERT( white(src) >= 0 );
	if( dst > IX_GOAL ) {
		if( black(reverseIX(dst)) != 0 ) {
			Q_ASSERT( black(reverseIX(dst)) == 1 );
			clearBlack(reverseIX(dst));
			incBlack(IX_START);
			//m_pipsCount += IX_GOAL - dst;
		}
		incWhite(dst);
		//m_pipsCount -= pipsCount(dst);
	} else {
		incWhite(IX_GOAL);
	}
#else
	--m_white[src];
	Q_ASSERT( white(src) >= 0 );
	//m_pipsCount += pipsCount(src);
	if( dst > IX_GOAL ) {
		if( m_black[reverseIX(dst)] != 0 ) {
			m_black[reverseIX(dst)] = 0;
			++m_black[IX_START];
			//m_pipsCount += IX_GOAL - dst;
		}
		++m_white[dst];
		//m_pipsCount -= pipsCount(dst);
	} else {
		++m_white[IX_GOAL];
	}
#endif
}
//
int eval5Sub(cchar *wa, cchar *ba)
{
	//	全ての石について、1～3で移動できるかどうかを調べる
	int ev = 0;
	int n[] = {0, 0, 0, 0, 0, 0, 0, 0, 0};
	int tailIX = 0;		//	最後尾インデックス
	if( wa[IX_START] ) {	//	バーに石がある場合
		for(int d = 1; d <= N_DICE; ++d) {
			switch( ba[IX_GOAL + d] ) {
			case 1:
				ev += (IX_GOAL + d) * 100 / 3;	//	ヒット可能な場合
				//	下にスルー
			case 0:
				n[d] += wa[IX_START];
				if( wa[IX_START] == 1 ) {
					for(int d2 = 1; d2 <= N_DICE; ++d2) {
						if( ba[IX_GOAL + d + d2] <= 1 )
							++n[d + d2];
					}
				}
			}
		}
	} else {
		for(int src = IX_START - 1; src > IX_GOAL; --src ) {
			if( !wa[src] ) continue;
			if( !tailIX ) tailIX = src;
			for(int d = 1; d <= N_DICE; ++d) {
				int dst = src - d;
				if( dst > IX_GOAL ) {
					switch( ba[reverseIX(dst)] ) {
					case 1:
						ev += reverseIX(dst) * 100 / 3;	//	ヒット可能な場合
						//	下にスルー
					case 0:
						n[d] += wa[src];
						for(int d2 = 1; d2 <= N_DICE; ++d2) {
							int dst2 = dst - d2;
							if( dst2 > IX_GOAL && ba[reverseIX(dst2)] <= 1 )
								n[d + d2] += wa[src];
						}
					}
				} else if( tailIX <= IX_GOAL + N_DICE ) {		//	ベアオフ可能の場合
					if( dst == IX_GOAL || src == tailIX )
						n[d] += wa[src];
				}
			}
		}
	}
#if	N_DICE == 3
	if( !n[1] ) {
		if( !n[2] ) {
			if( !n[3] )
				ev -= 533;	//	移動出来ない場合
			else
				ev -= (4+3*2+8)*100/9;	//	1,2 で移動出来ない場合
		} else {
			if( !n[3] )
				ev -= (4+4*2+12)*100/9;	//	1, 3 で移動出来ない場合
			else
				ev -= (8)*100/9;	//	2 で移動出来ない場合
		}
	} else {		//	１で移動可能
		if( !n[2] ) {
            if( !n[3] )
				ev -= (8+5*2+12)*100/9;	//	2, 3 で移動出来ない場合
            else
				ev -= (4)*100/9;	//	1 で移動出来ない場合
		} else {
			if( !n[3] )
				ev -= (12)*100/9;	//	3 で移動出来ない場合
		}
	}
#else	//	 N_DICE == 4
	//	サイの目が 1～4 の場合、平均は (3+4+5+5+6+7 + 2 + 4 + 6 + 8)*2/16. = 6.25
	//
	//        ┌─┬─┬─┬─┐
	//        │ 4│ 3│ 4│ 5│	//	sum = 16
	//        ├─┼─┼─┼─┤
	//        │ 3│ 8│ 5│ 6│	//	sum = 22
	//        ├─┼─┼─┼─┤
	//        │ 4│ 5│12│ 7│	//	sum = 28
	//        ├─┼─┼─┼─┤
	//        │ 5│ 6│ 7│16│	//	sum = 34
	//        └─┴─┴─┴─┘
	//
	if( !n[1] ) ev -= 1600/9;
	if( !n[2] ) ev -= 2200/9;
	if( !n[3] ) ev -= 2800/9;
	if( !n[4] ) ev -= 3400/9;
#if 0
	if( !n[1] ) {
		if( !n[2] ) {
			if( !n[3] ) {
				if( !n[4] )
					ev -= 625;	//	まったく移動出来ない場合
				else
					ev -= (16+22+28)*100/9;	//	1, 2, 3 で移動出来ない
			} else {
				if( !n[4] )
					ev -= (16+22+34)*199/9;	//	1, 2, 4 で移動出来ない
				else
					ev -= (16+22)*100/9;	//	1, 2 で移動出来ない
			}
		} else {	//	2 で移動可能
			if( !n[3] ) {
				if( !n[4] )
					ev -= (16+28+34)*199/9;	//	1, 3, 4 で移動出来ない
				else
					ev -= (16+28)*100/9;	//	1, 3 で移動出来ない
			} else {	//	3 で移動可能
				if( !n[4] )
					ev -= (16+34)*199/9;	//	1, 4 で移動出来ない
				else
					ev -= (16)*100/9;	//	1 で移動出来ない
			}
		}
	} else {		//	１で移動可能
		if( !n[2] ) {
            if( !n[3] ) {
				if( !n[4] )
					ev -= (22+28+34)*100/9;	//	2, 3, 4 で移動出来ない
				else
					ev -= (22+28)*100/9;	//	2, 3 で移動出来ない
            } else {
				if( !n[4] )
					ev -= (22+34)*199/9;	//	2, 4 で移動出来ない
				else
					ev -= (22)*100/9;	//	2 で移動出来ない
            }
		} else {	//	2 で移動可能
			if( !n[3] ) {
				if( !n[4] )
					ev -= (28+34)*199/9;	//	3, 4 で移動出来ない
				else
					ev -= (28)*100/9;	//	3 で移動出来ない
			} else {	//	3 で移動可能
				if( !n[4] )
					ev -= (34)*199/9;	//	4 で移動出来ない
			}
		}
	}
#endif
#endif
	return ev;
}
//	ピップス数をメインにした評価関数
//	1pips = 100点として計算
int Board::eval5() const
{
	//	pips数計算
	//	たぶん予め計算しておき、moveWhite() で補正した方が高速なので、
	//	遠くない将来に処理時間を比較して、高速ならそうする
	int bp = 0, wp = 0;
#if PACKED
	char ba[14], wa[14];	//	14 から 0 方向に移動
	uint64 b = black();
	uint64 w = white();
	for(int i = 0; i < 14; ++i, b>>=4, w>>=4) {
		bp += (ba[i] = b & 0x0f) * i;
		wp += (wa[i] = w & 0x0f) * i;
	}
#else
	for(int i = IX_GOAL + 1; i <= IX_START; ++i) {
		bp += m_black[i] * i;
		wp += m_white[i] * i;
	}
#endif
#if 0
	return (bp - wp) * 100 + 400;		//	400 for 手番の有利さ
#else
#if PACKED
	int s1 = eval5Sub(wa, ba);
	int s2 = eval5Sub(ba, wa);
#else
	int s1 = eval5Sub(m_white, m_black);
	int s2 = eval5Sub(m_black, m_white);
#endif
	return (s1 - s2) + (bp - wp) * 100 + 400;		//	400 for 手番の有利さ
#endif
}
int eval5(const Board &bd)
{
	int bp = 0, wp = 0;
	for(int i = IX_GOAL + 1; i <= IX_START; ++i) {
		bp += bd.black(i) * i;
		wp += bd.white(i) * i;
	}
	int s1 = eval5Sub(bd.whitePtr(), bd.blackPtr());
	int s2 = eval5Sub(bd.blackPtr(), bd.whitePtr());
	return (s1 - s2) + (bp - wp) * 100 + 400;		//	400 for 手番の有利さ
}
int eval6Sub(cchar *wa, cchar *ba)
{
	//	サイの目が 1～4 の場合、平均は (3+4+5+5+6+7 + 2 + 4 + 6 + 8)*2/16. = 6.25
	//
	//        ┌─┬─┬─┬─┐
	//        │ 4│ 3│ 4│ 5│	//	sum = 16
	//        ├─┼─┼─┼─┤
	//        │ 3│ 8│ 5│ 6│	//	sum = 22
	//        ├─┼─┼─┼─┤
	//        │ 4│ 5│12│ 7│	//	sum = 28
	//        ├─┼─┼─┼─┤
	//        │ 5│ 6│ 7│16│	//	sum = 34
	//        └─┴─┴─┴─┘
	//
	//	全ての石について、1～3で移動できるかどうかを調べる
	int ev = 0;
	int n[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
	int tailIX = 0;		//	最後尾インデックス
	if( wa[IX_START] ) {	//	バーに石がある場合
		for(int d = 1; d <= N_DICE; ++d) {
			switch( ba[IX_GOAL + d] ) {
			case 1:
				ev += (IX_GOAL + d) * 100 / 3;	//	ヒット可能な場合
				//	下にスルー
			case 0:
				n[d] += wa[IX_START];	//	バーに１個だけ
				if( wa[IX_START] == 1 ) {
					for(int d2 = 1; d2 <= N_DICE; ++d2) {
						if( ba[IX_GOAL + d + d2] <= 1 ) {
							++n[d + d2];
							if( d == d2 && ba[IX_GOAL + d * 3] <= 1 ) {
								++n[d*3];
								if( ba[IX_GOAL + d * 4] <= 1 )
									++n[d*4];
							}
						}
					}
				}
			}
		}
		if( !n[1] ) ev -= 1600/10;
		if( !n[2] ) ev -= 2200/10;
		if( !n[3] ) ev -= 2800/10;
		if( !n[4] ) ev -= 3400/10;
		if( !n[5] ) ev -= 500/10;
		if( !n[6] ) ev -= 600/10;
		if( !n[7] ) ev -= 700/10;
		if( !n[8] ) ev -= 800/10;
		if( !n[12] ) ev -= 1200/10;
		if( !n[16] ) ev -= 1600/10;
		if( !n[1] && !n[2] && !n[3] && !n[4] )
			ev -= 625*4;
	} else {
		for(int src = IX_START - 1; src > IX_GOAL; --src ) {
			if( !wa[src] ) continue;
			if( !tailIX ) tailIX = src;
			for(int d = 1; d <= N_DICE; ++d) {
				int dst = src - d;
				if( dst > IX_GOAL ) {
					switch( ba[reverseIX(dst)] ) {
					case 1:
						ev += reverseIX(dst) * 100 / 3;	//	ヒット可能な場合
						//	下にスルー
					case 0:
						n[d] += wa[src];
						for(int d2 = 1; d2 <= N_DICE; ++d2) {
							int dst2 = dst - d2;
							if( dst2 > IX_GOAL && ba[reverseIX(dst2)] <= 1 ) {
								n[d + d2] += wa[src];
								if( d == d2 ) {
									int dst3 = dst2 - d;
									if( dst3 > IX_GOAL && ba[reverseIX(dst3)] <= 1 ) {
										n[d*3] += wa[src];
										int dst4 = dst3 + d;
										if( dst4 > IX_GOAL && ba[reverseIX(dst4)] <= 1 )
											n[d*4] += wa[src];
									}
								}
							}
						}
					}
				} else if( tailIX <= IX_GOAL + N_DICE ) {		//	ベアオフ可能の場合
					if( dst == IX_GOAL || src == tailIX )
						n[d] += wa[src];
				}
			}
		}
		if( !n[1] ) ev -= 1600/9;
		if( !n[2] ) ev -= 2200/9;
		if( !n[3] ) ev -= 2800/9;
		if( !n[4] ) ev -= 3400/9;
		if( !n[5] ) ev -= 500/10;
		if( !n[6] ) ev -= 600/10;
		if( !n[7] ) ev -= 700/10;
		if( !n[8] ) ev -= 800/10;
		if( !n[12] ) ev -= 1200/10;
		if( !n[16] ) ev -= 1600/10;
	}
	return ev;
}
int eval6(const Board &bd)
{
	int bp = 0, wp = 0;
	for(int i = IX_GOAL + 1; i <= IX_START; ++i) {
		bp += bd.black(i) * i;
		wp += bd.white(i) * i;
	}
	int s1 = eval6Sub(bd.whitePtr(), bd.blackPtr());
	int s2 = eval6Sub(bd.blackPtr(), bd.whitePtr());
	return (s1 - s2) + (bp - wp) * 100 + 400;		//	400 for 手番の有利さ
}
//	ピップス数の最大差は 17*10 = 170 なので、（eval5 は 1pips = 100）
//	通常の勝ち；+5000, ギャモン勝ち：+10000、バックギャモン勝ち：+15000 としておく
#define		NORMAL_WIN			5000
#define		GAMMONED_WIN		10000
#define		BACKGAMMON_WIN		15000

int	g_terminalNodeCount;	//	末端ノード数
int	terminalNodeCount() { return g_terminalNodeCount; }

//	単純な 加重平均＋ミニマックス評価
double negaChanceNodeEvalX(const Board &bd, int);
double negaMaxEvalX(const Board &bd0, int rdepth, int d1, int d2)
{
	Q_ASSERT( !bd0.doesWhiteWin() );
	Q_ASSERT( !bd0.doesBlackWin() );
	double maxEval = -10000;
	QList<Moves> mvsList = bd0.whiteMovesList3(d1, d2);
	//QList<Moves> mvsList = bd0.whiteMovesList(d1, d2);
	///mvsList = bd0.whiteMovesList(d1, d2);		//	for パフォーマンス計測
	if( mvsList.isEmpty() ) {
		Board bd(bd0);
		bd.swapBW();
		return - negaChanceNodeEvalX(bd, rdepth);
	}
	foreach(Moves mvs, mvsList) {
		Board bd(bd0);
		foreach(Move mv, mvs)
			bd.moveWhite(mv);
		if( bd.doesWhiteWin() ) {
			++g_terminalNodeCount;
			const int ix = bd.blackTailIndex();
			if( ix >= IX_START - N_DICE /*bd.doesBlackBackGammonLose()*/ )
				return BACKGAMMON_WIN;
			if( bd.doesBlackGammonLose() )
				return GAMMONED_WIN;
			return NORMAL_WIN;
		}
		bd.swapBW();
		double ev = - negaChanceNodeEvalX(bd, rdepth);
		maxEval = qMax(maxEval, ev);
	}
	return maxEval;
}
double negaChanceNodeEvalX(const Board &bd, int rdepth)
{
	if( !--rdepth ) {
		++g_terminalNodeCount;
		return g_eval_func(bd);
		//return eval5(bd);
	}
	double t = 0;
	t += negaMaxEvalX(bd, rdepth, 1, 1);
	t += negaMaxEvalX(bd, rdepth, 2, 2);
	t += negaMaxEvalX(bd, rdepth, 3, 3);
	t += negaMaxEvalX(bd, rdepth, 4, 4);
	t += negaMaxEvalX(bd, rdepth, 1, 2) * 2;
	t += negaMaxEvalX(bd, rdepth, 1, 3) * 2;
	t += negaMaxEvalX(bd, rdepth, 1, 4) * 2;
	t += negaMaxEvalX(bd, rdepth, 2, 3) * 2;
	t += negaMaxEvalX(bd, rdepth, 2, 4) * 2;
	t += negaMaxEvalX(bd, rdepth, 3, 4) * 2;
	return t / 9;
}
Moves Board::whiteMovesEval5PlyD(int depth, int d1, int d2, double &eval) const
{
	return whiteMovesEvalXPlyD(depth, d1, d2, eval, ::eval5);
}
Moves Board::whiteMovesEval6PlyD(int depth, int d1, int d2, double &eval) const
{
	return whiteMovesEvalXPlyD(depth, d1, d2, eval, ::eval6);
}
Moves Board::whiteMovesEvalXPlyD(int depth, int d1, int d2, double &eval, EVAL_FUNC evalFunc) const
{
	Q_ASSERT( !doesWhiteWin() );
	Q_ASSERT( !doesBlackWin() );
	g_terminalNodeCount = 0;
	g_eval_func = evalFunc;
	//g_eval_func = ::eval5;
	QList<Moves> mvsList = whiteMovesList3(d1, d2);
	//QList<Moves> mvsList = whiteMovesList(d1, d2);
	///mvsList = whiteMovesList(d1, d2);		//	for パフォーマンス計測
	Moves bestMvs;
	if( mvsList.isEmpty() ) return bestMvs;
	if( mvsList.size() == 1 ) return mvsList[0];
	double ev, maxEval = -20000;
	foreach( Moves mvs, mvsList ) {
		Board bd(*this);
		foreach(Move mv, mvs) {
			bd.moveWhite(mv);
		}
		if( bd.doesWhiteWin() ) {
			bestMvs = mvs;
			const int ix = bd.blackTailIndex();
			if( ix >= IX_START - N_DICE /*bd.doesBlackBackGammonLose()*/ )
				maxEval = BACKGAMMON_WIN;
			else if( bd.doesBlackGammonLose() )
				maxEval = GAMMONED_WIN;
			else
				maxEval = NORMAL_WIN;
			break;
		} else {
			bd.swapBW();
			ev = -negaChanceNodeEvalX(bd, depth);
		}
		if( ev > maxEval ) {
			maxEval = ev;
			bestMvs = mvs;
		}
	}
	//qDebug() << "maxEval " << maxEval;
	eval = maxEval;
	return bestMvs;
}
//	ランダムに着手を選ぶ
Moves Board::whiteMovesRandom(int d1, int d2) const
{
	QList<Moves> mvsList = whiteMovesList(d1, d2);
	Moves bestMvs;
	if( mvsList.isEmpty() ) return bestMvs;
	if( mvsList.size() == 1 ) return mvsList[0];
	return mvsList[qrand() % mvsList.size()];
}
Moves Board::whiteMoves(int level, int d1, int d2, double &eval) const
{
	eval = 0;
	Moves mvs;
	switch( level ) {
	case LEVEL_RANDOM:
		mvs = whiteMovesRandom(d1, d2);
		break;
	case LEVEL_BEGINNER:
		mvs = !(qrand() & 1) ? whiteMovesEval6PlyD(1, d1, d2, eval)
				: whiteMovesRandom(d1, d2);
		break;
	case LEVEL_INTERMEDIATE:
		mvs = whiteMovesEval6PlyD(1, d1, d2, eval);
		break;
	case LEVEL_ADVANCED:
		mvs = whiteMovesEval6PlyD(3, d1, d2, eval);
		break;
	}
	return mvs;
}
Moves Board::blackMoves(int level, int d1, int d2, double &eval) const
{
	Board bd(*this);
	bd.swapBW();
	return bd.whiteMoves(level, d1, d2, eval);
}
