/*$Id$*/
package nicobrowser;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public final class Config {

    /**
     * feedurl.txtから取得した情報を格納する.
     */
    public static class NicoFeed {

        private final String url;
        private final int number;

        public NicoFeed(String url, int number) {
            this.url = url;
            this.number = number;
        }

        /** @return 上位いくつまでダウンロード対象とするか. */
        public int getNumber() {
            return number;
        }

        /** @return フィードのURL. */
        public String getUrl() {
            return url;
        }

        @Override
        public String toString() {
            return getUrl() + ", " + getNumber();
        }
    }
    private static Log log = LogFactory.getLog(Config.class);
    private static Config instance;
    private final Properties properties;
    private static final String APPLICATION_NAME = "nicobrowser";
    private static final String CONFIG_NAME = APPLICATION_NAME + ".properties";
    private static final String FEEDURL_NAME = "feedurl.txt";
    private static final File APP_HOME = new File(System.getProperty("user.home", "."), "." + APPLICATION_NAME);
    private static final File CONFIG_FILE = new File(APP_HOME, CONFIG_NAME);
    private static final File FEEDURL_FILE = new File(APP_HOME, FEEDURL_NAME);
    private static final String P_PATH_DB = "path.db";
    private static final String P_PATH_SAVEFILE = "path.savefile";
    private static final String P_FILE_ENCODING = "encoding";
    private static final String P_NICOVIDEO_MAIL = "nicovideo.mail";
    private static final String P_NICOVIDEO_PASSWORD = "nicovideo.password";
    private static final String P_DOWNLOAD_RETRY = "download.retry";
    private static final String P_DOWNLOAD_MYLIST = "download.mylist";

    /**
     * プログラム実行に必要なコンフィグファイルを作成する.
     * @return 今回コンフィグを作成したのであればtrue. 既に存在していたため, ファイル作成を行わなかった場合にはfalse.
     * @throws java.io.IOException ファイル作成に失敗した.
     */
    public static boolean createNewConfigFiles() throws IOException {
        boolean result = false;
        if (!CONFIG_FILE.exists()) {
            createNewConfigFile(CONFIG_FILE);
            result = true;
            log.info("コンフィグファイルを作成しました: " + CONFIG_FILE.getCanonicalPath());
        }
        if (!FEEDURL_FILE.exists()) {
            InputStream resource = null;
            try {
                resource = ClassLoader.getSystemResourceAsStream("resources/" + FEEDURL_NAME);
                createNewFeedFile(resource, FEEDURL_FILE);
                result = true;
                log.info("FEED URLファイルを作成しました: " + FEEDURL_FILE.getCanonicalPath());
            } finally {
                if (resource != null) {
                    resource.close();
                }
            }
        }
        return result;
    }

    private static void createNewConfigFile(File file) throws IOException {
        ArrayList<CharSequence> props = new ArrayList<CharSequence>();

        StringBuilder dbpath = new StringBuilder(P_PATH_DB + "=");
        File dbDir = new File(APP_HOME, "db");
        dbDir.mkdirs();
        //Windowsのパス区切りバックスペースをエスケープするための処理も入れている.
        dbpath.append(dbDir.getAbsolutePath().replaceAll("\\\\", "\\\\\\\\"));
        props.add(dbpath);

        StringBuilder savepath = new StringBuilder(P_PATH_SAVEFILE + "=");
        File saveDir = new File(APP_HOME, "flv");
        saveDir.mkdirs();
        savepath.append(saveDir.getAbsolutePath().replaceAll("\\\\", "\\\\\\\\"));
        props.add(savepath);

        props.add(P_FILE_ENCODING + "=" + System.getProperty("file.encoding"));

        props.add(P_NICOVIDEO_MAIL + "=set.your@mail.address");
        props.add(P_NICOVIDEO_PASSWORD + "=set_your_password");
        props.add(P_DOWNLOAD_RETRY + "=3");
        props.add(P_DOWNLOAD_MYLIST + "=");

        FileUtils.writeLines(file, props);
    }

    private static void createNewFeedFile(InputStream resource, File dest) throws IOException {
        List<String> list = new ArrayList<String>();
        BufferedReader br = new BufferedReader(new InputStreamReader(resource, "UTF-8"));
        while (true) {
            String text = br.readLine();
            if (text == null) {
                break;
            }
            list.add(text);
        }
        FileUtils.writeLines(dest, list);
    }

    private Config() {
        properties = new Properties();
        FileInputStream fis = null;
        try {
            fis = new FileInputStream(CONFIG_FILE);
            properties.load(fis);
        } catch (IOException ex) {
            log.fatal("コンフィグの読み込みに失敗: " + CONFIG_FILE);
            throw new RuntimeException(ex);
        } finally {
            if (fis != null) {
                try {
                    fis.close();
                } catch (IOException ex) {
                    log.warn("", ex);
                }
            }
        }

        try {
            List urls = FileUtils.readLines(FEEDURL_FILE);
        } catch (IOException ex) {
            log.fatal("コンフィグの読み込みに失敗: " + CONFIG_FILE);
            throw new RuntimeException(ex);
        }
    }

    public static Config getInstance() {
        if (instance == null) {
            instance = new Config();
        }
        return instance;
    }

    /**
     * @return ニコニコ動画ログインID.
     */
    public String getNicoMail() {
        return properties.getProperty(P_NICOVIDEO_MAIL);
    }

    /**
     * @return ニコニコ動画ログインパスワード.
     */
    public String getNicoPassword() {
        return properties.getProperty(P_NICOVIDEO_PASSWORD);
    }

    /** @return DBパス */
    public String getDbFile() {
        return new File(properties.getProperty(P_PATH_DB), "nicodb").getAbsolutePath();
    }

    /** @return 保存先の指定. */
    public String getSrcSaveDir() {
        return properties.getProperty(P_PATH_SAVEFILE);
    }

    /** @return feedurl.txtの文字エンコーディング. */
    public String getEncoding() {
        String res = properties.getProperty(P_FILE_ENCODING, System.getProperty("file.encoding"));
        return res;
    }
//    /**
//     * video/mp4のストリーム保存時、ファイルの拡張子に
//     * .mp4を使用する(true)か、使用しない(.flv)か.
//     * @return mp4拡張子を使用する場合true.
//     */
//    public boolean isExtMp4Use() {
//        String res = properties.getProperty("download.ext.mp4");
//        if ("false".equalsIgnoreCase(res)) {
//            return false;
//        }
//        return true;
//    }
//

    /**
     * 失敗したダウンロードファイルの最大リトライ回数を取得する.
     * @return リトライ回数.
     */
    public int getMaxRetry() {
        String res = properties.getProperty(P_DOWNLOAD_RETRY);
        return Integer.parseInt(res);
    }
//
//    /**
//     * 上位何位までの動画をダウンロードするか.
//     * @return ダウンロードするファイルの最下位.
//     */
//    public int getMaxDownloadNumber() {
//        String res = properties.getProperty("download.number");
//        return Integer.parseInt(res);
//    }
//

    public List<String> getDownLoadMyList() {
        List<String> list = new ArrayList<String>();
        String res = properties.getProperty(P_DOWNLOAD_MYLIST);
        String[] nums = res.split(",");
        for (int i = 0; i < nums.length; i++) {
            String text = nums[i].trim();
            if (!text.isEmpty()) {
                list.add(text);
            }
        }
        return list;
    }

    public List<NicoFeed> getNicoFeeds() {
        List<NicoFeed> list = new ArrayList<NicoFeed>();
        try {
            List lines = FileUtils.readLines(FEEDURL_FILE, getEncoding());
            for (Object line : lines) {
                final String str = line.toString();
                if (str.isEmpty() || str.startsWith("#")) {
                    // 空行, コメント行はスキップ.
                    continue;
                }

                String[] values = str.split(",");
                NicoFeed feed = new NicoFeed(values[0].trim(), Integer.parseInt(values[1].trim()));
                list.add(feed);
            }
        } catch (IOException ex) {
            log.error("ファイルが見つかりません: " + FEEDURL_FILE);
        }
        return list;
    }
}
