/*
 * arizona-spi.c  --  Arizona SPI bus interface
 *
 * Copyright 2012 Wolfson Microelectronics plc
 *
 * Author: Mark Brown <broonie@opensource.wolfsonmicro.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/err.h>
#include <linux/module.h>
#include <linux/pm_runtime.h>
#include <linux/regmap.h>
#include <linux/regulator/consumer.h>
#include <linux/slab.h>
#include <linux/spi/spi.h>
#include <linux/of.h>

#include <linux/mfd/arizona/core.h>

#include "arizona.h"

#ifdef CONFIG_ACPI
#include <linux/acpi.h>

const struct acpi_gpio_params reset_gpios = {1, 0, false};
const struct acpi_gpio_params ldoena_gpios = {2, 0, false};

const struct acpi_gpio_mapping arizona_acpi_gpios[] = {
	{ "reset-gpios", &reset_gpios, 1,},
	{ "ldoena_gpios", &ldoena_gpios, 1},
	{ },
};
#endif

static int arizona_spi_probe(struct spi_device *spi)
{
	const struct spi_device_id *id;
	struct arizona *arizona;
	const struct regmap_config *regmap_config = NULL;
	unsigned long type;
	int ret;
#ifdef CONFIG_ACPI
	struct acpi_device *adev;
	acpi_handle handle;
#endif

	if (spi->dev.of_node)
		type = arizona_of_get_type(&spi->dev);
	else {
		if (( type = arizona_acpi_get_type(&spi->dev))) {
			if (ACPI_COMPANION(&spi->dev))
				spi->irq = acpi_dev_gpio_irq_get(ACPI_COMPANION(&spi->dev), 0);
		} else {
			id = spi_get_device_id(spi);
			if (!id || !id->driver_data)
				return -ENODEV;

			type = id->driver_data;
			dev_err(&spi->dev, "matched spi\n");
		}
	}

	switch (type) {
	case WM5102:
		if (IS_ENABLED(CONFIG_MFD_WM5102))
			regmap_config = &wm5102_spi_regmap;
		break;
	case WM5110:
	case WM8280:
		if (IS_ENABLED(CONFIG_MFD_WM5110))
			regmap_config = &wm5110_spi_regmap;
		break;
	case WM1831:
	case CS47L24:
		if (IS_ENABLED(CONFIG_MFD_CS47L24))
			regmap_config = &cs47l24_spi_regmap;
		break;
	default:
		dev_err(&spi->dev, "Unknown device type %ld\n", type);
		return -EINVAL;
	}

	if (!regmap_config) {
		dev_err(&spi->dev,
			"No kernel support for device type %ld\n", type);
		return -EINVAL;
	}

	arizona = devm_kzalloc(&spi->dev, sizeof(*arizona), GFP_KERNEL);
	if (arizona == NULL)
		return -ENOMEM;

	arizona->regmap = devm_regmap_init_spi(spi, regmap_config);
	if (IS_ERR(arizona->regmap)) {
		ret = PTR_ERR(arizona->regmap);
		dev_err(&spi->dev, "Failed to allocate register map: %d\n",
			ret);
		return ret;
	}
#ifdef CONFIG_ACPI
	handle = ACPI_HANDLE(&spi->dev);

	if (!handle || acpi_bus_get_device(handle,&adev)) {
		dev_err(&spi->dev, "unable to get ACPI handle\n");
		return -ENODEV;
	}

	ret = acpi_dev_add_driver_gpios(ACPI_COMPANION(&spi->dev), arizona_acpi_gpios);
#endif
	arizona->type = type;
	arizona->dev = &spi->dev;
	arizona->irq = spi->irq;

	return arizona_dev_init(arizona);
}

static int arizona_spi_remove(struct spi_device *spi)
{
	struct arizona *arizona = spi_get_drvdata(spi);

	arizona_dev_exit(arizona);

	return 0;
}

static const struct spi_device_id arizona_spi_ids[] = {
	{ "wm5102", WM5102 },
	{ "wm5110", WM5110 },
	{ "wm8280", WM8280 },
	{ "wm1831", WM1831 },
	{ "cs47l24", CS47L24 },
	{ },
};
MODULE_DEVICE_TABLE(spi, arizona_spi_ids);

static const struct acpi_device_id arizona_acpi_match[] = {
	{
		.id = "WM510205",
		.driver_data = WM5102,
	},
	{
		.id = "WM510204",
		.driver_data = WM5102,
	},
	{ },
};
MODULE_DEVICE_TABLE(acpi, arizona_acpi_match);

static struct spi_driver arizona_spi_driver = {
	.driver = {
		.name	= "arizona",
		.pm	= &arizona_pm_ops,
		.of_match_table	= of_match_ptr(arizona_of_match),
		.acpi_match_table = ACPI_PTR(arizona_acpi_match),
	},
	.probe		= arizona_spi_probe,
	.remove		= arizona_spi_remove,
	.id_table	= arizona_spi_ids,
};

module_spi_driver(arizona_spi_driver);

MODULE_DESCRIPTION("Arizona SPI bus interface");
MODULE_AUTHOR("Mark Brown <broonie@opensource.wolfsonmicro.com>");
MODULE_LICENSE("GPL");
