/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.bench;

import org.awk4j.bench.target.Trim;

import java.util.function.Function;

import static org.awk4j.bench.Bench.*;

/**
 * Benchmark Framework 2 - Functional programming.
 *
 * @author kunio himei.
 */
public class F02Function {

    // ラッパー（Wrapper）
    static final Function<String, String> rTrimRegex01Classic = Trim::rTrimRegex01Classic;
    static final Function<String, String> rTrimRegex02New = Trim::rTrimRegex02New;
    static final Function<String, String> trim03Original = String::trim;
    static final Function<String, String> trim04String = Trim::trim04String;
    static final Function<CharSequence, String> trim05CharSeq = Trim::trim05CharSeq;
    static final Function<String, String> rTrim06String = Trim::rTrim06String;
    static final Function<StringBuilder, StringBuilder> rTrim07Builder = Trim::rTrim07Builder;

    //////////////////////////////////////////////////////////////////////
    // ドライバ - driver (T: String と CharSequence を受け付ける)
    static <T> int driver00Generics(Function<T, String> fnc, String ans) {
        long nano = Bench.driver_loop_time;
        int count = 0;
        String rs;
        do {
            long start = System.nanoTime();
            //noinspection unchecked
            rs = fnc.apply((T) BENCH_DATA);
            nano -= System.nanoTime() - start;
            if (isDriverComplete(++count)) break; // Driver is Complete
        } while (0 < nano);
        assert ans.equals(rs) : rs;
        return count;
    }

    // driver - String
    static int driver01String(Function<String, String> fnc, String ans) {
        long nano = Bench.driver_loop_time;
        int count = 0;
        String rs;
        do {
            long start = System.nanoTime();
            rs = fnc.apply(BENCH_DATA);
            nano -= System.nanoTime() - start;
            if (isDriverComplete(++count)) break; // Driver is Complete
        } while (0 < nano);
        assert ans.equals(rs) : rs;
        return count;
    }

    // driver - CharSequence
    @SuppressWarnings("SameParameterValue")
    static int driver02CharSequence(Function<CharSequence, String> fnc, String ans) {
        long nano = Bench.driver_loop_time;
        int count = 0;
        String rs;
        do {
            long start = System.nanoTime();
            rs = fnc.apply(BENCH_DATA);
            nano -= System.nanoTime() - start;
            if (isDriverComplete(++count)) break; // Driver is Complete
        } while (0 < nano);
        assert ans.equals(rs) : rs;
        return count;
    }

    // ドライバ - driver
    @SuppressWarnings("SameParameterValue")
    static int driver03Builder(Function<StringBuilder, StringBuilder> fnc, String ans) {
        long nano = Bench.driver_loop_time;
        int count = 0;
        do {
            sb.setLength(0);
            sb.append(BENCH_DATA);
            long start = System.nanoTime();
            fnc.apply(sb);
            nano -= System.nanoTime() - start;
            if (isDriverComplete(++count)) break; // Driver is Complete
        } while (0 < nano);
        assert ans.contentEquals(sb) : sb;
        return count;
    }

    private static final StringBuilder sb = new StringBuilder(BENCH_DATA.length());

    //////////////////////////////////////////////////////////////////////
    // クルー - crew
    @SuppressWarnings("SameParameterValue")
    static void crew(int[] ANSWER) {
        ANSWER[1] = driver01String(rTrimRegex01Classic, ANS_RIGHT);
        Bench.cool();
        ANSWER[2] = driver01String(rTrimRegex02New, ANS_RIGHT);
        Bench.cool();
        ANSWER[3] = driver01String(trim03Original, ANS_BOTH);
        Bench.cool();
        ANSWER[4] = driver01String(trim04String, ANS_BOTH);
        Bench.cool();
        ANSWER[5] = driver02CharSequence(trim05CharSeq, ANS_BOTH);
        Bench.cool();
        ANSWER[6] = driver01String(rTrim06String, ANS_RIGHT);
        Bench.cool();
        ANSWER[7] = driver03Builder(rTrim07Builder, ANS_RIGHT);
    }
    // クルー - crew for Generics
    @SuppressWarnings("SameParameterValue")
    static void GenericsCrew(int[] ANSWER) {
        ANSWER[1] = driver00Generics(rTrimRegex01Classic, ANS_RIGHT);
        Bench.cool();
        ANSWER[2] = driver00Generics(rTrimRegex02New, ANS_RIGHT);
        Bench.cool();
        ANSWER[3] = driver00Generics(trim03Original, ANS_BOTH);
        Bench.cool();
        ANSWER[4] = driver00Generics(trim04String, ANS_BOTH);
        Bench.cool();
        ANSWER[5] = driver00Generics(trim05CharSeq, ANS_BOTH);
        Bench.cool();
        ANSWER[6] = driver00Generics(rTrim06String, ANS_RIGHT);
        Bench.cool();
        ANSWER[7] = driver03Builder(rTrim07Builder, ANS_RIGHT);
    }
}
/*
    // Warmup Target - ウオームアップ ターゲット.
    //* F01Classic#warmupTarget を使用する.
    static void warmupTarget() {
        rTrimRegex01Classic.apply(BENCH_DATA); // #1
        rTrimRegex02New.apply(BENCH_DATA); // #2
        trim03Original.apply(BENCH_DATA); // #3 Original
        trim04String.apply(BENCH_DATA); // #4
        trim05CharSeq.apply(BENCH_DATA); // #5
        rTrim06String.apply(BENCH_DATA); // #6
        sb.setLength(0);
        sb.append(BENCH_DATA);
        rTrim07Builder.apply(sb); // #7
    }
*/