/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.bench;

import java.util.Date;

/**
 * Benchmark Var.2 - 単位時間に実行可能な回数を求める.
 * Find the number of times that can be executed in a unit time.
 * <p>
 * ・テストケース - test case
 * ・ベンチマーク - benchmark
 * ・クルー - crew
 * ・ドライバ - driver
 * ・ターゲット - target
 *
 * @author kunio himei.
 */
class BenchV2 {

    @SuppressWarnings("unused")
    private static final String PATTERN_01 = "^^);"; // LATIN1
    @SuppressWarnings("unused")
    private static final String PATTERN_02 = "※"; // UTF-16
    private static final String pattern = PATTERN_01;
    private static final String SPACE_256 = " ".repeat(256);
    private static final String test_data = " " + pattern + SPACE_256;
    private static final String ans_right = " " + pattern;
    private static final String ans_both = pattern;
    private static final long NANO = 1000_000; // 1ms = 10^6 ns.

    private static long loop_time; // Main.mainTime or Main.loop (ns)
    private static int cooling_time; // Main.cool or Main.ice (ms)

    private static long active_time; // active time (ms)
    private static long warmup_time; // warmup time (ms)
    private static long prefetch_time; // Prefetch time (ms)
    private static long driver_time; // Driver time (ms)

    private static boolean isPrint = false; // True, Production mode
    private static boolean isDriverWarmup; // Driver Compile mode

    /*
     * Warm up - ウオームアップ.
     */
    private static void warmup() {
        long start = System.currentTimeMillis();
        if (Main.prefetch > 0) {
            active_time = System.currentTimeMillis();
            var sb = new StringBuilder(test_data.length() + 7);
            for (int i = 0; i < Main.prefetch; i++) { // First group
                // 最重要ターゲットを最初にコンパイルする。
                // Compile the most important targets first.
                sb.setLength(0);
                sb.append(test_data);
                Trim.rTrim07Builder(sb); // #7 Target
                Trim.trim04String(test_data); // #4
                Trim.trim05CharSeq(test_data); // #5
                Trim.rTrim06String(test_data); // #6
            }
            ice();
            for (int i = 0; i < Main.prefetch; i++) { // 2nd group
                Trim.rTrimRegex01Classic(test_data); // #1
                Trim.rTrimRegex02New(test_data); // #2
                test_data.trim(); // #3 Original
            }
            prefetch_time = System.currentTimeMillis() - active_time;
            ice();
            // コンパイルによりターゲット呼び出し及び時刻取得の誤差が軽減.
            // Compiling reduces errors when calling the target and getting the time.
            isDriverWarmup = true;
            active_time = System.currentTimeMillis();
            benchV2(Main.mainTime * NANO);
            driver_time = System.currentTimeMillis() - active_time;
            isDriverWarmup = false;
        }
        warmup_time = System.currentTimeMillis() - start;
    }

    // クールダウン - cool down
    private static void cool() {
        sleep(cooling_time);
    }

    // 強制冷却 - Forced cooling
    private static void ice() {
        sleep(Main.ice);
    }

    private static void sleep(int millis) {
        try {
            if (millis > 0) {
                active_time += millis;
                Thread.sleep(millis);
            }
        } catch (Throwable e) {
            throw new IllegalStateException(e);
        }
    }

    //////////////////////////////////////////////////////////////////////
    // ベンチマーク - benchmark
    static void bench() {
        long start = System.currentTimeMillis();
        if (Main.warmup) {
//            cooling_time = Main.ice; // NOTE Forced cooling
            cooling_time = Main.cool; // cool down
            warmup();
        }
        cooling_time = Main.cool; // cool down
        isPrint = true;

        for (int i = 0; i < 12; i++) {
            ice();
            benchV2(Main.mainTime * NANO);
            System.err.println();
        }
        double elapsed = (System.currentTimeMillis() - start) / 1000.;

        System.err.print("\tbenchTrim: " + elapsed + " sec. (" + Main.title + ")\n" +
                "\t-priority: " + Main.priority + " (" + new Date() + ")\n" +
                "\t-mainTime: " + Main.mainTime + " ms per driver (Adjust the benchmark time)\n" +
                "\t-warmup: " + Main.warmup + ", " + (warmup_time / 1000.) + " sec.\n" +
                "\t  -prefetch Target & Driver: " + Main.prefetch + " times\n" +
                "\t  (Target: " + prefetch_time + " ms, Driver: " +
                driver_time + " ms, active time)\n" +
                "\t-cool down: " + Main.cool + " ms per driver\n" +
                "\t-ice: " + Main.ice + " ms per benchmark\n");
    }

    //////////////////////////////////////////////////////////////////////
    // クルー - crew
    @SuppressWarnings("SameParameterValue")
    private static void benchV2(long time) {
        loop_time = time;
        bench1Regex();
        cool();
        bench2Regex();
        cool();
        bench3TrimOriginal();
        cool();
        bench4TrimString();
        cool();
        bench5TrimCharSeq();
        cool();
        bench6rTrimString();
        cool();
        bench7rTrimBuilder();
    }

    private static void bench1Regex() {
        long nano = loop_time;
        int count = 0;
        String rs;
        do {
            long start = System.nanoTime();
            rs = Trim.rTrimRegex01Classic(test_data);
            nano -= System.nanoTime() - start;
            count++;
            if (isDriverWarmup && count > Main.prefetch) break;
        } while (0 < nano);
        assert ans_right.equals(rs) : rs;
        if (isPrint)
            System.err.println("\t1. regex.Classic\t" + count);
    }

    private static void bench2Regex() {
        long nano = loop_time;
        int count = 0;
        String rs;
        do {
            long start = System.nanoTime();
            rs = Trim.rTrimRegex02New(test_data);
            nano -= System.nanoTime() - start;
            count++;
            if (isDriverWarmup && count > Main.prefetch) break;
        } while (0 < nano);
        assert ans_right.equals(rs) : rs;
        if (isPrint)
            System.err.println("\t2. regex.New\t" + count);
    }

    private static void bench3TrimOriginal() {
        long nano = loop_time;
        int count = 0;
        String rs;
        do {
            long start = System.nanoTime();
            rs = test_data.trim();
            nano -= System.nanoTime() - start;
            count++;
            if (isDriverWarmup && count > Main.prefetch) break;
        } while (0 < nano);
        assert ans_both.equals(rs) : rs;
        if (isPrint)
            System.err.println("\t3. Trim.Original\t" + count);
    }

    private static void bench4TrimString() {
        long nano = loop_time;
        int count = 0;
        String rs;
        do {
            long start = System.nanoTime();
            rs = Trim.trim04String(test_data);
            nano -= System.nanoTime() - start;
            count++;
            if (isDriverWarmup && count > Main.prefetch) break;
        } while (0 < nano);
        assert ans_both.equals(rs) : rs;
        if (isPrint)
            System.err.println("\t4. Trim.String\t" + count);
    }

    private static void bench5TrimCharSeq() {
        long nano = loop_time;
        int count = 0;
        String rs;
        do {
            long start = System.nanoTime();
            rs = Trim.trim05CharSeq(test_data);
            nano -= System.nanoTime() - start;
            count++;
            if (isDriverWarmup && count > Main.prefetch) break;
        } while (0 < nano);
        assert ans_both.equals(rs) : rs;
        if (isPrint)
            System.err.println("\t5. Trim.CharSeq\t" + count);
    }

    private static void bench6rTrimString() {
        long nano = loop_time;
        int count = 0;
        String rs;
        do {
            long start = System.nanoTime();
            rs = Trim.rTrim06String(test_data);
            nano -= System.nanoTime() - start;
            count++;
            if (isDriverWarmup && count > Main.prefetch) break;
        } while (0 < nano);
        assert ans_right.equals(rs) : rs;
        if (isPrint)
            System.err.println("\t6. rTrim.String\t" + count);
    }

    private static void bench7rTrimBuilder() {
        var sb = new StringBuilder(test_data.length());
        long nano = loop_time;
        int count = 0;
        do {
            sb.setLength(0);
            sb.append(test_data);
            long start = System.nanoTime();
            Trim.rTrim07Builder(sb);
            nano -= System.nanoTime() - start;
            count++;
            if (isDriverWarmup && count > Main.prefetch) break;
        } while (0 < nano);
        assert ans_right.contentEquals(sb) : sb;
        if (isPrint)
            System.err.println("\t7. rTrim.Builder\t" + count);
    }
}