/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.space;

import org.jetbrains.annotations.NotNull;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * cutSpace - Tools.
 *
 * @author kunio himei.
 */
class Tools {

    static int RSTART; // Match start
    static int RLENGTH; // Match length

    /**
     * match(r,s) - AWK specifications.
     *
     * @param r     regexp
     * @param s     input
     * @param group number 0, 1.. m.groupCount()
     * @throws IndexOutOfBoundsException 指定されたインデックスを持つグループがパターンに含まれない場合
     */
    static boolean match(@NotNull Pattern r, CharSequence s, int group) {
        Matcher m = r.matcher(s);
        if (m.find()) {
            RSTART = m.start(group);
            RLENGTH = m.end(group) - RSTART;
            return true;
        }
        return false;
    }

    /**
     * Trim.
     * ベンチマーク結果に基づく高速アルゴリズムを採用しました.
     * while の複合条件を if 2つに分けても性能差は、ありません.<br>
     * We adopted a high-speed algorithm based on the benchmark results.
     * There is no difference in performance even if the compound condition
     * of while is divided into two ifs.
     * <p>
     * <pre><code>
     *  　while (0 <= --i && ' ' >= s.charAt(i))
     *      ; // do nothing
     *      ↓
     *  　while (0 <= --i)
     *      if (' ' < s.charAt(i)) break;　</code></pre>
     * <p>
     * <pre>Benchmark result of Trim.
     * 1. regex.Classic	4.3  /[\s]+$/
     * 2. regex.New	    3.0  /[\0- ]+$/
     * 3. Trim().Org    1.2  Original String#trim()
     * 4. Trim().New    1.1  Customize String#trim()
     * 5. Str.Old	    1.1
     * 6. Str.New	    1.1
     * 7. Build.Old	    1    - compound if
     * 8. Build.New	    1    - Simple if </pre>
     * <p>
     * Trim (String).
     * ポイントは、空白判定です.（The point is a blank judgment）
     * オリジナルは、一見シンプルですがループ中で２度演算しスマートで無いため書き換えました.
     * The original is seemingly simple, but I rewrote it because
     * it is not smart because it is calculated twice in the loop.
     *
     * @see String#trim()
     */
    @SuppressWarnings("StatementWithEmptyBody")
    @NotNull
    static String trim(@NotNull String s) {
        int end, length = end = s.length();
        int st = 0;
        while ((st < end) && (' ' >= s.charAt(st))) {
            st++;
        }
//        while ((st < end) && (' ' >= s.charAt(end-1))) {
//            end--;
//        }
        while ((st < end--) && (' ' >= s.charAt(end))) {
            // do nothing
        }
        return ((++end < length) || (st > 0)) ?
                s.substring(st, end) : s;
    }

    /**
     * Left trim. (String)
     */
    @SuppressWarnings({"unused"})
    @NotNull
    static String lTrim(@NotNull String s) {
        int i = -1, len = s.length();
        while (++i < len)
            if (' ' < s.charAt(i)) break;
        return i > 0 ? s.substring(i) : s;
    }

    /**
     * Right trim. (String)
     */
    @NotNull
    static String rTrim(@NotNull String s) {
        int len, i = len = s.length();
        while (0 <= --i)
            if (' ' < s.charAt(i)) break;
        return ++i < len ? s.substring(0, i) : s;
    }

    /**
     * Right trim. (StringBuilder)
     * - High performance because no string generation is required.
     * - point. 文字列の生成が不要なため高性能.
     */
    @NotNull
    static StringBuilder rTrim(@NotNull StringBuilder sb) {
        int len, i = len = sb.length();
        while (0 <= --i)
            if (' ' < sb.charAt(i)) break;
        if (++i < len) sb.setLength(i);
        return sb;
    }

    /**
     * Right trim twins. (Source, Template)
     */
    static void rTrimTwins(@NotNull StringBuilder sb, @NotNull StringBuilder sx) {
        int len = rTrim(sx).length(); // This is the first for DEL measures.
        sb.setLength(len);
    }

    /**
     * charAt - Gets the character of the specified index.
     *
     * @param cs Input, does not accept empty.
     * @throws AssertionError - IndexOutOfBoundsException, Ref. Template#parse ※
     */
    static char charAt(@NotNull CharSequence cs, int ix) {
        int len = cs.length();
        assert 0 <= ix && ix < len :
                String.format("IndexOutOfBoundsException Index: %d/%d [%s]",
                        ix, len, cs);
        return toWhiteSpace(cs.charAt(ix));
    }

    /**
     * charAtLast - Get the character of the last index.
     *
     * @param cs Input, does not accept empty.
     * @throws AssertionError - EmptyStringException
     */
    static char charAtLast(@NotNull CharSequence cs) {
        int ix = cs.length() - 1;
        assert 0 <= ix : "EmptyStringException";
        return toWhiteSpace(cs.charAt(ix));
    }

    /**
     * Backward match. (char)
     *
     * @param cs Input, Allows an empty string.
     */
    static boolean endsWith(@NotNull CharSequence cs, char c) {
        if (!cs.isEmpty())
            return toWhiteSpace(c) == charAtLast(cs); // The input is not empty
        return false;
    }

    /*
     * @return a Blank if White Space.
     */
    private static final String WHITE_SPACES = " \t\n\r" + Template.RS;

    static char toWhiteSpace(char c) {
        return 0 <= WHITE_SPACES.indexOf(c) ? ' ' : c;
    }
}