/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.space;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.*;
import java.nio.charset.StandardCharsets;

/**
 * cutSpace - html obfuscation
 *
 * @author kunio himei.
 */
public class Main {

    private static final String USAGE =
            """
                    $(CutSpace.jar) [-options] <input> 1><output> 2><debug>
                    -x(deprecated)
                    -e(expand) [Script, Css, Html]
                    -d(debug level) -1, 0...9
                    -v Version 1.0.2""";

    private static final StringBuilder sb = new StringBuilder(1024); // concat buffer
    private static BufferedReader reader;
    private static String in = "";
    static int lineNumber;

    @Nullable
    static String getLine() throws IOException {
        if (null != in) {
            in = reader.readLine();
            if (null != in) {
                lineNumber += 1;
                return Tools.rTrim(in);
            }
        }
        return null;
    }

    /*
     * Main.
     * - '\nl' String continuation.
     */
    public static void main(String @NotNull [] args) throws IOException {
        int ix = 0;
        for (; ix < args.length; ix++) {
            String arg = args[ix];
            if ('-' != arg.charAt(0)) break;
            if (arg.contains("-v")) {
                System.err.println(USAGE);
                return;

            } else if (arg.startsWith("-x")) { // -x(deprecated)
                Parser.deprecated = true;

            } else if (arg.startsWith("-e")) { // -e(expand)
                Parser.expand = arg.substring(2).toUpperCase();

            } else if (arg.startsWith("-d")) { // -d(debug)
                char c = arg.charAt(arg.length() - 1);
                Parser.debug = Character.digit(c, 10);
            }
        }
        InputStream inputStream =
                ix >= args.length ? System.in :
                        new FileInputStream(args[ix]);
        reader = new BufferedReader(
                new InputStreamReader(inputStream,
                        StandardCharsets.UTF_8), 1024 * 8);
        try {
            while (true) {
                String input = getLine();
                if (null == input) break;
                if (Tools.endsWith(input, '\\')) { // <\nl> continuation (cat)
                    do {
                        // NOTE Continuation line - Simply join (単純に結合する)
                        sb.append(input, 0, input.length() - 1);
                        input = getLine();
                    } while (null != input && Tools.endsWith(input, '\\'));
                    if (null != input) sb.append(input);
                    input = sb.toString();
                    sb.setLength(0);
                }
                Parser.execute(input);
            }
        } finally {
            Modules.flush(Parser.out);
            System.out.close();
            Modules.flush(Parser.err);
            System.err.close();
            reader.close();
        }
    }
}