/**
 * Copyright (C) 2023 awk4j - https://ja.osdn.net/projects/awk4j/
 * <p>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * <p>
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * <p>
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
use std::thread;
use std::thread::JoinHandle;
use std::time::Duration;

use crate::chmod;
use crate::files;
use crate::files::DD;
use crate::iomod;
use crate::thmod;

// https://doc.rust-jp.rs/book-ja/ch16-03-shared-state.html
static mut HANDLES: Vec<JoinHandle<()>> = Vec::new();

// デーモンスレド
pub fn spawnx(thread: i32) {
    for _ in 0..thread {
        let handle: JoinHandle<()> = thread::spawn(|| loop {
            let _pop = thmod::get();
            match _pop {
                Some(x) => task(x),
                None => {
                    if thmod::is_done() {
                        break;
                    } else {
                        sleep(); // 起動直後でリクエスト未到着など
                    }
                }
            };
        });
        unsafe {
            HANDLES.push(handle);
            // println!("spawn: {}", HANDLES.len());
        }
    }
}

// スレッドの完了を待ち合わせる (called from main)
pub fn joinall() {
    unsafe {
        while HANDLES.len() > 0 {
            // println!("await: {}", HANDLES.len());
            let handle: JoinHandle<()> = HANDLES.pop().unwrap();
            let _: () = handle.join().unwrap();
        }
    }
}

const IDLING_TIME: u64 = 100; // ms.

// https://runebook.dev/ja/docs/rust/std/thread/fn.sleep
// この機能はブロック化されており、asyncの機能では使用しないでください
fn sleep() {
    if !thmod::interrupted() {
        // println!("IDLING");
        thread::sleep(Duration::from_millis(IDLING_TIME));
    }
}

fn task(dd: DD) {
    thmod::progress(&dd.input); // プログレスを表示
    if dd.action != files::SKIP {
        // スキップ以外ならアクションを実行
        if dd.cmr_mode == files::_RENAME {
            iomod::rename_file(&dd.input, &dd.output); // Rename
        } else if dd.algorithm == files::_STD {
            iomod::copy(&dd.input, &dd.output); // Copy, Move
        } else if dd.algorithm == files::_MAXBUF {
            chmod::copymax(&dd.input, &dd.output); // maxbuf
        } else {
            chmod::copych(&dd.input, &dd.output); // channel
        }
    }
    if dd.cmr_mode != files::_COPY {
        // 入力ファイルを削除
        let _ = iomod::remove_file(&dd.input); // Move, Rename
    }
    thmod::decrement_req_number(); // 処理要求(キュー数)を減算
}
