/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.spawn;

import plus.io.Device;
import plus.io.Io;
import plus.io.IoHelper;
import plus.spawn.system.UtilInterface;
import plus.util.Escape;
import plus.util.NumHelper;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * [%command%] implementation of the `sort` command.
 * <p>
 * Usage: sort [OPTION]... [FILE]... <br>
 * 文字列をソートする<br>
 * <br>
 * -b 各行の比較の際に、行頭の空白を無視する <br>
 * -d アルファベット、数字、空白以外を無視する <br>
 * -f ソートの際に、小文字を対応する大文字と同じに扱う。例えば `b' は `B' と同じにソートされる <br>
 * -g 数値として比較する <br>
 * -n 先頭の文字列 (空白が前置されていても良い) を数値文字列として比較する <br>
 * -r 比較の結果を逆順にする。より大きなキー値を持つ行が、より早く現われるようになる <br>
 * -tSEP 各行からソートキーを検索する際、文字 SEP
 * をフィールドのセパレーターにする。デフォルトでは、フィールドは空白以外の文字と空白文字の間の空文字列 (empty string) によって分離される <br>
 * -u 等しいとされた行のうちの最初のものだけを表示する <br>
 * +POS1[-POS2] 各行でソートキーとして用いるフィールドを指定する。 POS1 から POS2 の直前の部分 (POS2
 * が与えられなかった場合は行末まで) がフィールドとなる。フィールドと文字位置は 0 から始まる <br>
 * -POS2 end it at POS2(default end of line) <br>
 * -kPOS1[,POS2] ソートキーを指定する別法。フィールドと文字位置は 1 から始まる <br>
 * <br>
 * 位置の指定は f.c という書式で行う。 f は用いるフィールドの番号、 c は先頭文字の番号で 0 から始ま る部分文字列を指定する。 <br>
 * +POS や -POS 引き数にはオプション文字 "bdfginr"
 * を付加することができる。この場合、そのフィールドにはグローバルな順序オプションは適用されなくなる。 -b オプションは +POS 及び -POS
 * の片方にも両方にも指定でき、グローバルオプションから継承された場合は両方に適用される。 -n または -M オプションが指定された場合には、 -b
 * が暗黙に指定されたとみなされ、 -b がキー指定の +POS と -POS の両方に適用される。キーは複数のフィールドにまたがっていても構わない。
 * </p>
 *
 * @author kunio himei.
 */
public final class Sort extends StringWriter implements UtilInterface {

    /**
     * [%enclosure%] アルファベット、数字、空白を識別する正規表現.
     */
    static final Pattern RX_ALNUM = Pattern.compile("[^a-zA-Z0-9 \t]");
    /**
     * [%enclosure%] 制御文字を識別する正規表現.
     */
    static final Pattern RX_CNTRL = Pattern.compile("\\p{Cntrl}");
    /**
     * [%enclosure%] 行頭空白を識別する正規表現.
     */
    static final Pattern RX_TOP_SPACE = Pattern.compile("^\\s+");
    /**
     * 入力ｽﾄﾘｰﾑのバッファサイズ.
     */
    private static final int DEFAULT_BYTE_BUFFER_SIZE = 1024 * 8;
    /**
     * index[1].
     */
    private static final int IDX1 = 1;
    /**
     * index[2].
     */
    private static final int IDX2 = 2;
    /**
     * USAGE.
     */
    private static final String USAGE = """
            Usage: sort [OPTION]... [FILE]...
            Write sorted concatenation of all FILE(s) to standard output.
              -b    ignore leading blanks
              -d    consider only blanks and alphanumeric characters
              -f    lower case to upper case characters
              -g    compare according to general numerical value
              -n    compare according to string numerical value
              -r    reverse the result of comparisons
              -tSEP use SEP(field separator) instead of non-blank to blank transition
              -u    unique
              -kPOS1[,POS2] start a key at POS1(origin 1),
                            end it at POS2(default end of line)
              +POS1[-POS2]  start a key at POS1, end it at POS2(default end of ine)
              -POS2         end it at POS2(default end of line)
                  --help      display this help and exit (should be alone)

            POS is F[.C][OPTS], where F is the field number and C the character position in the field; both are origin 0.
            If neither -t nor -b is in effect, characters in a field are counted from the beginning of the preceding whitespace.
            OPTS is one or more single-letter ordering arg, which override global ordering arg for that key. If no key is given, use the entire line as the key.
            With no FILE, or when FILE is -, read standard input.""";
    private static final Object LOCK = new Object(); // SYNC static.
    /**
     * [%enclosure%] ソートキー (POS1, POS2, flags) の配列.
     */
    final List<int[]> poskeys = new ArrayList<>();
    /**
     * Comparator.
     */
    private final Comparator<Object> comparator = new SortComparator();
    /**
     * field-separator=SEP use SEP instead of non-blank to blank transition.
     */
    private final String fs;
    /**
     * グローバルオプション.
     */
    private final int[] globals = new int[2];
    /**
     * ソートキーオプションが指定されている (フィールドへの分割が必要).
     */
    private final boolean hasKeyOptions;
    /**
     * 入力ファイル名配列.
     */
    private final ConcurrentLinkedQueue<String> inputFiles = new ConcurrentLinkedQueue<>();
    /**
     * このストリーム.
     */
    private final Writer out;
    /**
     * クラス構築時に一時的に使用する、 キー位置 (POS1, POS2, flags).
     */
    private int[] pos;

    /**
     * ☆ Array sort.
     */
    public Sort(String[] args) throws IOException {
        this(args, null);
    }

    /**
     * sort.
     */
    public Sort(String[] args, final Writer output)
            throws IOException {
        this.pos = new int[]{-1, 0, -1, 0, 0};
        String t = null; // field-separator
        String outfile = null; // output file
        for (String arg : args) {
            char c = arg.charAt(0);
            if (('-' == c)) {
                int len = arg.length();
                c = ((1 < len) ? arg.charAt(IDX1) : ' ');
                if (1 == len) {
                    this.inputFiles.add("-"); // 標準入力

                } else if (Character.isDigit(c)) { // -POS2
                    optPOS1(arg, IDX1, '-', this.globals[Symbol.G_OPTIONS]);

                } else if ('k' == c) { // -kPOS1[,POS2]
                    optPOS1(arg, IDX2, 'k', this.globals[Symbol.G_OPTIONS]);

                } else if ('t' == c) { // -tSEPARATOR
                    String sep = arg.substring(IDX2);
                    Matcher m = Symbol.RX_FS.matcher(sep);
                    if (m.find()) {
                        sep = ((null == m.group(1)) ? ((null == m.group(2)) ? m
                                .group(3) : m.group(2)) : m.group(1));
                    }
                    if (1 == sep.length()) { // 1文字
                        t = sep;
                    } else if (1 < sep.length()) { // '\t' など
                        t = Escape.outputFilter(sep);
                    }
                } else if ('o' == c) { // -o output
                    outfile = plus.spawn.Helper.filename(arg.substring(IDX2)); // output

                } else if (0 <= "Mbdfginruv".indexOf(c)) { // option flags
                    optMbdfginruv(arg, IDX1, this.globals);

                } else {
                    throw new IllegalArgumentException('`' + arg + '`');
                }
            } else if ('+' == c) { // +POS1[-POS2]
                optPOS1(arg, IDX1, '+', this.globals[Symbol.G_OPTIONS]);

            } else {
                this.inputFiles.add(arg); // 入力ファイル
            }
        }
        if ((0 < this.pos[Symbol.POS_1]) || this.poskeys.isEmpty()) {
            if (0 > this.pos[Symbol.POS_1]) {
                this.pos[Symbol.POS_1] = 0;
            }
            if (0 == this.pos[Symbol.POS_FLAGS]) {
                this.pos[Symbol.POS_FLAGS] = (Symbol.OFLAG_XMASK & this.globals[Symbol.G_OPTIONS]);
            }
            this.poskeys.add(this.pos);
        }
        if (1 == this.poskeys.size()) {
            this.pos = this.poskeys.get(0);
            boolean hasKey = false;
            for (int i : this.pos) {
                if (0 < i) {
                    hasKey = true;
                    break;
                }
            }
            this.hasKeyOptions = hasKey; // ソートキーオプションが指定されている
        } else {
            this.hasKeyOptions = true;
        }
        this.out = ((null == outfile) ? output : Device
                .openOutput(">", outfile));
        this.fs = ((null == t) ? "[ \\t]" : t); // field-separator;

        if ((0 != (Symbol.OFLAG_VG & this.globals[Symbol.G_OPTIONS]))) {
            StringBuilder sb = new StringBuilder();
            sb.append(plus.spawn.Helper.mkString("sort", args));
            if (0 != this.globals[Symbol.G_OPTIONS]) {
                sb.append(Integer.toHexString(this.globals[Symbol.G_OPTIONS]));
            }
            for (int[] p : this.poskeys) {
                sb.append('(').append(p[Symbol.POS_1]).append('.')
                        .append(p[Symbol.POS_1START]).append(',')
                        .append(p[Symbol.POS_2]).append('.')
                        .append(p[Symbol.POS_2END]).append(',')
                        .append(Integer.toHexString(p[Symbol.POS_FLAGS]))
                        .append(')');
            }
            if (!this.inputFiles.isEmpty()) {
                sb.append(this.inputFiles);
            }
            if (null != outfile) {
                sb.append(outfile);
            }
            System.err.println(sb.toString());
        }
    }

    /**
     *
     */
    public static void main(String[] args) throws IOException {
        if ((0 != args.length) && args[0].startsWith("--help")) {
            System.out.println(USAGE);
        } else {
            UtilInterface me = new Sort(args, Device.openOutput("",
                    Io.STDOUT));
            if (me.hasInput()) {
                IoHelper.copy(Io.STDIN, (Writer) me);
            }
            me.close();
        }
    }

    /**
     * 入力データをソートして返す.
     */
    public String[] apply(String[] source) {
        int srclen = source.length;
        Object[] arr = new Object[srclen];
        for (int i = 0; srclen > i; i++) {
            String[] flds;
            if (this.hasKeyOptions) { // ソートキーオプションが指定されている
                // 入力データをフィールド配列に分割
                flds = source[i].split(this.fs);
            } else {
                flds = new String[]{source[i]}; // 単一のフィールド
            } // ( Original Array position: Integer, Files: String[] )
            arr[i] = new Object[]{i, flds};
        }
        // System.err.println("SORT in: " + srclen);
        Arrays.sort(arr, this.comparator);
        // System.err.println("SORT out: " + arr.length);

        List<String> list = new ArrayList<>(srclen);
        String line = null;
        for (Object o : arr) {
            int idx = (Integer) ((Object[]) o)[0];
            String x = source[idx];
            if (0 == (Symbol.OFLAG_UG & this.globals[Symbol.G_OPTIONS])) {
                list.add(x);
            } else {
                if (!x.equals(line)) {
                    list.add(x);
                }
            }
            line = x;
        }
        return list.toArray(new String[0]);
    }

    /**
     * このストリームを閉じる.
     */
    @Override
    public void close() throws IOException {
        if (null == this.out) throw new NullPointerException("this.out");
        ArrayList<String> list;
        synchronized (LOCK) { // SYNC. close.
            String src = super.toString();
            if (src.isEmpty()) {
                list = new ArrayList<>();
            } else {
                String[] arr = plus.spawn.Helper.split(src); // 入力データを配列に分割
                list = new ArrayList<>(Arrays.asList(arr));
                super.getBuffer().setLength(0); // clear
            }
        }
        String x = this.inputFiles.poll(); // 入力ファイルを取得する
        while (null != x) {
            BufferedReader in = new BufferedReader(Device.openInput(x),
                    DEFAULT_BYTE_BUFFER_SIZE);
            String line = in.readLine();
            while (null != line) {
                list.add(line);
                line = in.readLine();
            }
            Io.close(in);
            x = this.inputFiles.poll();
        }
        String[] arr = apply(list.toArray(new String[0])); // sort
        list.clear(); // clear
        try {
            for (String line : arr) {
                IoHelper.writeln(this.out, line);
            }
            this.out.flush();
        } finally {
            try {
                Io.close(this.out);
            } catch (IOException e) { // ひき逃げ
                // POSTIT パイプは閉じられている場合がある ('U179qawk1ans')
                // System.err.println("closed: "+e.getMessage());
            }
        }
    }

    /**
     * サブプロセスの終了値を返す.
     */
    @Override
    public int exitValue() {
        return 0;
    }

    /**
     *
     */
    @Override
    public boolean hasInput() {
        return this.inputFiles.isEmpty();
    }

    /**
     * コマンドラインオプションを解析する .
     */
    private int optMbdfginruv(String arg, int index,
                              int[] flag) {
        int len = arg.length();
        int i = index;
        for (; len > i; i++) {
            char c = arg.charAt(i);
            if ('b' == c) {
                flag[0] |= Symbol.OFLAG_B;
            } else if ('d' == c) {
                flag[0] |= Symbol.OFLAG_D;
            } else if ('f' == c) {
                flag[0] |= Symbol.OFLAG_F;
            } else if ('g' == c) {
                flag[0] |= Symbol.OFLAG_G;
            } else if ('i' == c) {
                flag[0] |= Symbol.OFLAG_I;
            } else if ('n' == c) {
                flag[0] |= Symbol.OFLAG_N;
            } else if ('r' == c) {
                flag[0] |= Symbol.OFLAG_R;
            } else if ('u' == c) {
                this.globals[Symbol.G_OPTIONS] |= Symbol.OFLAG_UG;
            } else if ('v' == c) {
                this.globals[Symbol.G_OPTIONS] |= Symbol.OFLAG_VG;
            } else {
                break; // その他は読み飛ばし
            }
        } // System.err.println("(OPT" + flag[0] + ":" + arg.substring(i) + ")");
        return i;
    }

    /**
     * キー位置オプションを解析する.
     * <p>
     * <li>+POS1 [-POS2] 各行でソートキーとして用いるフィールドを指定する.<br>
     * POS1 から POS2 の直前の部分 (POS2 が与えられなかった場合は行末まで) がフィールドとなる.<br>
     * フィールドと文字位置は 0 から始まる.<br>
     * <li>-k POS1[,POS2] ソートキーを指定する別法.<br>
     * フィールドと文字位置は 1 から始まる.<br>
     */
    private void optPOS1(String arg, int index, char opt,
                         int flags) {
        // +POS1[-POS2] -kPOS1[,POS2] ペアオプションの出現の判定
        boolean is2ndParm = (('+' == this.globals[Symbol.G_OLD_POS_OPTION]) && ('-' == opt))
                || (('k' == this.globals[Symbol.G_OLD_POS_OPTION]) && (',' == opt));
        this.globals[Symbol.G_OLD_POS_OPTION] = opt; // 過去のキー位置指定オプションを退避

        if (!is2ndParm && (0 <= this.pos[Symbol.POS_1])) {
            this.poskeys.add(this.pos);
            this.pos = new int[]{
                    -1,
                    0,
                    -1,
                    0,
                    (0 == this.pos[Symbol.POS_1])
                            ? (Symbol.OFLAG_XMASK & flags)
                            : this.pos[Symbol.POS_1]};
        }
        int i = index;
        i = optPOS12(arg, i, (is2ndParm) ? Symbol.POS_2 : Symbol.POS_1, opt);

        if ((i < arg.length())
                && ((',' == arg.charAt(i)) || ('-' == arg.charAt(i)))) {
            i++;
            optPOS12(arg, i, Symbol.POS_2, opt);
        }
    }

    /**
     * キー位置オプションを解析する.
     */
    private int optPOS12(String arg, int index, int ix,
                         char opt) {
        int len = arg.length();
        int i = index;
        Matcher m = Symbol.RX_POS_NUMBER.matcher(arg.substring(i));
        int x = 0;
        if (m.find()) {
            x = Integer.parseInt(m.group());
            i += m.end();
        } // 'k' フィールドと文字位置は 1 から始まる
        this.pos[ix] = ((('k' == opt) && (0 < x)) ? (x - 1) : x);
        if ((i < len) && ('.' == arg.charAt(i)) && (++i < len)) {
            m = Symbol.RX_POS_NUMBER.matcher(arg.substring(i));
            x = 0;
            if (m.find()) {
                x = Integer.parseInt(m.group());
                i += m.end();
            }
            this.pos[ix + Symbol.POS_1START] = ((('k' == opt) && (0 < x))
                    ? (x - 1) : x);
        }
        int[] flag = new int[1];
        i = optMbdfginruv(arg, i, flag);
        if (0 != flag[0]) {
            this.pos[Symbol.POS_FLAGS] = (Symbol.OFLAG_XMASK & flag[0]);
        }
        return i;
    }

    /**
     * Symbols.
     */
    private static final class Symbol {

        /**
         * +POS1[-POS2] -kPOS1[,POS2] ペアオプションの出現の判定.
         */
        static final int G_OLD_POS_OPTION = 1;
        /**
         * global option flags.
         */
        static final int G_OPTIONS = 0;

        /**
         * ignore leading blanks.
         */
        static final int OFLAG_B = 1;
        /**
         * consider only blanks and alphanumeric characters.
         */
        static final int OFLAG_D = 2;
        /**
         * fold lower case to upper case characters.
         */
        static final int OFLAG_F = 4;
        /**
         * compare according to general numerical value.
         */
        static final int OFLAG_G = 8;
        /**
         * consider only printable characters.
         */
        static final int OFLAG_I = 16;
        /**
         * compare according to string numerical value.
         */
        static final int OFLAG_N = 32;
        /**
         * reverse the result of comparisons.
         */
        static final int OFLAG_R = 64;

        /**
         * unique (global).
         */
        static final int OFLAG_UG = 128;
        /**
         * verbose (global).
         */
        static final int OFLAG_VG = 256;

        /**
         * ローカルオプションを抽出するビットマスク.
         */
        static final int OFLAG_XMASK = OFLAG_UG - 1;

        /**
         * position 1.
         */
        static final int POS_1 = 0;
        /**
         * position 1 start.
         */
        static final int POS_1START = 1;
        /**
         * position 2.
         */
        static final int POS_2 = 2;
        /**
         * position 2 end.
         */
        static final int POS_2END = 3;
        /**
         * option flags.
         */
        static final int POS_FLAGS = 4;

        // "[^"\\]*(\\.[^"\\]*)+"
        /**
         * オプション -t フィールドセパレータ (field separator).
         */
        static final Pattern RX_FS = Pattern.compile("^([^'\"\\s]+)"
                + "|\"([^\"\\\\]*(?:\\\\.[^\"\\\\]*)*)\""
                + "|'([^'\\\\]*(?:\\\\.[^'\\\\]*)*)'");

        /**
         * オプション -n で数値文字列として判断する数値を表す正規表現 (decimal digit).
         */
        static final Pattern RX_IS_NUMBER = Pattern.compile("^[-+]?\\d");

        /**
         * オプション POS の数値を表す正規表現 (decimal digit).
         */
        static final Pattern RX_POS_NUMBER = Pattern.compile("^\\d+");
    }

    /**
     * Comparator.
     */
    class SortComparator implements Comparator<Object> {

        /**
         * 比較の結果を返す.
         * a negative integer, zero, or a positive integer as the first
         * argument is less than, equal to, or greater than the second.
         */
        @Override
        public int compare(Object a1, Object a2) {
            // 入力データを取得 ( Integer, String[] )
            String[] in1 = (String[]) ((Object[]) a1)[1];
            String[] in2 = (String[]) ((Object[]) a2)[1];
            int inlen1 = in1.length;
            int inlen2 = in2.length;

            for (int[] p : Sort.this.poskeys) {
                int pos1 = p[Symbol.POS_1];
                int pos2 = (0 <= p[Symbol.POS_2]) ? p[Symbol.POS_2]
                        : (Math.max(in1.length, in2.length) - 1);
                int start = p[Symbol.POS_1START];
                int end = p[Symbol.POS_2END];
                int flag = p[Symbol.POS_FLAGS];

                /* reverse the result of comparisons. */
                int reverse = (0 == (Symbol.OFLAG_R & flag)) ? 1 : -1; // 比較の結果を逆順にする
                // System.err.println("pos(" + pos1 + "," + pos2 + ")" + flag);

                boolean hasNext = true;
                for (int i = pos1; hasNext && (pos2 >= i); i++) {
                    // POS2 が指定されており、終了文字位置が無指定なら次の繰り返しで POS2 を評価しない
                    hasNext = ((0 > p[Symbol.POS_2]) || !(((pos2 - 1) <= i) && (0 >= end)));

                    String f1 = (inlen1 > i) ? in1[i] : "";
                    String f2 = (inlen2 > i) ? in2[i] : "";
                    f1 = mkKey(start, end, flag, f1);
                    f2 = mkKey(start, end, flag, f2);
                    int sgn;
                    if (0 != (Symbol.OFLAG_G & flag)) { // 数値比較
                        /* compare according to general numerical value. */
                        sgn = Double.compare(NumHelper.parseNumber(f1).doubleValue(), //
                                NumHelper.parseNumber(f2).doubleValue());

                    } else {
                        int i1 = (0 == (Symbol.OFLAG_F & flag)) ? f1
                                .compareTo(f2) : f1.compareToIgnoreCase(f2);
                        if (0 != (Symbol.OFLAG_N & flag)) { // 先頭の文字列を数値比較
                            /* compare according to string numerical value. */
                            if (Symbol.RX_IS_NUMBER.matcher(f1).find()
                                    && Symbol.RX_IS_NUMBER.matcher(f2).find()) {
                                int diff = f1.length() - f2.length();
                                sgn = ((0 == diff) ? f1.compareTo(f2) : diff);
                            } else {
                                /* fold lower case to upper case characters. */
                                sgn = i1;
                            }
                        } else {
                            /* fold lower case to upper case characters. */
                            sgn = i1;
                        }
                    }
                    if (0 != sgn) {
                        return sgn * reverse;
                    }
                }
            }
            return Integer.compare(inlen1, inlen2);
        }

        /**
         * フィールド文字列を返す.
         * フィールド文字列
         */
        private String mkKey(int start, int end, int flag,
                             String src) {
            /* ignore leading blanks. */
            String x = (0 == (Symbol.OFLAG_B & flag)) ? src : RX_TOP_SPACE
                    .matcher(src).replaceFirst(""); // 行頭の空白を無視する

            /*
             * フィールド文字列を切り出す.
             *
             * 位置の指定は f.c という書式で行う.<br>
             * f は用いるフィールドの番号、 c は先頭文字の番号で、 +POS の場合はフィールドの先頭から、-POS
             * の場合は直前のフィールドの最後から数える.<br>
             * .c の部分は省略でき、その場合はフィールドの先頭の文字となる.<br>
             * -b オプションが指定された場合の .c の起点は、 +POS ならフィールドに最初に現われた空白以外の文字となり、 -POS
             * なら直前のフィールド以降に最初に現われた空白以外の文字となる.<br>
             */
            if ((0 != start) || (0 != end)) {
                int len = x.length();
                int istart = (0 >= start) ? 0 : Math.min(start, len);
                int iend = (0 >= end) ? len : Math.min(end, len);
                x = ((istart >= iend) ? "" : x.substring(istart, iend));
            }

            /* consider only printable characters. */
            if (0 != (Symbol.OFLAG_I & flag)) { // 制御文字を無視する
                x = RX_CNTRL.matcher(x).replaceAll(""); // [\x00-\x1F\x7F]
            }

            /* consider only blanks and alphanumeric characters. */
            if (0 != (Symbol.OFLAG_D & flag)) { // アルファベット、数字、空白以外を無視する
                x = RX_ALNUM.matcher(x).replaceAll("");
            }
            return x;
        }
    }
}