/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.gen

import groovy.transform.CompileStatic
import plus.lex.Flags
import plus.lex.Keyword
import plus.lex.Node
import plus.lex.Symbols

/**
 * Code Generator - Function.
 *
 * @author kunio himei.
 */
@CompileStatic
abstract class GenFunc extends GenHelper {

    /** 文.
     */
    void statement(Object[] arr) {
        yyIndent += 1
        for (e in arr) {
            switch (e) {
                case Node.YyNop: // Annotation インライン注釈を食べる.
                    // Lex.simpleStmt.Annotationで削除済.
                    // System.err.println("eat: " + e)
                    break
                case Node.Stmt:
                    Node.Stmt x = e as Node.Stmt
                    gen(simpleStmt(x))
                    break
                case Node.FnI: // 関数定義行.
                    Node.FnI x = e as Node.FnI
                    Node.Func fn = functionMap.get(x.name)
                    // System.out.println(".statement: " + isClosure(fn) + " " + x.name)
                    if (isClosure(fn)) {
                        closureDecl(fn) // Groovy CLOSURE を生成.
                    } else {
                        // functionDecl(fn) // Function - 後で展開.
                    }
                    break
                default:
                    gen(eval(e))
            }
        }
        yyIndent -= 1
    }

    /** 単純な文.
     */
    String simpleStmt(Node.Stmt e) {
        String a = (isNil(e.e)) ? '' : eval(e.e)
        if (Keyword.SymBREAK == e.id) return 'break'
        else if (Keyword.SymCONTINUE == e.id) return ''
        else if (Keyword.SymEXIT == e.id) return 'throw new plus.exception.ExitException(' + a + ')'
        else if (Keyword.SymNEXT == e.id) return 'throw plus.exception.NextException.THIS'
        else if (Keyword.SymNEXTFILE == e.id) return 'nextfile(); throw plus.exception.NextException.THIS'
        else if (Keyword.SymRETURN == e.id) {
            String x = (isNil(a)) ? '' : parseNumber(a)
            return "return ${x}"
        }
        throw new IllegalStateException(e.class.getName() + " " + e)
    }

    /** 関数定義.
     */
    void functionDecl(Node.Func e) {
        for (x in e.comment) gen('//' + x) // 注釈を生成
        for (x in e.annotation) gen(x) // 関数 Annotation 生成
        currentFunc.push(e)
        List buf = []
        Object scope = Symbols.startLocal(e.name)
        for (int i = 0; e.argsLength > i; i++) { // 仮引数
            Node.FnP p = e.parm[i] // REMIND OutOfBounds!
            String x = (isNil(p.sType)) ? '' : p.sType
            if (Flags.isVarArgs(p.nType))  // 可変長引数
                x += '...'
            buf.add((x + ' ' + p.name).trim())
            Symbols.markLocal(p.name, p.nType)

            if (Flags.isVarArgs(p.nType)) // REMIND patch 可変長引数
                break
        }
        String args = mkString(buf.toArray(), '(', ', ', ')')
        String rtn = isNil(e.sType) ? 'def' : e.sType // 復帰タイプ
        if (isClosure(e)) {
            String callName = simpleFunctionName(e.name)
            gen("${rtn} ${callName}${args} {")
            gen("  Futures.await('${e.name}')") // Lambda式の解析完了を通知
        } else {
            gen("${rtn} ${e.name}${args} {")
        }
        if (0 < e.parm.length - e.argsLength) {
            yyIndent += 1 // ローカル変数の初期値を設定
            int parmlen = e.parm.length
            int j = e.argsLength
            while (parmlen > j) {
                String name = e.parm[j].name
                int nType = e.parm[j].nType
                String claz = newClass(nType)
                if (exists(claz)) {
                    gen("def ${name} = ${claz}")
                } else if (Flags.isAssign(nType)) { // ローカル変数への代入あり
                    gen(mkGlobalNil(name))
                } else {
                    gen("def ${name} = 0")
                }
                Symbols.markLocal(name, nType)
                j += 1
            }
            yyIndent -= 1
        }
        statement(e.stmt)
        Symbols.endLocal(scope)
        currentFunc.pop()
        gen('}')
    }

    /** CLOSURE 定義.
     * NOTE 注意！ レガシー仕様の、仮引数でのローカル変数定義は、仕様から削除する.
     */
    void closureDecl(Node.Func e) {
        for (x in e.comment) gen('//' + x) // 注釈を生成
        // CLOSURE に、Annotation は、付与できない.
        for (x in e.annotation) gen('// ' + x) // コメントを生成
        if (exists(e.annotation))
            System.err.println(
                    " 'Annotation' cannot be added to ${e.name}.")
        currentFunc.push(e)
        List buf = []
        Object scope = Symbols.startLocal(e.name)
        String name = simpleFunctionName(e.name) // Simple Name.
        for (int i = 0; e.argsLength > i; i++) { // 仮引数
            Node.FnP p = e.parm[i]
            String x = (isNil(p.sType)) ? '' : p.sType
            if (Flags.isVarArgs(p.nType))  // 可変長引数
                x += '...'
            buf.add((x + ' ' + p.name).trim())
            Symbols.markLocal(p.name, p.nType)
        }
        String args = mkStringNil(buf.toArray(), ' ', ', ', ' ->')
        gen("Closure ${name} = {${args}")
        statement(e.stmt)
        Symbols.endLocal(scope)
        currentFunc.pop()
        gen('}')
        excludeMap.put(e.name, -1) // 関数生成除外フラグを設定.
    }

    // ------------------------------------------------------------------------
    // Functions.
    // ------------------------------------------------------------------------
    /**
     * 関数呼出し.
     */
    Object callFunction(String name, Object[] args) {
        if (functionMap.containsKey(name))
            return userFunction(name, args, evalArgs(args))
        return builtinFunction(name, args, evalArgs(args))
    }

    /**
     * ユーザ関数呼出し - Calling Built-in Functions.
     */
    private String userFunction(String name, Object[] args, Object[] vals) {
        Node.Func fn = functionMap.get(name)
        Object[] carr = callArgs(fn, args, vals)
        String parm = mkString(carr, '', ', ', '')
        String callName = isClosure(fn) ?
                simpleFunctionName(name) : name
        return "${callName}(${parm})"
    }

    /**
     * 組み込み関数 - Calling Built-in Functions.
     */
    private String builtinFunction(String name, Object[] args, Object[] vals) {
        switch (name) {
            case Node.INDEX_OP:
                return mkIndex(args) // これは、argsで良い.
            case Node.CONCAT_OP:
                return '_cat' + mkString(vals, '(', ', ', ')')
            case 'gsub':
                // Fallthrough //
            case 'sub':
                return genSub(name, args, vals)
            case 'assertEquals':
                // Fallthrough //
            case 'assertTrue':
                return "${name}(${scriptLineNumber}" +
                        mkString(vals, ', ', ', ', ')')
        }
        return builtInFuncName(name) + mkString(vals, '(', ', ', ')')
    }

    /**
     * gsub,sub(r,s[,t]) の実装.
     */
    private String genSub(String name, Object[] args, Object[] vals) {
        Object[] arr = vals
        String var = ""
        if (2 < args.length) {
            switch (args[2]) {
                case Node.YyVariable:
                    var = mkVerName(args[2] as Node.YyVariable)
                    // Targetが、'$0'の時は、呼び出しパラメータから削除(2重代入防止).
                    if ('_$[0]' == var)
                        arr = Arrays.copyOf(vals, 2)
                    break
                default:
                    throw new IllegalArgumentException("Target must be a variable. " +
                            args[2] + " " + args[2].getClass())
            }
        }
        String parm = mkString(arr, '', ', ', '')
        if (2 < arr.length) {
            varSequence += 1
            String cc = '_$' + varSequence // CLOSURE で実装.
            return "{int ${cc}=${name}(${parm}); ${var}=RESULT; ${cc}}.call()"
        }
        return "${name}(${parm})"
    }

    //* 単純名称(Simple Name)を返す.
    static String simpleFunctionName(String name) {
        return name.replaceFirst(/^.*[$]/, '')
    }
}