/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.spawn;

import java.util.Arrays;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * LibVar (singleton).
 * <p/>
 * The class to which this annotation is applied is immutable.
 *
 * @author kunio himei.
 */
final class Helper {

    /**
     * 正規表現にマッチしたグループ(3).
     */
    private static final int REGX_GROUP3 = 3;

    /**
     * ファイル名を表す正規表現 (file name).
     */
    private static final Pattern RX_FILENAME = Pattern.compile("^(?:" + //
            "([^'\"\\s]+)" //
            + "|\"([^\"\\\\]*(?:\\\\.[^\"\\\\]*)*)\"" // "([^\"\\]*(?:\\.[^\"\\]*)*)"
            + "|'([^']*)'" // '([^']*)'
            + ')');

    /**
     * 改行を表す正規表現文字列.
     */
    private static final String RX_NEWLINES = "\r\n|\r|\n"; // プラットフォーム非依存の改行

    /**
     * 1 要素の文字列配列.
     */
    private static final String[] STRING_ONE_ARRAY = {""};

    /**
     * Don't let anyone instantiate this class.
     */
    private Helper() {
        super();
    }

    /**
     * ファイル名を返す.
     */
    static String filename(String x) {
        Matcher m = RX_FILENAME.matcher(x);
        if (m.find()) {
            // System.err.println(matcher + "(" + matcher.group(1) + ":"
            //  + matcher.group(2) + ":" + matcher.group(3) + ")");
            String name = (null != m.group(1)) ? m.group(1) : ((null != m
                    .group(2)) ? m.group(2) : ((null != m.group(REGX_GROUP3))
                    ? m.group(REGX_GROUP3) : ""));
            assert null != name;
            if (!name.isEmpty()) {
                return name;
            }
        }
        throw new IllegalArgumentException(m.toString());
    }

    /**
     * ☆ Displays all elements of the Array in a string using start, end, and
     * separator strings.
     */
    @SuppressWarnings("SameParameterValue")
    static String mkString(String start, Object[] args) {
        StringBuilder sb = new StringBuilder("`");
        sb.append(start);
        for (Object m : args) {
            sb.append(' ');
            if (m instanceof Object[]) {
                sb.append(Arrays.toString((Object[]) m));
            } else {
                sb.append(m.toString().replace("\t", "\\t")
                        .replace("\r", "\\r").replace("\n", "\\n"));
            }
        }
        sb.append('`');
        return sb.toString();
    }

    /**
     * 行区切り文字列で分割した文字列を返す.
     */
    static String[] split(String x) {
        String[] a = x.split(RX_NEWLINES); // 入力データを配列に分割
        return (0 == a.length) ? STRING_ONE_ARRAY : a;
    }
}