/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

/**
 * Lexical Analyzer - Comment.
 *
 * @author kunio himei.
 */
abstract class Comment extends Advance {

    //* 行注釈を表わす正規表現.
    private static final LexRegx
            rxComment = new LexRegx("^(\\s*)(#.*|//.*|/\\*.*?\\*/)(.*)");
    //* 注釈開始 '/＊'.
    private static final LexRegx
            rxCommentStart = new LexRegx("^(\\s*)(/\\*.*)");
    //* 注釈終了 '＊/'.
    private static final LexRegx
            rxCommentEnd = new LexRegx("^(.*?\\*/)(.*)");
    //* 注釈バッファ.
    private final LexArray<String> commentBuf = new LexArray<>();
    //* 注釈処理が有効かどうか.
    private boolean isEnableComment;
    //* ブロック注釈の途中かどうか.
    private boolean isInComment;

    /**
     * マッチしたコメント位置を更新してコメントを格納する.
     */
    private void comment(String a, String b, int pre,
                         int post) {
        String x = a.substring(pre, a.length() - post);
        super.yyLexColumn += pre;
        super.yyText = b;
        if (isEnableComment && !x.isEmpty()) {
            commentBuf.add(x);
        }
    }

    /**
     * 注釈の解析をする.
     */
    @Override
    void doComment() {
        boolean hasNext = true;
        while (hasNext && !super.yyText.isEmpty()) {
            if (rxComment.find(super.yyText)) {
                String a = rxComment.group(2);
                String b = rxComment.group(3);
                if (a.startsWith("##")) {
                    isEnableComment = true;
                    comment(a, "", 2, 0); // ##
                } else if (a.startsWith("/*")) {
                    if (a.startsWith("/**")) {
                        isEnableComment = true;
                        comment(a, b, 3, 2); // /**
                    } else {
                        comment(a, b, 2, 2);
                    }
                } else if (a.startsWith("#")) {
                    comment(a, "", 1, 0); // #
                } else if (a.startsWith("//")) {
                    comment(a, "", 2, 0); // //
                } else {
                    throw new IllegalArgumentException("Comment: " + a);
                }
                isEnableComment = false;
            } else if (rxCommentStart.find(super.yyText)) {
                String a = rxCommentStart.group(2);
                if (isInComment) {
                    throw new IllegalArgumentException("Comment: " + a);
                }
                if (a.startsWith("/**")) {
                    isEnableComment = true;
                    comment(a, "", 3, 0); // /**
                } else if (a.startsWith("/*")) {
                    comment(a, "", 2, 0);
                }
                isInComment = true;
            } else if (rxCommentEnd.find(super.yyText)) {
                String a = rxCommentEnd.group(1);
                String b = rxCommentEnd.group(2);
                if (isInComment) { // */
                    comment(a, b, 0, 2);
                    isInComment = false;
                    isEnableComment = false;
                } else {
                    hasNext = false;
                }
            } else {
                if (isInComment) {
                    comment(super.yyText, "", 0, 0);
                }
                hasNext = false; // ..
            }
            lTrim();
            if (super.yyText.isEmpty()) {
                hasNext = false;
            }
        }
    }

    /**
     * 注釈バッファをクリアして、Listを返す.
     */
    LexArray<String> getComment() {
//        LexArray<String> x = commentBuf.cloneList(); // clone
//        commentBuf.clear();
//        return x;
        return commentBuf.moveList();
    }

    /**
     * 注釈バッファをクリアして、Arrayを返す.
     */
    String[] getCommentArray() {
//        if (!commentBuf.isEmpty()) {
//            String[] a = new String[commentBuf.size()];
//            return getComment().toArray(a);
//        }
//        return EMPTY_STRING_ARRAY;
        return commentBuf.moveArray();
    }

    /**
     * 注釈が存在するかどうかを返す.
     */
    boolean hasComment() {
        return !isEnableComment && !commentBuf.isEmpty();
    }
}