/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.eval;

import plus.concurrent.AtomicInterface;
import plus.lex.Keyword;
import plus.lex.Node;
import plus.lex.Term;
import plus.reflect.Listener;
import plus.reflect.Reflection;

/**
 * Evaluate - Invoke.
 *
 * @author kunio himei.
 */
abstract class EvalInvoke extends EvalFunc {

    /**
     * Import実装 - called from Invoke, Try-Catch, _is(Node.Bc).
     *
     * @param obj [Path.]Class
     *            NOTE Groovyは、Pathの一部を指定することはできない.
     *            - コンパイラとの互換性に注意.
     */
    Object applyImport(Object obj) {
        if (!(obj instanceof String) || // 文字列以外のインスタンスは、処理不要.
                isNil(obj)) return obj;
        String claz = "." + obj; // Path[.Class]
        for (String path : sTree.userImport)
            if (path.endsWith(claz))
                return path;
        for (String path : sTree.preImport)
            if (path.endsWith(claz))
                return path;
        return obj;
    }

    /**
     * Invoke.
     */
    Object invoke(Node.Invoke e) {
//        System.err.println(".invoke: '" + e.obj + "#" + e.name + "'");
        Object[] args = evalArgs(e.args);
        if (Keyword.SymHAS == e.id) { // has
            Tuple2 rr = getClassInfo(e.obj); // ☆
            return _has(rr.obj, e.name, args);
        } else if ((Keyword.SyyFVal == e.id) ||
                (Keyword.SyyFValFuture == e.id)) { // Function Value
            Tuple2 rr = getClassInfo(e.obj); // ☆
            // isXX の場合は、objで Invoke.
            Object x = (rr.isXX) ? rr.obj : null;
            return (Keyword.SyyFVal == e.id) ?
                    new Function(x, e.name, args) :
                    new ConcurrentFunction(x, e.name, args);
        } else if (Keyword.SymINVOKE == e.id) { // invoke
            return invokeImpl(e, args);
        } else if (Keyword.SyyNEW == e.id) { // new
            Tuple2 rr = getClassInfo(e.obj); // ☆
            return Reflection.newInstance(rr.obj.toString(), args);
        }
        throw new IllegalStateException("unmatch: " + e);
    }

    /**
     * Invoke 実装.
     */
    private Object invokeImpl(Node.Invoke e, Object[] args) {
        Tuple2 rr = getClassInfo(e.obj); // ☆
        String name = e.name;
        Object rs = null;
        for (boolean hasNext = true; hasNext; ) {
            String strObj = rr.obj.toString();
            if (rr.isXX) {
                if ((rr.obj instanceof Listener) && "apply".equals(name)) {
                    rs = ((Listener) rr.obj).apply(
                            (0 == args.length) ? EMPTY_ARRAY : args);
                } else {
                    rs = _invoke(rr.obj, name, args);
                }
                hasNext = false;
            } else if (_has(strObj)) { // Object が存在するとき
                Class<?> clazz = classForName(strObj);
                rs = _invoke(clazz, name, args);
                hasNext = false;
            } else {
                int ix = strObj.lastIndexOf('.');
                if (0 <= ix) {
                    rr.obj = strObj.substring(0, ix); // path.Class
                    name = strObj.substring(ix + 1); // method
                } else {
                    Object clazz = _getValue(strObj, null);
                    if (isNil(clazz)) {
                        clazz = strObj; // 変数でない場合は、リテラル
                        rs = _invoke(clazz, name, args);
                        hasNext = false;
                    }
                }
            }
        }
        return rs;
    }

    /**
     * クラスオブジェクトの属性を返す.
     */
    private Tuple2 getClassInfo(Object e) {
        if (e instanceof AtomicInterface) {
            return new Tuple2(e, false); // ☆false Atomic
        } else if (e instanceof Term.YyValue) {
            Term.YyValue x = (Term.YyValue) e;
            Object obj = applyImport(x.value);
            return new Tuple2(obj, false); // ☆false Value
        } else if (e instanceof Node.YyVariable) {
            Node.YyVariable x = (Node.YyVariable) e;
            Object obj = applyImport(_getValue(x.name, mkIndex(x.index)));
            return new Tuple2(obj, true); // ★true Variable
        } else {
            Object obj = applyImport(eval(e));
            return new Tuple2(obj, true); // ★true etc.
        }
    }

    /**
     * タプル戻り値.
     */
    static class Tuple2 {
        final boolean isXX; // TODO フラグisXXの意味を調査する.
        Object obj;

        Tuple2(Object obj, boolean isXX) {
            this.obj = obj;
            this.isXX = isXX;
        }
    }
}