/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.concurrent;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import plus.util.NumHelper;

import java.io.InterruptedIOException;
import java.io.Writer;
import java.nio.channels.Channel;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

/**
 * リンクノードに基づく、任意のバウンド形式のブロッキングキュー.
 *
 * @author Kunio Himei.
 */
public final class SimpleQueue extends LinkedBlockingQueue<Object>
        implements Channel {

    private static final long serialVersionUID = 1L;

    //* 待機中止するまでの時間.
    private static final long WAIT_MILLIS = 1;
    //* データ出力ストリーム.
    private final transient Writer writer = new TWriter();
    //* このｽﾄﾘｰﾑは閉じているか?.
    private volatile boolean isClose;

    //* 数値を書き込む.
    public void append(int x) {
        super.offer(x);
    }

    //* イミュータブル（変更不可）オブジェクトに変換した数値を書き込む.
    public void append(Number x) {
        super.offer(NumHelper.normalise(x));
    }

    //* 文字列を書き込む.
    public void append(String x) {
        super.offer(x);
    }

    //* キューの先頭の要素を取得して削除する.
    @SuppressWarnings("unchecked")
    public <E> E apply() throws InterruptedIOException {
        E rs = null;
        try {
            while ((null == rs) && isOpen()) {
                rs = (E) super.poll(WAIT_MILLIS, TimeUnit.MILLISECONDS);
            }
        } catch (InterruptedException e) { // 待機中に割り込みが発生した
            Thread.currentThread().interrupt(); // (InterruptedException)
            throw new InterruptedIOException(e.getMessage());
        }
        return rs;
    }

    //* キューの先頭の要素を取得して削除する.
    public double doubleValue() throws InterruptedIOException {
        Object x = apply();
        return (null == x) ? Double.MIN_VALUE : NumHelper.doubleValue(x);
    }

    //* キューの先頭の要素を取得して削除する.
    @Nullable
    public String getValue() throws InterruptedIOException {
        Object x = apply();
        return (null == x) ? null : x.toString();
    }

    //* キューの先頭の要素を取得して削除する.
    public int intValue() throws InterruptedIOException {
        Object x = apply();
        return (null == x) ? Integer.MIN_VALUE : NumHelper.intValue(x);
    }

    //* データを出力するストリームを返す.
    public Writer getWriter() {
        return this.writer;
    }

    //* 現在のチャネルの状態がオープンであるかどうかを返す.
    @Override
    public boolean isOpen() {
        return !this.isClose && !super.isEmpty();
    }

    //* ストリームを閉じる.
    @Override
    public void close() {
        this.isClose = true;
    }

    //* 出力ストリームの実装.
    private class TWriter extends Writer {

        //* ストリームを閉じる.
        @Override
        public void close() {
            SimpleQueue.this.close();
        }

        //* 何もしない.
        @Override
        public void flush() {
            // do nothing
        }

        //* 文字配列の一部を書き込む.
        @Override
        public void write(@NotNull final char[] buf, int off, int len) {
            SimpleQueue.this.append(String.valueOf(buf, off, len));
        }

        //* 文字列を書き込む.
        @Override
        public void write(@NotNull final String x) {
            SimpleQueue.this.append(x);
        }
    }
}