/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.eval;

import org.jetbrains.annotations.Nullable;
import plus.BiIO;
import plus.concurrent.AtomicNumber;
import plus.lex.Parser;
import plus.runtime.BuiltInVar;
import plus.util.NumHelper;

/**
 * Evaluate - Variable Accessor.
 *
 * @author kunio himei.
 */
abstract public class EvalVar extends BiIO {

    static final int NORMAL_RETURN = 0; // 正常復帰.
    static final int NIL = 0; // REMIND グローバル変数の初期値.
    static final Object[] EMPTY_ARRAY = {}; // 空のオブジェクト配列.
    static final Log logging = new Log(); // Logging
    Parser.SyntaxTree sTree;  // 構文木
    boolean isGlobalDef; // グローバル定義中か?

    //* 行番号を返す.
    static int scriptLineNumber() {
        return logging.getLineNumber();
    }

    /**
     * 変数値を取得する.
     */
    static Object _getValue(String name, Object index) {
        BuiltInVar x = BuiltInVar.forName(name);
        if (exists(x))
            return x.getAt(index);
        return _VARIABLES._getAt(name, index);
    }

    /**
     * 変数値を設定する.
     */
    static Object _putValue(String name, Object index, String op, Object val) {
        BuiltInVar x = BuiltInVar.forName(name);
        if (exists(x)) {
            if (isNil(index))
                return x.calculate(op, val);
            x.putAt(index, val);
            return val;
        }
        return _VARIABLES._putAt(op, name, index, val);
    }

    /**
     * Reference Variable - メソドが、引数に、値を返すために使用する.
     */
    static Object _getRefValue(String id, Object index) {
        BuiltInVar x = BuiltInVar.forName(id);
        if (exists(x))
            return x.getAt(index);
        return _VARIABLES._getRefVar(id, index);
    }

    /**
     * 参照変数値を設定する.
     */
    static Object _putRefValue(String id, Object index, Object value) {
        BuiltInVar x = BuiltInVar.forName(id);
        if (exists(x)) {
            x.putAt(index, value);
            return value;
        }
        return _VARIABLES._putRefVar(id, index, value);
    }

    //* ------------------------------------------------------------------------
    //* Helpers.
    //* ------------------------------------------------------------------------
    /*
     * Atomicを複製する.
     * - EvalAssign#assignStmt, Frame#calculate
     */
    public static Object _cloneAtom(Object e) {
        if (e instanceof Number)
            return new AtomicNumber((Number) e); // Atomic に昇格 ☆
        return e;
    }

    /**
     * 変数値を剝がす.(プリミティブオブジェクトにする)
     * - EvalAssign#assignStmt
     */
    static Object _strip(Object e) {
        if (e instanceof Number)
            return NumHelper.normalise((Number) e);
        return e;
    }

    /**
     * 関数呼出しパラメータの評価値配列を返す.
     * NOTE Groovyの副作用(sideEffects)に合わせて後方から評価する.
     */
    Object[] evalArgs(Object[] a) {
        Object[] arr = new Object[a.length];
        for (int i = a.length; --i >= 0; )
            arr[i] = _strip(eval(a[i])); // ストリップ値
        return arr;
    }

    /**
     * 配列の添え字を返す.
     */
    String mkIndex(Object[] a) {
        return (isNil(a)) ? null : _index(evalArgs(a));
    }

    /**
     * オブジェクトの評価結果を返す.
     */
    @Nullable
    abstract Object eval(Object e);
}