/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

/**
 * Lexical Analyzer - Type Flags (BitMap).
 *
 * @author kunio himei.
 */
public final class Flags {
    /**
     * 文字列 (String).
     */
    public static final int T07STR = 2;
    /**
     * AnyRef (Object).
     */
    public static final int T11ANY = 16 | 15;
    /**
     * Array (Map).
     */
    public static final int T16ARRAY = 1024;
    /**
     * 複数の return が出現 (コンパイラで使用).
     */
    public static final int T23RETURNS = 131072;
    /**
     * Atomic属性 (コンパイラで使用).
     */
    public static final int T24ATOM = 262144;
    /**
     * Void.
     */
    static final int T00VOID = 16;
    /**
     * 数値 (Double).
     */
    static final int T01NUM = 1;
    /**
     * 整数 (Integer).
     */
    public static final int T03INT = 32 | T01NUM;
    /**
     * 実数 (Double).
     */
    public static final int T06DBL = 256 | T01NUM;
    /**
     * 論理値 (Boolean).
     */
    static final int T02BOOL = 16 | T01NUM;
    /**
     * 長整数 (Long).
     */
    static final int T04LONG = 64 | T01NUM;
    /**
     * 実数 (Float).
     */
    static final int T05FLOAT = 128 | T01NUM;
    /**
     * 正規表現.
     */
    static final int T08REGX = 16 | T07STR;
    /**
     * STRNUM.
     */
    static final int T09STNUM = 32 | T07STR | T01NUM;
    /**
     * Class.
     */
    static final int T10CLAZ = 8;
    /**
     * MASK (数値 文字列 OBJECT).
     */
    static final int T12X15M = T11ANY;
    /**
     * OBJECT.
     */
    static final int T13OBJ = 256 | T11ANY;
    /**
     * MASK (数値 文字列 論理値 正規表現).
     */
    static final int T14X512M = 512 - 1;
    /**
     * 可変長引数: Variable Arguments.
     */
    static final int T15VARG = 512;
    /**
     * 関数値: Function Value.
     */
    static final int T17FVAL = 2048;
    /**
     * 参照変数: Reference Variable.
     */
    static final int T18REF = 4096;
    /**
     * 変数の変更不可.
     */
    static final int T19VALTYPE = 8192;
    /**
     * 変数への代入有り.
     */
    static final int T20ASSIGN = 16384;
    /**
     * for var/val.
     */
    static final int T21FORVAR = 32768;
    /**
     * クロージャ.
     */
    static final int T22CLOSURE = 65536; // CLOSURE を参照している.
    /**
     * 属性未設定.
     */
    static final int T25UNDEFIN = 524288;
    /**
     * 初期値 NIL.
     */
    public static final int T26NIL = T25UNDEFIN | T09STNUM;
    /**
     * Typeは無効.
     */
    static final int T27DISABLE = -1;

    /**
     * Don't let anyone instantiate this class.
     */
    private Flags() {
        super();
    }

    /**
     * 関数からの AnyRef 戻り値 属性かどうかを返す.
     */
    public static boolean isAnyRefReturn(int x) {
        return (T11ANY | T20ASSIGN) == x;
    }

    /**
     * AnyType 属性かどうかを返す.
     */
    static boolean isAnyType(int x) {
        return T11ANY == (x & T12X15M);
    }

    /**
     * ARRAY 属性かどうかを返す.
     */

    public static boolean isArray(int x) {
        return 0 != (x & T16ARRAY);
    }

    /**
     * ASSIGN 属性かどうかを返す.
     */
    public static boolean isAssign(int x) {
        return 0 != (x & T20ASSIGN);
    }

    /**
     * ATOMIC 属性かどうかを返す.
     */
    public static boolean isAtomic(int x) {
        return 0 != (x & T24ATOM);
    }

    /**
     * BOOLEAN 属性かどうかを返す.
     */
    static boolean isBoolean(int x) {
        return T02BOOL == (x & T14X512M);
    }

    /**
     * CLASS 属性かどうかを返す.
     */
    static boolean isClass(int x) {
        return T10CLAZ == (x & T12X15M);
    }

    /**
     * CLOSURE 属性かどうかを返す.
     */
    public static boolean isClosure(int x) {
        return 0 != (x & T22CLOSURE);
    }

    /**
     * DOUBLE 属性かどうかを返す.
     */
    static boolean isDouble(int x) {
        return T06DBL == (x & T14X512M);
    }

    /**
     * 関数値属性かどうかを返す.
     */
    public static boolean isFuncVar(int x) {
        return 0 != (x & T17FVAL);
    }

    /**
     * INTEGER 属性かどうかを返す.
     */
    public static boolean isInteger(int x) {
        return T03INT == (x & T14X512M);
    }

    /**
     * NUMBER 属性かどうかを返す.
     */
    public static boolean isNumber(int x) {
        return T01NUM == (x & T12X15M);
    }

    /**
     * 参照変数属性かどうかを返す.
     */
    public static boolean isReference(int x) {
        return 0 != (x & T18REF);
    }

    /**
     * 文字列属性かどうかを返す.
     */
    public static boolean isString(int x) {
        return T07STR == (x & T12X15M);
    }

    /**
     * STRNUM 属性かどうかを返す.
     */
    public static boolean isStrNum(int x) {
        return T09STNUM == (x & T14X512M);
    }

    /**
     * 可変長引数属性かどうかを返す.
     */
    public static boolean isVarArgs(int x) {
        return 0 != (x & T15VARG);
    }

    /**
     * Void 属性かどうかを返す.
     */
    static boolean isVoid(int x) {
        return T00VOID == (x & T12X15M);
    }
}