/*
 * Copyright (C) 2011 awk4j - http://awk4j.sourceforge.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.io;

import org.jetbrains.annotations.NotNull;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.SocketAddress;
import java.util.concurrent.locks.ReentrantLock;

/**
 * [%inet%] データグラムソケットの実装 (シングルスレッド).
 *
 * @author kunio himei.
 */
final class InetDatagramSocket extends OutputStream implements Connectable,
        ReadWriteable {

    /**
     * byte バッファーサイズ.<br>
     * IP最大パケット長(65535)-IPヘッダ長(20)-UDPヘッダ長(8)
     */
    private static final int DEFAULT_BUFFER_SIZE = 65507;
    /**
     * Lock Object.
     */
    private final ReentrantLock SYNC = new ReentrantLock(); // SYNC.
    /**
     * データグラムソケット.
     */
    private final DatagramSocket socket;
    /**
     * ソケットアドレス.
     */
    private final SocketAddress socketAddress;
    /**
     * 接続待ち時間.
     */
    private final int timeout;
    /**
     * リモートアドレス.
     */
    private volatile SocketAddress remoteAddress;

    /**
     * リモートソケットの構築.
     *
     * @param address ソケットアドレス
     * @param tout    接続待ち時間
     */
    InetDatagramSocket(final SocketAddress address, final int tout)
            throws IOException {
        this.socketAddress = address;
        this.socket = new DatagramSocket();
        this.socket.setReuseAddress(true); // SO_REUSEADDR 有効化
        this.socket.setBroadcast(true); // SO_BROADCAST 有効化
        this.socket.connect(address); // リモートアドレスに接続
        this.timeout = tout;
        // System.err.println("InetDatagramSocket: " + address);
    }

    /**
     * ローカルソケットの構築.
     *
     * @param address   ソケットアドレス
     * @param localport ポート番号 (not used)
     * @param tout      接続待ち時間
     */
    InetDatagramSocket(final SocketAddress address, final int localport,
                       final int tout) throws IOException {
        this.socketAddress = address;
        this.socket = new DatagramSocket(address); // バインド
        assert 0 <= localport;
        this.timeout = tout;
        // System.err.println("InetDatagramSocket: " + address + ": " + localport);
    }

    /**
     * ソケットを閉じる.
     */
    @Override
    public void close() {
        if (!this.socket.isClosed()) {
            // System.err.println("InetDatagramSocket#close");
            this.socket.close();
        }
    }

    /**
     * 接続されたソケットインターフェイスを返す.
     */
    @Override
    public ReadWriteable connect() {
        return this; // このクラスを返す
    }

    /**
     * ソケットリーダを返す (マルチスレッド呼び出しは不可).
     */
    @Override
    public InputStream getInputStream() throws IOException {
        final DatagramPacket packet = new DatagramPacket(
                new byte[DEFAULT_BUFFER_SIZE], DEFAULT_BUFFER_SIZE);
        this.socket.setSoTimeout(this.timeout); // 接続待ち時間
        this.SYNC.lock(); // SYNC getInputStream.
        try {
            this.socket.receive(packet);
            // REMIND: マルチスレッドでの使用は不可
            // 送信者アドレスを退避
            this.remoteAddress = packet.getSocketAddress();
            //System.err.println("InetDatagramSocket#getInputStream: '"
            //  + new String(packet.getData(), packet.getOffset(), packet.getLength()) + "'");
        } finally {
            this.SYNC.unlock();
        }
        return new Treader(packet.getData(), packet.getOffset(),
                packet.getLength());
    }

    /**
     * ソケットライタを返す.
     */
    @Override
    public OutputStream getOutputStream() {
        return this;
    }

    /**
     * 指定されたバイト配列を書き込む.
     *
     * @param b   バイト配列
     * @param off データの開始オフセット
     * @param len 書き込むバイト数
     */
    @Override
    public void write(@NotNull final byte[] b, final int off, final int len)
            throws IOException {
        // System.err.println("InetDatagramSocket: " + new String(b, off, len));
        final SocketAddress address = ((null == this.remoteAddress) || this.socket
                .isConnected()) ? this.socketAddress : this.remoteAddress;
        this.socket.send(new DatagramPacket(b, off, len, address));
    }

    /**
     * 指定されたバイトを書き込む (abstract).
     *
     * @param b byte
     */
    @Override
    public void write(final int b) throws IOException {
        final byte[] minibuf = {(byte) b}; // local array
        this.write(minibuf, 0, 1);
    }

    /**
     * ソケットリーダの実装.
     */
    private final class Treader extends ByteArrayInputStream {

        /**
         * ソケットリーダの構築.
         *
         * @param buffer バイト配列
         * @param offset データの開始オフセット
         * @param length データのバイト数
         */
        Treader(final byte[] buffer, final int offset, final int length) {
            super(buffer, offset, length);
        }

        /**
         * ソケットを閉じる.
         */
        @Override
        public void close() {
            // super.close(); // 配列での実装 のため何もしない
            InetDatagramSocket.this.close();
        }
    }
}