/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus;

import plus.reflect.Reflection;
import plus.runtime.BuiltInVar;
import plus.runtime.RegExp;
import plus.runtime.RunHelper;
import plus.util.Compare;
import plus.util.NumHelper;

import java.util.*;

// ------------------------------------------------------------------------
// Built-in - Accessor.
// ------------------------------------------------------------------------
public class BiBase extends BiAccessor {
    //* ------------------------------------------------------------------------
    //* System Functions.
    //* ------------------------------------------------------------------------
    private static double randomSeed; // 乱数ジェネレータのシード
    private static Random random = new Random();

    public static double rand() {
        return random.nextDouble();
    }

    public static double srand(Object... x) {
        double oldseed = randomSeed;
        randomSeed = (0 == x.length) ?
                systime() : NumHelper.doubleValue(x[0]);
        random = new Random((long) randomSeed);
        return oldseed;
    }

    //* 基準時点 (1970/01/01 00:00:00 UTC,) からの経過時間(sec.)を返す.
    public static double systime() {
        return 0.001 * System.currentTimeMillis();
    }

    //* 指定したタイムスタンプをフォーマットして返す.
    // strftime([format [,timestamp]])
    public static String strftime(Object... x) {
        if (1 > x.length) return new java.util.Date().toString();
        long timestamp = (1 < x.length) ?
                (long) (NumHelper.numberValue(x[1]).doubleValue() * 1000) :
                java.lang.System.currentTimeMillis();
        return RunHelper.strftime(x[0].toString(), timestamp);
    }

    public static int system(String x) {
        return RunHelper.system(x);
    }

    //* ------------------------------------------------------------------------
    //* String Functions.
    //* ------------------------------------------------------------------------
    /*
     * オブジェクトの長さ または 数を返す.
     */
    public static int length(Object... args) {
        if (isNil(args)) {
            return BuiltInVar.$.getAt(0).toString().length(); // $0
        }
        return RunHelper.length(args[0]);
    }

    /**
     * 正規表現 r でターゲット文字列 t を s で一括置換し、
     * 置換数(RNUMBER)を設定して、置換後の文字列を返す.
     * gsub(r, s, t)
     */
    public static int gsub(Object... args) {
        return RegExp.gsub(args);
    }

    /**
     * 正規表現 r でターゲット文字列 t を s で置換し、
     * 置換数(RNUMBER)を設定して、置換後の文字列を返す.
     * sub(r, s, t)
     */
    public static int sub(Object... args) {
        return RegExp.sub(args);
    }

    /**
     * s が、 r に適合する位置を、また適合しないときは、0 を返し、
     * RSTART と RLENGTH を設定する.
     */
    public static int match(Object s, Object r) {
        return RegExp.match(s, r);
    }

    /**
     * 文字列(s)を正規表現(r)で区切って分割し結果(StringNumber)を配列(a)に設定する.
     */
    public static int split(Object... x) {
        String s = x[0].toString();
        //noinspection unchecked
        Map<String, Object> a = (Map<String, Object>) x[1];
        String r = (2 < x.length) ?
                x[2].toString() : BuiltInVar.FS.toString();
        return RunHelper.split(s, a, r);
    }

    /**
     * source 配列の要素をソート。dest(または、source)に出力し要素数を返す.
     */
    public static int asort(Object... a) {
        Set<?> src = new TreeSet<>(((Map<?, ?>) a[0]).values());
        return asortImpl(src, a);
    }

    /**
     * source 配列の添字をソート。dest(または、source)に出力し要素数を返す.
     */
    public static int asorti(Object... a) {
        Set<?> src = new TreeSet<>(((Map<?, ?>) a[0]).keySet());
        return asortImpl(src, a);
    }

    private static int asortImpl(Set<?> source, Object... a) {
        @SuppressWarnings("unchecked")
        Map<String, Object> dest = (2 > a.length) ?
                (Map<String, Object>) a[0] : (Map<String, Object>) a[1];
        dest.clear();
        int i = 0;
        for (Object x : source) dest.put(Integer.toString(++i), x);
        return source.size();
    }

    //* ------------------------------------------------------------------------
    //* Arithmetic Functions.
    //* ------------------------------------------------------------------------
    public static int abs(int x) {
        return Math.abs(x);
    }

    public static double abs(Number x) {
        return Math.abs(x.doubleValue());
    }

    public static double atan2(Number x, Number yy) {
        return Math.atan2(x.doubleValue(), yy.doubleValue());
    }

    public static double cos(Number x) {
        return Math.cos(x.doubleValue());
    }

    public static double exp(Number x) {
        return Math.exp(x.doubleValue());
    }

    public static double log(Number x) {
        return Math.log(x.doubleValue());
    }

    public static double sin(Number x) {
        return Math.sin(x.doubleValue());
    }

    public static double sqrt(Number x) {
        return Math.sqrt(x.doubleValue());
    }

    //* ------------------------------------------------------------------------
    //* Helpers.
    //* ------------------------------------------------------------------------
    public static String _cat(Object... args) { //
        return RunHelper.concat("", null, args);
    }

    public static boolean _if(Object x) {
        return Compare.boolValue(x);
    }

    //* `< > == != <= >= 正規表現適合 ~ !~'
    public static boolean _if(String op, Object left, Object right) { //
        return Compare.compareTo(op, left, right);
    }

    //* クラスの存在確認.
    public static boolean _has(String name) {
        return Reflection.hasClass(name);
    }

    //* メソドまたはフィールドの存在確認 (Duck Typing)
    public static boolean _has(Object x, String name, Object... args) {
        return Reflection.has(x, name, args);
    }

    /**
     * 配列キーの存在確認 - for Java.
     */
    public static boolean _in(Object arr, Object... key) {
        if (arr instanceof Map<?, ?>) { // Map
            return Compare.compareIn(arr, _index(key));
        }
        return Compare.compareIn(arr, // List, []
                NumHelper.numberValue(key[0]));
    }

    //* 数値キーの存在確認 - for Groovy.
    public static boolean _in(Map<?, ?> arr, Number key) {
        Number x = NumHelper.normalise(key);
        return (arr.containsKey(x) ||
                arr.containsKey(x.toString())); // 数値または、文字列でチェックする.
    }

    //* 配列のアクセスキー、double値は、CONVFMTで変換して返す.
    public static String _index(Object... index) {
        if (isNil(index)) return "";
        if ((1 == index.length) && (index[0] instanceof Number)) {
            Number num = NumHelper.normalise(((Number) index[0]));
            if ((num instanceof Integer) || (num instanceof Long))
                return num.toString();
        }
        return RunHelper.concat(BuiltInVar.SUBSEP.toString(), null, index);
    }

    //* ------------------------------------------------------------------------
    //* Internal Modules.
    //* ------------------------------------------------------------------------

    /**
     * Invoke Method or Field.
     */
    public static Object _invoke(Object x, String name, Object... args) {
        return Reflection.invoke(x, name, args);
    }

    /**
     * 指定されたオブジェクトが、このクラスと代入互換の関係にあるかどうかを返す.
     */
    public static boolean _is(Object x, String claz) {
        return classForName(claz).isInstance(x);
    }

    /**
     * クラスオブジェクトを返す.
     */
    public static Class<?> classForName(Object x) {
        return Reflection.classForName(x);
    }

    //* ------------------------------------------------------------------------
    //* Utils.
    //* ------------------------------------------------------------------------
    //* 評価対象のオブジェクトに対するイテレータを返す.
    public static Iterator<?> iterator(Object a) {
        if (a instanceof Iterator) return (Iterator<?>) a;
        if (a instanceof Object[])
            return Arrays.asList((Object[]) a).iterator();
        if (a instanceof Collection) return ((Collection<?>) a).iterator();
        if (a instanceof Map) return ((Map<?, ?>) a).keySet().iterator();
        throw new IllegalStateException(a.getClass().getName());
    }

    //* ------------------------------------------------------------------------
    //* Assert. メッセージが、'!'のときは、throwする.
    //* ------------------------------------------------------------------------
    public static boolean assertEquals(int lineNumber, Object... a) {
        boolean bo = _if("==", a[0], a[1]);
        if (!bo) {
            Object expect = (a[0] instanceof String) ? "'" + a[0] + "'" : a[0]; // 期待値
            Object actual = (a[1] instanceof String) ? "'" + a[1] + "'" : a[1]; //実際の値
            Object[] info = Arrays.copyOfRange(a, 2, a.length);
            String x = isNil(info) ? "" : " " + Arrays.toString(info);
            System.err.println(lineNumber + "\tassertEquals " + expect + " : " + actual + x);
            if (exists(info) && info[0].equals("!"))
                throw new AssertionError(a[1].getClass().getName());
        }
        return bo;
    }

    public static boolean assertTrue(int lineNumber, Object... a) {
        boolean bo;
        Object condition = a[0];
        if (condition instanceof Boolean)
            bo = ((Boolean) condition);
        else if (condition instanceof Number)
            bo = 0 != ((Number) condition).longValue();
        else bo = false;
        if (!bo) {
            Object[] info = Arrays.copyOfRange(a, 1, a.length);
            String x = isNil(info) ? "" : " " + Arrays.toString(info);
            System.err.println(lineNumber + "\tassertTrue" + x);
            if (exists(info) && info[0].equals("!"))
                throw new AssertionError(condition.getClass().getName());
        }
        return bo;
    }
}