/*
 * Copyright (C) 2010 awk4j - http://awk4j.sourceforge.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.io;

import plus.BiIO;

import java.io.File;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileTime;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static java.nio.file.StandardCopyOption.COPY_ATTRIBUTES;
import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;

/**
 * nanoShell - Built-in command
 *
 * @author kunio himei.
 */
public class NanoTools {
    private Analysis ana;
    private boolean isRemove;

    /**
     * @param x command line arguments
     */
    private void Initialize(String input, Object... x) {
        ana = new Analysis(input);
        isRemove = false; // never delete
        optAll.clear(); // そのモジュールで利用可能なオプション
        optIn.clear(); // 明に適用されたオプション
        optARGV.clear(); // ユーザ定義された(生)オプション
        for (Object o : x) {
            String[] opts = o.toString().trim().toLowerCase()
                    .split("\\s+");
            optARGV.addAll(List.of(opts));
        }
        optARGV.remove("");
        __STDOUT = defaultOUT;
        __STDERR = defaultERR;
        redirect(x);
    }

    private void close() {
        closeImpl(__STDOUT);
        closeImpl(__STDERR);
    }

    private void closeImpl(Redirect re) {
        try {
            BiIO.fflush(re.file);
            if (re.defined)
                BiIO.close(re.file);
        } catch (IOException e) {
            throw new RuntimeException(throwMessage("close", re.name));
        }
    }

    /**
     * Parsing redirect definitions
     *
     * @param x command line arguments (生データ)
     */
    private void redirect(Object... x) {
        for (Object o : x) {
            if (o instanceof String e) {
                int ix1 = e.indexOf('>');
                if (ix1 < 0) continue;
                int ix2 = e.indexOf('>', ix1 + 2); // skip >>
                if (ix2 >= 0) {
                    ix2 = e.lastIndexOf(' ', ix2);
                    if (ix2 >= 0) {
                        redirectImpl(e.substring(0, ix2));
                        redirectImpl(e.substring(ix2 + 1));
                    } else {
                        throw new IllegalArgumentException(throwMessage(
                                "redirect", e));
                    }
                } else {
                    redirectImpl(e);
                }
            }
        }
    }

    // <- SPACE
    private static final char RE_QUOTE = '\'';
    private static final char BOM_QUOTE_BE = '\uFEFF'; // UTF16-BE <- \'
    private static final Pattern IS_REDIRECT = Pattern
            .compile("([12]?)(>{1,2})\\s*(" + RE_QUOTE + "?)(.+)");

    private void redirectImpl(String input) {
        String src = input.trim()
                .replace("\\" + RE_QUOTE, Character.toString(BOM_QUOTE_BE)); // \'
        Matcher m = IS_REDIRECT.matcher(src);
        if (m.find()) {
            String rno = getValue(m.group(1));
            String rid = getValue(m.group(2));
            boolean brackets = !getValue(m.group(3)).isEmpty();
            String file = getValue(m.group(4));
            if (brackets) {
                int ix = file.lastIndexOf(RE_QUOTE); // '...'
                if (ix >= 0) { // 終端の ' を剥がす
                    file = file.substring(0, ix);
                } else {
                    throw new IllegalArgumentException(throwMessage(
                            "Paired <'> mistake", input));
                }
            }
            file = file.trim()
                    .replace(BOM_QUOTE_BE, RE_QUOTE);
            String name = rno + rid + file;
            Redirect re = new Redirect(true, name,
                    rid, file, rno.equals("2"));
            if (re.isErr) this.__STDERR = re;
            else this.__STDOUT = re;
            optIn.add(name); // Used in option display

//            messageGREEN("re " + input, file);
        }
    }

    private Redirect __STDOUT; // リダイレクト
    private Redirect __STDERR; //
    private static final Redirect defaultOUT =
            new Redirect(false, "", "", Io.STDOUT, false);
    private static final Redirect defaultERR =
            new Redirect(false, "", "", Io.STDERR, true);

    private static final int STATUS_ERROR = 3;
    private static final int STATUS_WARNING = 2;
    private static final int STATUS_FILE = 1; // request file
    private static final int STATUS_INFORMATION = STATUS_FILE;
    private static final int STATUS_NORMAL = 0; // folder or file

    private static final int FLAG_ROOT = 1;
    private static final int FLAG_DIRECTORY = 2;
    private static final int FLAG_FILE = 4;

    /**
     * @param input Check paths for deletion
     * @param flags Allow root specification ('/') (e.g. tree)
     * @return STATUS
     */
    private static int checkPath(Path input, int flags) {
        Path path = input.normalize().toAbsolutePath();
        if (Files.isDirectory(path)) {
            boolean isRoot = path.equals(path.getRoot());
            if (isRoot && (flags & FLAG_ROOT) == 0) {
                rootCannotBeSpecified(input);
                return STATUS_ERROR; // ルートフォルダは指定できない
            }
//            messageGREEN("ck dir " + input, Files.exists(path));
            return STATUS_NORMAL;
        } else if ((flags & FLAG_FILE) != 0) {
//            messageGREEN("ck file " + input, Files.exists(path));
            return Files.exists(path) ?
                    STATUS_NORMAL : // ファイルが検出された
                    STATUS_FILE; // まだ作成されていない
        }
        noSubfolderExist(input);
        return STATUS_WARNING; // ファイル・フォルダが存在しない
    }

    private static final Pattern IS_PATH_HAS_FILE_UNIX = Pattern
            .compile("(.*/)*([^/]*)$");
    private static final Pattern IS_PATH_HAS_FILE_WIN = Pattern
            .compile("(.*[/\\\\])*([^/\\\\]*)$");
    private static final Pattern IS_PATH_HAS_FILE =
            File.separatorChar == '/' ?
                    IS_PATH_HAS_FILE_UNIX :
                    IS_PATH_HAS_FILE_WIN;

    /**
     * 出力パスが ./foo/ - '/'で終わるとき、フォルダを作成する
     */
    private static Path createOutFolder(String output) {
        String src = output.trim();
        Path out = Path.of(src).normalize();
        if (Files.exists(out)) return out; // shortcut
        try {
            Matcher m = IS_PATH_HAS_FILE.matcher(src);
            if (m.matches()) {
                boolean hasFile = !getValue(m.group(2)).isEmpty();
                Path path = hasFile ? out.getParent() : out;
                if (null != path) {
                    Files.createDirectories(path);
                }
//                messageGREEN("createOutFolder", m.group(1) + " " + m.group(2));
            }
        } catch (IOException e) {
            throw new RuntimeException(throwMessage("md " + src, e));
        }
        return out;
    }

    // Optional parameter variable
    private final Map<String, Object> optAll = new TreeMap<>(); // そのモジュールで利用可能なオプション
    private final Set<String> optIn = new TreeSet<>(); // 明に適用されたオプション
    private final Set<String> optARGV = new HashSet<>(32); // ユーザ定義された(生)オプション

    private static int MAX_PATH = 260; // 長いパスの警告を表示する敷居値
    private static int DISPLAY_WIDTH = 40; // 画面への表示幅

    private static boolean isLonger260(File input) {
        return input.getPath().length() > MAX_PATH;
    }

    private static final String FILE = "file"; // tree
    private static final String MX_260 = "260"; // ls
    private static final String MX_no260 = "no260"; // ls
    private static final String ROOT = "root"; // remove
    private static final String SYNC = "sync"; // copy
    private static final String SIMPLE = "0"; // ls size
    private static final String COMMA = ","; // ls size
    private static final String UNIT = "k"; // ls size
    private static final String PATH = "path"; // ls path length
    private static final String noRECURSIVE = "noRecursive";
    private static final String MAX_PATH_KEY = "MAX_PATH"; // tree, ls
    private static final String DISPLAY_WIDTH_KEY = "DISPLAY_WIDTH"; // all

    /*
     * 大小文字を区別しないで前方一致で比較
     */
    private boolean applyOption(String name, boolean... value) {
        boolean val = value.length == 0; // 省略時は真
        String key = name.toLowerCase();
        String key2 = key.startsWith("no") ?
                '-' + key.substring(2) : key;
        for (String x : optARGV) {
            if (x.isEmpty()) continue;
            if (key.startsWith(x) || key2.startsWith(x)) {
                optIn.add(name);
                optAll.put(name, val);
                return val;
            }
        }
        optAll.put(name, !val);
        return !val;
    }

    private static final Pattern IS_SET = Pattern
            .compile("(\\w+)=([-+]?\\d+)");

    private int applyShellVariable(String name, int value) {
        String key = name.toLowerCase();
        for (String x : optARGV) {
            if (x.isEmpty()) continue;
            Matcher m = IS_SET.matcher(x);
            while (m.find()) {
                String g1 = getValue(m.group(1));
                if (key.startsWith(g1)) {
                    String g2 = getValue(m.group(2));
                    int val = Integer.parseInt(g2);
                    String var = name + '=' + g2;
                    optIn.add(BLUE + var + RESET);
                    return val;
                }
            }
        }
        optIn.add(name + '=' + value);
        return value;
    }

    private boolean getBooOption(String name) {
        if (optAll.containsKey(name)) {
            Object o = optAll.get(name);
            if (o instanceof Boolean e)
                return e;
        }
        throw new RuntimeException(throwMessage("No option", name));
    }

    private void resetOptions(String... x) {
        for (String key : x) {
            if (optAll.containsKey(key)) {
                optAll.put(key, false);
                optIn.remove(key);
            } else {
                throw new RuntimeException(throwMessage("No option", key));
            }
        }
    }

    private String listOptions() {
        StringBuilder sb = new StringBuilder(64);
        sb.append(BLUE);
        for (String x : optIn) {
            sb.append(' ').append(x);
        }
        return sb.append(RESET).toString();
    }

    // System フォルダをスキップする
    private static boolean isSystem(Path path) {
        String str = path.toString();
        return str.contains("System Volume Information") ||
                str.contains("$RECYCLE.BIN");
    }

    private static boolean isSystem(File path) {
        return isSystem(path.toPath());
    }

    // ワイルドカードにマッチング
    private boolean isMatch(Path path) {
        return isMatch(path.toFile());
    }

    private boolean isMatch(File path) {
        if (path.exists() && !path.isDirectory()) {
            if (ana.alwaysTrue) return true;
            String name = path.getName();
            return ana.regex.matcher(name).matches();
        }
        return false;
    }

    /**
     * long time = path.lastModified();
     */
    private static void setLastModified(Path path, long time) {
        File file = path.toFile();
        if (0 < time && time != file.lastModified()) {
            if (!file.setLastModified(time))
                throw new RuntimeException(throwMessage("setLastModified", path));
        }
    }

    /**
     * throw を発生させない (フォルダが空でない場合の対策)
     *
     * @param path フォルダ or ファイル
     */
    private boolean safetyRemove(Path path) {
        return isRemove && Files.exists(path) &&
                !isSystem(path) && path.toFile().delete();
    }

    private static final boolean _STDOUT = true;
    private static final boolean _STDERR = false;

    /**
     * @param stdout [Default value: hasSTDOUT] else, hasSTDERR
     */
    private void printX(boolean stdout, String x) {
        Redirect re = stdout ? __STDOUT : __STDERR;
        BiIO.print(re.rid, re.file, x);
    }

    /**
     * @param stdout True: hasSTDOUT, False: hasSTDERR
     * @param type   [Default value: path value] else, USE_PATH_NAME
     */
    @SuppressWarnings("SameParameterValue")
    private void printX(boolean stdout, File file, boolean... type) {
        Redirect re = stdout ? __STDOUT : __STDERR;
        if (re.defined()) {
            printX(stdout, applySlashSeparator(file, type));
        } else {
            printX(stdout, truncatePath(file, type));
        }
    }

    /**
     * @param stdout True: hasSTDOUT, False: hasSTDERR
     * @param type   [Default value: path value] else, USE_PATH_NAME
     */
    @SuppressWarnings("SameParameterValue")
    private String sprintX(boolean stdout, File file, boolean... type) {
        Redirect re = stdout ? __STDOUT : __STDERR;
        if (re.defined()) {
            return applySlashSeparator(file, type);
        } else {
            return truncatePath(file, type);
        }
    }

    /**
     * set - シェル変数の設定.
     */
    public int set(Object... x) {
        Initialize("./", x);
        boolean hasARGV = optARGV.size() > 0;
        MAX_PATH = applyShellVariable(MAX_PATH_KEY, MAX_PATH);
        DISPLAY_WIDTH = applyShellVariable(DISPLAY_WIDTH_KEY, DISPLAY_WIDTH);
        String args = hasARGV ? listOptions() : "";
        messageTitle("set", args);
        if (!hasARGV) {
            for (String k : optIn) {
                System.out.println(k);
            }
        }
        messageCYAN("Number of processed", optIn.size());
        return STATUS_NORMAL;
    }

    /**
     * copy the file.
     *
     * @param input  file or folder
     * @param output file or folder
     */
    public int copy(String input, String output, Object... x) {
        Initialize(input, x);
        Path in = ana.path;
        Path out = createOutFolder(output);
        isRemove = applyOption(SYNC);
        boolean isRecursive = applyOption(noRECURSIVE, false);
        String args = ana.virtualPath + ' ' + output + listOptions();
        messageTitle("copy", args);
        if (in.compareTo(out) == 0) {
            inputAndOutputAreSamePath(in);
            return STATUS_ERROR; // in, out are the same
        }
        int rc = checkPath(in, FLAG_ROOT | FLAG_DIRECTORY | FLAG_FILE);
        int ro = checkPath(out, FLAG_ROOT | FLAG_DIRECTORY | FLAG_FILE);
        rc = Math.max(rc, ro);
        if (rc <= STATUS_INFORMATION) {
            Set<File> set = new TreeSet<>();
            if (isRemove) {
                copySync(set, in, out); // synchronous copy
                if (set.size() > 0) {
                    for (File sync : set) {
                        printX(_STDERR, sync);
                    }
                    messageCYAN("Synchronized", set.size());
                    set.clear();
                }
            }
            copyImpl(set, in, out, isRecursive);
            for (File file : set) {
                printX(_STDOUT, file);
            }
            isRemove = true; // always delete
            setDateForChildElement(out); // clean up folder
            messageCYAN("Number of processed", set.size());
        }
        close();
        return rc;
    }

    private void copyImpl(Set<File> set, Path input, Path output,
                          boolean recursive) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            File[] files = input.toFile().listFiles();
            if (null != files) {
                for (File file : files) {
                    if (isSystem(file)) continue; // skip System folder
                    Path newIn = file.toPath();
                    Path newOut = output.resolve(newIn.getFileName());
                    if (file.isDirectory()) {
                        if (recursive) {
                            createDirectory(newIn, newOut); // 日時を設定
                            //noinspection ConstantValue
                            copyImpl(set, newIn, newOut, recursive);
                        }
                    } else {
                        atomicSingleCopy(set, newIn, newOut);
                    }
                }
            }
        } else {
            atomicSingleCopy(set, input, output);
        }
    }

    /**
     * Synchronous copy
     */
    private void copySync(Set<File> syn, Path input, Path output) {
        Map<String, File> map = new HashMap<>(256);
        if (isSystem(input) || isSystem(output)) return; // skip System folder
        if (Files.isDirectory(output)) { // output
            File[] files = output.toFile().listFiles();
            if (null != files) {
                for (File file : files) {
                    if (isSystem(file)) continue; // skip System folder
                    map.put(file.getName(), file);
                }
            }
        }
        if (Files.exists(output)) {
            File file = output.toFile();
            map.put(file.getName(), file);
        }
        if (Files.isDirectory(input)) { // input
            File[] files = input.toFile().listFiles();
            if (null != files) {
                for (File file : files) {
                    if (isSystem(file)) continue; // skip System folder
                    String name = file.getName();
                    Path newIn = file.toPath();
                    Path newOut = output.resolve(name);
                    map.remove(name);
                    if (file.isDirectory()) {
                        copySync(syn, newIn, newOut);
                    }
                }
            }
        }
        if (Files.exists(input)) {
            String name = input.toFile().getName();
            map.remove(name);
        }
        for (File file : map.values()) {
            copySyncRemove(syn, file.toPath()); // Synchronous
        }
    }

    /**
     * Synchronous copy - 指定されたパスを無条件に削除
     *
     * @param input 　file or folder
     */
    private void copySyncRemove(Set<File> syn, Path input) {
        if (isSystem(input)) return; // skip System folder
        File[] files = input.toFile().listFiles();
        if (null != files) {
            for (File file : files) {
                if (isSystem(file)) continue; // skip System folder
                Path in = file.toPath();
                if (file.isDirectory()) {
                    copySyncRemove(syn, in);
                    safetyRemove(in); // フォルダが空でない場合の対策
                } else {
                    if (safetyRemove(in)) { // フォルダが空でない場合の対策
                        syn.add(file);
                    }
                }
            }
        }
        boolean isFile = !Files.isDirectory(input);
        if (safetyRemove(input) && isFile) { // フォルダが空でない場合の対策
            syn.add(input.toFile());
        }
    }

    /**
     * move the file.
     *
     * @param input  file or folder
     * @param output file or folder
     */
    public int move(String input, String output, Object... x) {
        Initialize(input, x);
        Path in = ana.path;
        Path out = createOutFolder(output);
        isRemove = true; // always delete
        boolean isRecursive = applyOption(noRECURSIVE, false);
        String args = ana.virtualPath + ' ' + output + listOptions();
        messageTitle("move", args);
        if (in.compareTo(out) == 0) {
            inputAndOutputAreSamePath(in);
            return STATUS_ERROR; // in, out are the same
        }
        int rc = checkPath(in, FLAG_ROOT | FLAG_DIRECTORY | FLAG_FILE);
        int ro = checkPath(out, FLAG_ROOT | FLAG_DIRECTORY | FLAG_FILE);
        rc = Math.max(rc, ro);
        if (rc <= STATUS_INFORMATION) {
            Set<File> set = new TreeSet<>();
            moveImpl(set, in, out, isRecursive);
            for (File file : set) {
                printX(_STDOUT, file);
            }
            setDateForChildElement(out); // clean up folder
            messageCYAN("Number of processed", set.size());
        }
        close();
        return rc;
    }

    private void moveImpl(Set<File> set, Path input, Path output,
                          boolean recursive) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            File[] files = input.toFile().listFiles();
            if (null != files) {
                for (File file : files) {
                    if (isSystem(file)) continue; // skip System folder
                    Path newIn = file.toPath();
                    Path newOut = output.resolve(newIn.getFileName());
                    if (file.isDirectory()) {
                        if (recursive) {
                            createDirectory(newIn, newOut); // 日時が設定されない
                            //noinspection ConstantValue
                            moveImpl(set, newIn, newOut, recursive);
                            safetyRemove(newIn); // 子要素がなければ削除
                        }
                    } else {
                        atomicSingleMove(set, newIn, newOut);
                    }
                }
            } else if (Files.exists(input)) {
                long lastMod = input.toFile().lastModified();
                setLastModified(output, lastMod);
            }
        } else {
            atomicSingleMove(set, input, output);
            safetyRemove(input); // 子要素がなければ削除
        }
        safetyRemove(input); // 子要素がなければ削除
    }

    /**
     * 子要素の日付を親フォルダに設定 - Set date for child element
     * <p>
     * move の原因不明の不具合で入力フォルダの日付が取得できないため作成
     */
    private long setDateForChildElement(Path input) {
        if (isSystem(input)) return 0; // skip System folder
        long modTime = 0;
        if (Files.isDirectory(input)) {
            File[] files = input.toFile().listFiles();
            if (null != files) {
                for (File file : files) {
                    if (isSystem(file)) continue; // skip System folder
                    Path path = file.toPath();
                    long time;
                    if (file.isDirectory()) {
                        time = setDateForChildElement(path);
                    } else {
                        time = file.lastModified();
                    }
                    if (modTime < time) modTime = time;
                }
            }
        }
        if (Files.isDirectory(input) && !safetyRemove(input))
            setLastModified(input, modTime); // 空のフォルダを削除
        return modTime;
    }

    /**
     * 単一のファイルを差分コピー (出力側に存在しない、日付時刻・サイズが異なる場合)
     */
    private synchronized void atomicSingleCopy(Set<File> set,
                                               Path input, Path output) {
        try {
            output = isAtomicSameFile(input, output);
            if (output == null) return;

            Files.deleteIfExists(output);
            Files.copy(input, output, COPY_ATTRIBUTES, REPLACE_EXISTING);

            if (Files.exists(output)) {
                set.add(input.toFile());
            }
        } catch (IOException e) {
            throw new RuntimeException(throwMessage("copy", e));
        }
    }

    /**
     * ファイルをターゲット・ファイルに移動するか、そのファイル名を変更
     */
    private synchronized void atomicSingleMove(Set<File> set,
                                               Path input, Path output) {
        try {
            output = isAtomicSameFile(input, output);
            if (output == null) return;

            Files.deleteIfExists(output);
//          Files.move(input, output, ATOMIC_MOVE); // 別ボリュームは NG
            Files.move(input, output, REPLACE_EXISTING);

            if (Files.exists(output)) {
                set.add(input.toFile());
                safetyRemove(input); // ダメ押しで削除
            }

        } catch (IOException e) { // AtomicMoveNotSupportedException
            throw new RuntimeException(throwMessage("move", e));
        }
    }

    // FAT の時間誤差を除外
    private static final long FAT_TIME_ERROR = 2000; // 2sec.

    /**
     * ファイル属性が同一かどうかの判定
     *
     * @param input  パス(ファイル名)
     * @param output パス(ファイル名) or フォルダ
     * @return null 出力パス(ファイル名)、null: 要件に該当しないため skip
     */
    private Path isAtomicSameFile(Path input, Path output) {
        if (!Files.exists(input) || Files.isDirectory(input))
            throw new RuntimeException(throwMessage("File required", input));
        if (!isMatch(input)) return null; // wildcard mismatch
        if (isSystem(input)) return null; // skip System folder
        if (Files.exists(output) &&  // out がフォルダならファイル名を設定
                Files.isDirectory(output)) {
            output = output.resolve(input.getFileName());
        }
        if (input.compareTo(output) == 0) return null; // 同じファイル

        Path inParent = input.getParent();
        if (inParent != null) { // in の親ディレクトリを取得
            Path outParent = output.getParent();
            if (outParent != null) { // out の親ディレクトリを作成
                createDirectory(inParent, outParent);
            }
        }
        if (Files.exists(output) && !Files.isDirectory(output) &&
                input.getFileName().equals(output.getFileName())) {
            File iFile = input.toFile();
            File oFile = output.toFile(); // Check file attributes
            boolean length = iFile.length() == oFile.length();
            long iMod = iFile.lastModified() / FAT_TIME_ERROR;
            long oMod = oFile.lastModified() / FAT_TIME_ERROR;
            if (length && iMod == oMod) { // FAT(誤差2sec) <> NTFS
//                messageGREEN("same file", output);
                return null;
            }
        }
        return output;
    }

    /**
     * 出力側のディレクトリを作成し、日時を設定
     */
    private static void createDirectory(Path input, Path output) {
        try {
            if (Files.isDirectory(input)) {
                FileTime fileTime = Files.getLastModifiedTime(input);

                if (!Files.exists(output))
                    Files.createDirectories(output); // ディレクトリを作成

                Files.setLastModifiedTime(output, fileTime); // 日時を設定
            }
        } catch (IOException e) {
            throw new RuntimeException(throwMessage("Create directory", output));
        }
    }

    /**
     * remove - 指定されたパス以下にあるワイルドカードに一致するファイルを削除する
     * - Delete files matching wildcards under the specified path
     * - Path, Files version
     *
     * @param input Requires at least one sub-path
     * @param x     is remove [ false ]
     */
    public int remove(String input, Object... x) {
        Initialize(input, x);
        Path in = ana.path;
        isRemove = true; // always delete
        boolean isRoot = applyOption(ROOT);
        boolean isRecursive = applyOption(noRECURSIVE, false);
        String args = ana.virtualPath + listOptions();
        messageTitle("remove", args);
        int flags = FLAG_DIRECTORY | FLAG_FILE;
        if (isRoot) flags |= FLAG_ROOT;
        int rc = checkPath(in, flags);
        if (rc <= STATUS_INFORMATION) {
            Set<File> set = new TreeSet<>();
            removeImpl(set, in, isRecursive);
            for (File remove : set) {
                if (!remove.isDirectory())
                    printX(_STDOUT, remove);
            }
            setDateForChildElement(in); // clean up folder
            messageCYAN("Number of processed", set.size());
        }
        close();
        return rc;
    }

    private void removeImpl(Set<File> set, Path input,
                            boolean recursive) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (isSystem(path)) continue; // skip System folder
                    if (Files.isDirectory(path)) {
                        if (recursive) {
                            //noinspection ConstantValue
                            removeImpl(set, path, recursive);
                            if (safetyRemove(path))
                                set.add(path.toFile()); // 子供のいないフォルダ対策
                        }
                    } else {
                        if (isMatch(path) && safetyRemove(path)) {
                            set.add(path.toFile()); // フォルダが空でない場合の対策
                        }
                    }
                }
                safetyRemove(input); // フォルダが空でない場合の対策

            } catch (IOException e) { // DirectoryNotEmptyException
                throw new RuntimeException(throwMessage("remove", e));
            }
        } else if (isMatch(input) && Files.exists(input)) {
            safetyRemove(input);
            set.add(input.toFile());
        }
    }

    /**
     * tree - File version
     *
     * @param input Path to start listing
     */
    public int tree(String input, Object... x) {
        Initialize(input, x);
        Path in = ana.path;
        isRemove = false; // never delete
        boolean isFile = applyOption(FILE);
        boolean isRecursive = applyOption(noRECURSIVE, false);
        String args = ana.virtualPath + listOptions();
        messageTitle("tree", args);
        int rc = checkPath(in, FLAG_ROOT | FLAG_DIRECTORY);
        if (rc <= STATUS_INFORMATION) {
            int max = treeImpl(in.toFile(), "", isFile, isRecursive);
            messageCYAN("MAX_PATH", max + " chars");
        }
        close();
        return rc;
    }

    @SuppressWarnings("ConstantValue")
    private int treeImpl(File input, String indent, boolean isFile, boolean recursive) {
        if (isSystem(input)) return 0; // skip System folder
        int max = 0;
        if (input.isDirectory()) {
            File[] files = input.listFiles();
            if (null != files) {
                TreeSet<File> set = new TreeSet<>(List.of(files));
                for (File file : set) {
                    if (isSystem(file)) continue; // skip System folder
                    if (file.isFile() && isMatch(file)) {
                        if (isFile) {
                            String x = indent + "| " + truncateTree(file);
                            printX(_STDOUT, x);
                        }
                        max = Math.max(max, file.getPath().length());
                    }
                }
                for (File file : set) {
                    if (isSystem(file)) continue; // skip System folder
                    if (file.isDirectory()) {
                        if (isSystem(file)) continue; // skip System folder
                        String x = indent + '/' + truncateTree(file);
                        printX(_STDOUT, x);
                        max = Math.max(max, file.getPath().length());
                        if (recursive) {
                            max = Math.max(max, treeImpl(file,
                                    indent + " ", isFile, recursive));
                        }
                    }
                }
            } else {
                if (isMatch(input)) {
                    if (isFile) {
                        String x = indent + "| " + truncateTree(input);
                        printX(_STDOUT, x);
                    }
                    max = input.getPath().length();
                }
            }
        }
        return max;
    }

    /**
     * ls - list segments
     *
     * @param input Path to start listing
     */
    public int ls(String input, Object... x) {
        Initialize(input, x);
        Path in = ana.path;
        isRemove = false; // never delete
        boolean isPath = applyOption(PATH);
        boolean isSimple = applyOption(SIMPLE);
        boolean isComma = applyOption(COMMA);
        boolean isUnit = applyOption(UNIT);
        boolean is260 = applyOption(MX_260);
        boolean isNo260 = applyOption(MX_no260);
        boolean isRecursive = applyOption(noRECURSIVE, false);
        boolean isAttr = isPath || isSimple || isComma || isUnit;
        if (isPath) {
            resetOptions(SIMPLE, COMMA, UNIT); // オプションの無効化
        }
        String args = ana.virtualPath + listOptions();
        messageTitle("ls", args);
        int rc = checkPath(in, FLAG_ROOT | FLAG_DIRECTORY);
        if (rc <= STATUS_INFORMATION) {
            Set<File> set = new TreeSet<>();
            lsImpl(set, in.toFile(), is260, isNo260, isRecursive);
            for (File file : set) { // パスを切り詰めないで表示
                String ls = lsAttr(file, isAttr);
                printX(_STDOUT, ls);
            }
            messageCYAN("Number of processed", set.size());
        }
        close();
        return rc;
    }

    private void lsImpl(Set<File> set, File input,
                        boolean is260, boolean isNo260, boolean recursive) {
        if (isSystem(input)) return; // skip System folder
        if (input.isDirectory()) {
            File[] files = input.listFiles();
            if (null != files) {
                for (File file : files) {
                    if (isSystem(file)) continue; // skip System folder
                    if (file.isDirectory()) {
                        if (recursive) {
                            //noinspection ConstantValue
                            lsImpl(set, file, is260, isNo260, recursive);
                        }
                    } else {
                        lsSelect260(set, file, is260, isNo260);
                    }
                }
            } else {
                lsSelect260(set, input, is260, isNo260);
            }
        }
    }

    private String lsAttr(File input, boolean isAttr) {
        StringBuilder sb = new StringBuilder(256);
        if (isAttr) {
            SimpleDateFormat sdf = new SimpleDateFormat(
                    "yy/MM/dd HH:mm");
            String daytime = sdf.format(input.lastModified());
            String len = lsLength(
                    getBooOption(PATH) ?
                            input.getPath().length() : // path length
                            input.length()); // file size
            sb.append(daytime).append('\t');
            sb.append(len).append('\t');
        }
        String path = sprintX(_STDOUT, input);
        return sb.append(path).toString();
    }

    /**
     * Output long path
     *
     * @param is260   Output paths longer than 260
     * @param isNo260 Output paths under 260
     */
    private void lsSelect260(Set<File> set, File input,
                             boolean is260, boolean isNo260) {
        if (isMatch(input)) {
            if (is260) {
                if (isLonger260(input)) set.add(input);
            } else if (isNo260) {
                if (!isLonger260(input)) set.add(input);
            } else {
                set.add(input);
            }
        }
    }

    @SuppressWarnings("SpellCheckingInspection")
    private static final String NUMBER_UNIT = " KMGTPEZY";

    private String lsLength(long len) {
        if (getBooOption(PATH)) {
            return String.format("%, 7d", len); // max 32k, 6 char
        }
        if (getBooOption(COMMA)) {
            return String.format("%,d", len);
        }
        if (getBooOption(UNIT)) {
            for (int i = 0; i < NUMBER_UNIT.length(); i++) {
                if (len < 1024) {
                    String unit = Character.toString(NUMBER_UNIT.charAt(i));
                    return String.format("%3d %s", len, unit);
                }
                len /= 1024;
            }
        }
        return Long.toString(len);
    }

    private static final boolean USE_PATH_NAME = true; // Tree でファイル名を表示

    /**
     * 切り詰めたファイル名・IDを返す (returns a truncated path/filename)
     */
    private String truncateTree(File input) {
        StringBuilder sb = new StringBuilder(DISPLAY_WIDTH + 16);
        sb.append(sprintX(_STDOUT, input, USE_PATH_NAME)).append(' ');
        String path = input.getPath();
        String name = input.getName();
        String info = name.length() + "/" + path.length();
        String maxPathID = isLonger260(input) ? " *" : "";
        sb.append(maxPathID.isEmpty() ? info :
                color(MAGENTA, info + maxPathID));
        return sb.toString();
    }

    /**
     * パスを切り詰める
     *
     * @param type [Default value: path value] else, USE_PATH_NAME
     */
    private static String truncatePath(File input, boolean... type) {
        String path = applySlashSeparator(input, type);
        int len = path.length();
        if (DISPLAY_WIDTH < len) {
            int PathUnitHalfSize = DISPLAY_WIDTH / 2;
            return path.substring(0, PathUnitHalfSize) + "…" +
                    path.substring(len - PathUnitHalfSize);
        }
        return path;
    }

    /**
     * パス区切り文字を変換
     *
     * @param type [Default value: path value] else, USE_PATH_NAME
     */
    private static String applySlashSeparator(File input, boolean... type) {
        if (type.length != 0)
            return input.getName();
        String path = input.getPath();
//        messageGREEN("separatorChar", "<" + File.separatorChar + ">");
        return File.separatorChar == '/' ? path :
                path.replace('\\', '/');
    }

    //////////////////////////////////////////////////////////////////////
    // Message
    static String color(String color, String message) {
        return color + message + RESET;
    }

    private static void messageTitle(String name, Object arg) {
        System.out.println(color(YELLOW, name + ' ') + arg);
    }

    private static void messageCYAN(String name, Object arg) {
        System.err.println(color(CYAN, name + ": ") + arg);
    }

    @SuppressWarnings("unused")
    private static void messageGREEN(String name, Object arg) { // debug
        System.err.println(color(GREEN, name + ": ") + arg);
    }

    @SuppressWarnings("SameParameterValue")
    private static void messageMAGENTA(String name, Object arg) {
        System.err.println(color(MAGENTA, name + ": ") + arg);
    }

    private static void messageRED(String name, Object arg) {
        System.err.println(color(RED, name + ": ") + arg);
    }

    private static String throwMessage(String name, Object arg) {
        return color(RED, name + ": ") + arg;
    }

    // ルートは指定できません
    private static void rootCannotBeSpecified(Path path) {
        messageRED("Root folder cannot be specified",
                applySlashSeparator(path.toFile()));
    }

    // 入力と出力は同じファイル
    private static void inputAndOutputAreSamePath(Path path) {
        messageRED("Input and output are the same path",
                applySlashSeparator(path.toFile()));
    }

    // パスが存在しません
    private static void noSubfolderExist(Path path) {
        messageMAGENTA("Path does not defined",
                applySlashSeparator(path.toFile()));
    }

    private static final String RESET = "\033[m";       // \033[0m
    private static final String RED = "\033[91m";       // error
    private static final String GREEN = "\033[92m";     // debug
    private static final String YELLOW = "\033[93m";    // title
    private static final String BLUE = "\033[94m";              // options
    private static final String MAGENTA = "\033[95m";   // warning
    private static final String CYAN = "\033[96m";      // information

    //////////////////////////////////////////////////////////////////////
    // Analysis
    static class Analysis {
        private static final Pattern SPLIT_PATH = IS_PATH_HAS_FILE;
        //                Pattern.compile("(.*[/\\\\])*([^/\\\\]*)$");
        private static final Pattern WILD_CARD_ALL = Pattern
                .compile(".*");
        final String virtualPath; // path / WildCard
        final Path path; // ワイルドカードを除いた実際のパス
        final Pattern regex; // ワイルドカード
        final boolean alwaysTrue; // ワイルドカードが常に真かどうか

        Analysis(String input) {
            String path = input.trim();
            String wild = "";
            Pattern regex = WILD_CARD_ALL;
            Matcher m = SPLIT_PATH.matcher(path);
            if (m.matches()) {
                String g2 = getValue(m.group(2).trim());
                if (hasWildcard(g2)) {
                    path = getValue(m.group(1)).trim();
                    wild = g2;
                    regex = mkWildcard(g2);
                }
            }
            if (path.isEmpty()) path = "./";
            this.path = Path.of(path); // .normalize();
            this.virtualPath = path +
                    (wild.isEmpty() ? "" : NanoTools.color(NanoTools.BLUE, wild));
            this.regex = regex;
            alwaysTrue = WILD_CARD_ALL.equals(regex);
        }

        /**
         * hasWildcard - ワイルドカード指定かどうかを返す
         * - wildcard: * ? | endsWith .
         */
        private static boolean hasWildcard(String wild) {
            for (int i = 0; i < wild.length(); i++) {
                char c = wild.charAt(i);
                if (0 <= "*?|".indexOf(c)) return true;
            }
            return wild.endsWith(".");
        }

        /**
         * mkWildcard - 大小文字を区別せずにマッチする (Case-sensitive)
         * ワイルドカード (* ? .) 以外の記号は、正規表現として使用できる
         * e.g. /[fb]*.txt|*.log/
         *
         * @param wild wildcard
         */
        private static Pattern mkWildcard(String wild) {
            StringBuilder sb = new StringBuilder(128);
            if (wild.endsWith("."))
                wild = wild.substring(0, wild.length() - 1); // Delete the last .
            if (wild.isEmpty()) wild = "*"; // Default value
            for (int i = 0; i < wild.length(); i++) {
                char c = wild.charAt(i);
                switch (c) {
                    case '*' -> sb.append(".*");
                    case '?' -> sb.append('.');
                    case '.' -> sb.append("\\.");
                    default -> sb.append(c);
                }
            }
            return Pattern.compile(sb.toString(), Pattern.CASE_INSENSITIVE);
        }

    }

    //////////////////////////////////////////////////////////////////////
    // Redirect
    record Redirect(boolean defined, String name,
                    String rid, String file, boolean isErr) {
    }

    private static String getValue(String x) {
        return x == null ? "" : x;
    }
}