/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.runtime;

import plus.reflect.Cache;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Regular Expression (singleton).
 * <p>
 * The class to which this annotation is applied is immutable.
 *
 * @author kunio himei.
 */
public final class RegExp {

    //* 数値: FALSE(0).
    private static final int C_INTEGER0 = 0;
    //* 数値: TRUE(1).
    private static final int C_INTEGER1 = 1;
    //* '.'は行末記号を含む任意の文字にマッチする.
    private static final int REGX_DOTALL_FLAG = Pattern.DOTALL;

    /*
     * Simple decode String.
     */
    private static String simpleDecode(String s) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < s.length(); ) {
            char c = s.charAt(i++);
            if ('\\' == c) {
                c = s.charAt(i++);
                if ('t' == c)
                    sb.append('\t');
                else if ('f' == c)
                    sb.append('\f');
                else if ('r' == c)
                    sb.append('\r');
                else if ('n' == c)
                    sb.append('\n');
                else
                    sb.append('\\').append(c);
            } else
                sb.append(c);
        }
        return sb.toString();
    }

    //* (?idmsux-idmsux) 正規表現ではないが、マッチフラグ i、d、m、s、u、x のオン/オフを切り替える.
    //* "(?i)" CASE_INSENSITIVE
    //* 12   &    3 前方参照
    //* ((\*)&) | (\+[1-9])
    /**
     * 置換指示文字を解析する正規表現 /((\*)&)|(\+[1-9])/.
     */
    private static final Pattern RX_REPLACE =
            Pattern.compile("((\\\\*)&)|([\\\\\1-9])");

    /**
     * 正規表現 r でターゲット文字列 t を s で一括置換し、
     * で置換結果(RSTRING)をセットして、置換数を返す.
     * gsub(r, s [,t])
     */
    public static int gsub(Object... args) {
        String r = objectsToString(args[0], "");
        String s = objectsToString(args[1], "");
        String t = (3 > args.length) ?
                BuiltInVar.$.getAt(0).toString() :
                objectsToString(args[2], "");
        r = simpleDecode(r);
        s = simpleDecode(s);
        t = simpleDecode(t);
        boolean hasAmp = (0 <= s.indexOf('&')) || (0 <= s.indexOf('\\'));
        String ss = (0 <= s.indexOf('$')) ? s.replace("$", "\\$") : s;
        StringBuilder sb = new StringBuilder();
        Matcher m = compile(r).matcher(objectsToString(t, ""));
        int cc = 0; // replacement has '&' ?
        for (; m.find(); cc++) {
            m.appendReplacement(sb, (hasAmp) ? subreplace(m.group(), ss) : ss);
        }
        String tt = m.appendTail(sb).toString();
        if ((0 < cc) && (3 > args.length)) { // ターゲット変数が省略されている場合.
            BuiltInVar.$.putAt(0, tt);
            BuiltInVar.NF.intValue(); // bug! AWKplus.wc.awk
        }
        BuiltInVar.RESULT.put(tt);
        return cc;
    }

    /**
     * 正規表現 r でターゲット文字列 t を s で置換し、
     * で置換結果(RSTRING)をセットして、置換数を返す.
     * sub(r, s [,t])
     */
    public static int sub(Object... args) {
        String r = objectsToString(args[0], "");
        String s = objectsToString(args[1], "");
        String t = (3 > args.length) ?
                BuiltInVar.$.getAt(0).toString() :
                objectsToString(args[2], "");
        r = simpleDecode(r);
        s = simpleDecode(s);
        t = simpleDecode(t);
        boolean hasAmp = (0 <= s.indexOf('&')) || (0 <= s.indexOf('\\'));
        String ss = (0 <= s.indexOf('$')) ? s.replace("$", "\\$") : s;
        int cc = C_INTEGER0;
        Matcher m = compile(r).matcher(t);
        if (m.find()) {
            t = m.replaceFirst((hasAmp) ? subreplace(m.group(), ss) : ss);
            cc = C_INTEGER1;
        }
        if ((0 < cc) && (3 > args.length)) { // ターゲット変数が省略されている場合.
            BuiltInVar.$.putAt(0, t);
            BuiltInVar.NF.intValue(); // bug! AWKplus.wc.awk
        }
        BuiltInVar.RESULT.put(t);
        return cc;
    }

    /**
     * s が、 r に適合する位置を、また適合しないときは、0 を返し、
     * RSTART と RLENGTH をセットする.
     */
    public static int match(Object s, Object r) {
        String ss = objectsToString(s, "");
        String rr = objectsToString(r, "");
        ss = simpleDecode(ss);
        rr = simpleDecode(rr);
        Matcher m = plus.runtime.RegExp.compile(rr).matcher(ss);
        int start = 0, length = 0;
        if (m.find()) {
            start = 1 + m.start();
            length = m.end() - m.start();
        }
        BuiltInVar.RSTART.put(start);
        BuiltInVar.RLENGTH.put(length);
        return start;
    }

    /**
     * 置換対象文字列 s 中の '&' を解析する.
     * <p>
     * <li>'\[1-9]' 前方参照で置換.
     * <li>'&' 適合文字列で置換.<br>
     * POSIX 2001.
     * <table border="1">
     * <tr>
     * <th>source</th>
     * <td>&</td>
     * <td>\&</td>
     * <td>\\&</td>
     * <td>\\\&</td>
     * <td>\\\\&</td>
     * </tr>
     * <tr>
     * <th>internal</th>
     * <td>&</td>
     * <td>&</td>
     * <td>\&</td>
     * <td>\&</td>
     * <td>\\&</td>
     * </tr>
     * <tr>
     * <th>output</th>
     * <td>r</td>
     * <td>r</td>
     * <td>'&'</td>
     * <td>'&'</td>
     * <td>'\'r</td>
     * </tr>
     * </table>
     */
    private static String subreplace(CharSequence mat, String pat) {
        StringBuilder sb = new StringBuilder();
        Matcher m = RX_REPLACE.matcher(pat);
        int i;
        for (i = 0; m.find(i); i = m.end()) {
            String s = pat.substring(i, m.start());
            String x = (0 > s.indexOf('\\')) ? s : s
                    .replace("\\", "\\\\");
            sb.append(x); // 開始文字列を複写
            // 12   &    3 前方参照
            // ((\*)&) | (\+[1-9])
            if (null != m.group(1)) {
                if (0 == ((m.end() - m.start()) & 1)) { // %2
                    sb.append(m.group()); // [\]*&
                } else {
                    sb.append(m.group(2)).append(mat); // [\]*
                }
            } else { // 前方参照はエスケープしない(有効)
                sb.append(m.group(3)); // [\]+[1-9]
            }
        }
        if (i < pat.length()) {
            String s = pat.substring(i);
            String x = (0 > s.indexOf('\\')) ? s : s
                    .replace("\\", "\\\\");
            sb.append(x); // 残り文字列を複写
        }
        return sb.toString();
    }

    /**
     * コンパイル済みの正規表現を返す.
     */
    public static Pattern compile(String regex, int flags) {
        int f = flags | REGX_DOTALL_FLAG;
        String key = f + "," + regex;
        Pattern x = Cache.REGEXP.get(key);
        if (null == x) {
            x = Pattern.compile(regex, f);
            Cache.REGEXP.put(key, x);
        }
        return x;
    }

    /**
     * コンパイル済みの正規表現を返す (USE; IGNORECASE).
     */
    public static Pattern compile(String regex) {
        int ignoreCase = BuiltInVar.IGNORECASE.intValue(); // 英大小文字の区別指示子
        int flag = (0 == ignoreCase) ? 0 : Pattern.CASE_INSENSITIVE;
        return compile(regex, flag);
    }

    /*
     * Objects.toString(Object o, String nullDefault)
     */
    @SuppressWarnings("SameParameterValue")
    private static String objectsToString(Object o, String nullDefault) {
        return (o != null) ? o.toString() : nullDefault;
    }
}