/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

import java.util.HashMap;
import java.util.Map;

/**
 * Lexical Analyzer - Keywords.
 *
 * @author kunio himei.
 */
public enum Keyword {

    SymBEGIN("BEGIN", true),
    SymBREAK("break", true),
    SymCATCH("catch", true),
    SymCONTINUE("continue", true),
    SymDELETE("delete", true),
    SymDO("do", true),
    SymELSE("else", true),
    SymEND("END", true),
    SymEXIT("exit", true),
    SymFINALLY("finally", true),
    SymFOR("for", true),
    SymFUNCTION("function", true),
    SymGETLINE("getline", true),
    SymHAS("has", true),
    SymIF("if", true),
    SymIMPORT("import", true),
    SymINVOKE("invoke", true),
    SymNEXT("next", true),
    SymNEXTFILE("nextfile", true),
    SymPACKAGE("package", true),
    SymPRINT("print", true),
    SymPRINTF("printf", true),
    SymRETURN("return", true),
    SymTHROW("throw", true),
    SymTRY("try", true),
    SymVAL("val", true),
    SymVAR("var", true),
    SymWHILE("while", true),
    //* ここから下は、Symbolでない.
    SyyAnnotation("Annotation", false), // Annotation.
    SyyARRAY("ARRAY", false),
    SyyASSIGN("ASSIGN", false), // AssignHelper.
    SyyBDO("DO", false), // REMIND break があるとき.
    SyyBFOR("FOR", false), // REMIND break があるとき.
    SyyBWHILE("WHILE", false), // REMIND break があるとき.
    SyyCALL("CALL", false),
    SyyFN("Func", false), // Function.
    SyyFNI("FnI", false), // Function Information.
    SyyFNP("FnP", false), // Function Parameter.
    SyyFVal("FVAL", false), // Function Value.
    SyyFValFuture("FVALATOM", false), // Atomic Function Value.
    SyyNAME("NAME", false), // Variable Name.
    SyyNEW("NEW", false),
    SyyQIF("QIF", false), // question if.
    SyySIMPLE("SIMPLE", false); //  Simple Statement.
    /**
     * オブジェクトのリバース配列.
     */
    private static final Map<String, Keyword> CREVERCE_MAP = new HashMap<>();

    /* オブジェクトのリバース配列を構築. */
    static {
        for (Keyword x : Keyword.values()) {
            if (x.isSymbol) {
                CREVERCE_MAP.put(x.id, x);
            }
        }
    }

    private final String id; // オブジェクトの識別名.

    private final boolean isSymbol; // このオブジェクトが Symbol かどうか.

    /**
     * このコンテナを構築する.
     */
    Keyword(String x, boolean b) {
        this.id = x;
        this.isSymbol = b;
    }

    /**
     * 指定された識別名に対応するオブジェクトを返す.
     * <p>
     * {@link Enum#valueOf(Class, String)} と同一機能であるが Throw しない.
     */
    public static Keyword apply(String x) {
        return CREVERCE_MAP.get(x);
    }

    /**
     * 指定された識別名がコンテナーに存在するかどうかを返す.
     */
    public static boolean contains(String x) {
        return CREVERCE_MAP.containsKey(x);
    }

    /**
     * このオブジェクトの識別名を返す.
     */
    public static String toName(Keyword x) {
        return x.id;
    }

    /**
     * このオブジェクトのシンボル名を返す.
     */
    @Override
    public final String toString() {
        return '\'' + this.id;
    }
}