/*
 * Copyright (C) 2011 awk4j - http://awk4j.sourceforge.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.io;

import java.io.Closeable;
import java.io.IOException;
import java.net.ServerSocket;
import java.net.SocketAddress;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;

/**
 * [%inet%] サーバソケットの実装 (マルチスレッド対応).
 *
 * @author kunio himei.
 */
final class InetServerSocket implements Closeable, Connectable {

    /**
     * データグラム型ソケットの選択可能なチャネル.
     */
    private final ServerSocketChannel channel;

    /**
     * サーバソケット.
     */
    private final ServerSocket serverSocket;

    /**
     * 接続待ち時間.
     */
    private final int timeout;

    /**
     * サーバソケットの構築.
     *
     * @param address ソケットアドレス
     * @param tout    接続待ち時間
     */
    InetServerSocket(final SocketAddress address, final int tout)
            throws IOException {
        this.channel = ServerSocketChannel.open();
        this.serverSocket = this.channel.socket();
        this.serverSocket.bind(address);
        this.timeout = tout;
    }

    /**
     * サーバソケットを閉じる.
     */
    @Override
    public void close() throws IOException {
        // new Exceptions("InetServerSocket#close").printStackTrace();
        try {
            if (!this.serverSocket.isClosed()) {
                this.serverSocket.close();
            }
        } finally {
            if (this.channel.isOpen()) {
                this.channel.close();
            }
        }
    }

    /**
     * 接続されたソケットインターフェイスを返す.
     */
    @Override
    public ReadWriteable connect() throws IOException {
        // クライアントからの接続がくるまでここでブロックする
        // System.err.println("InetServerSocket#connect");
        this.serverSocket.setSoTimeout(this.timeout); // 接続待ち時間
        final SocketChannel sock = this.serverSocket.accept().getChannel();
        return new InetSocket(sock, this.timeout);
    }
}