/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

import plus.util.Escape;

import java.util.regex.Pattern;

/**
 * Lexical Analyzer - Term.
 *
 * @author kunio himei.
 */
public final class Term {

    /**
     * Constant Number 0.
     */
    public static final NUMBER NUMBER_ZERO = new NUMBER("0", 0);

    /**
     * Don't let anyone instantiate this class.
     */
    private Term() {
        super();
    }

    /**
     * '.' を含む変数名 または、引用符""で囲まない文字列.
     */
    public static class BOXING extends YyValue {

        public BOXING(String x) {
            super(x, Flags.T11ANY, x);
        }

        //* このオブジェクトの文字列表現を返す.
        @Override
        public final String toString() {
            return super.id;
        }
    }

    /**
     * 数値.
     */
    public static class NUMBER extends YyValue {

        //* 整数を表現する正規表現.
        private static final Pattern RX_INTEGER_NUMBER =
                Pattern.compile("^[-+]?\\d+$");

        /*
         * id: 正規化した文字列.
         * value: 数値 (double).
         */
        public NUMBER(String x, Number e) {
            super(x, getType(x, e), e);
        }

        //* 数値オブジェクトのタイプ属性を返す.
        private static int getType(String x, Number e) {
            if (!RX_INTEGER_NUMBER.matcher(x).matches()) {
                return Flags.T06DBL;
            } else if (0 == (e.doubleValue() - e.intValue())) {
                return Flags.T03INT;
            } else if (0 == (e.doubleValue() - e.longValue())) {
                return Flags.T04LONG;
            } else {
                return Flags.T06DBL;
            }
        }

        //* このオブジェクトの double表現を返す.
        public final double doubleValue() { //
            return ((Number) super.value).doubleValue();
        }

        //* このオブジェクトの文字列表現を返す.
        @Override
        public final String toString() {
            return super.id;
        }
    }

    /**
     * 正規表現.
     */
    public static class REGEXP extends YyValue {

        /*
         * id: オリジナル値 (両端の'/'は、含まない)
         * value: ダブルクォート文字列 - for Interpreter.
         */
        public REGEXP(String x) {
            super(x, Flags.T08REGX,
                    Escape.encodeString(x, "\"")); // ダブルクォート文字列化.
        }

        //* このオブジェクトの文字列表現(/.../)を返す - for Compiler.
        @Override
        public final String toString() { //
            return "/" + Escape.nopFilter(super.id) + "/"; // /nop/
        }
    }

    /**
     * 文字列.
     */
    public static class STRING extends YyValue {

        /**
         * id: オリジナル値 (両端の引用符は、含まない)
         * value: オリジナル値. - for Interpreter.
         */
        public STRING(String x) {
            super(x, Flags.T07STR, x);
        }

        //* シングルクォート文字列を返す - for Compiler.
        @Override
        public final String toString() { //
            return "'" + Escape.encodeString(super.id, "\\'") + "'"; // '\..'
        }
    }

    /**
     * 値ノード.
     */
    public static class YyValue {

        public final String id; // このオブジェクトの識別値.
        public final int nType; // このオブジェクトの属性値.
        public final Object value; // このオブジェクトの値.

        /**
         * このオブジェクトを構築する.
         */
        YyValue(String x, int typ, Object val) {
            this.id = x;
            this.nType = typ;
            this.value = val;
        }
    }
}