/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

import java.util.Objects;

/**
 * Type Helper.
 *
 * @author kunio himei.
 */
abstract class TypeHelper extends Lex {

    /**
     * 配列参照に伴うタイプ設定 in, NAME[].
     */
    static Node.YyVariable arrayType(Node.Arr e) {
        return updateType(e, Flags.T11ANY, Flags.T16ARRAY);
    }

    /**
     * 属性をマージして返す.
     */
    static int checkType(int t1, int t2) {
        int m1 = t1 & Flags.T12X15M;
        int m2 = t2 & Flags.T12X15M;
        if ((Flags.T26NIL != t1) && (Flags.T26NIL != t2) && (m1 != m2)) {
            if (Flags.T11ANY == m1) {
                return t1;
            } else if (Flags.T11ANY == m2) {
                return t2;
            } else if (Flags.isStrNum(t1)) {
                return t1;
            } else if (Flags.isStrNum(t2)) {
                return t2;
            } else if (((Flags.T01NUM == m1) || (Flags.T07STR == m1))
                    && ((Flags.T01NUM == m2) || (Flags.T07STR == m2))) {
                return t1 | Flags.T09STNUM; // 数値と文字列が混在した場合
            }
        }
        return (Flags.T26NIL == t1) ? t2 : t1;
    }

    /**
     * マスクした属性(数値, 文字列など)を返す.
     */
    static int getMaskType(Object e) {
        return Type.getNodeType(e) & Flags.T12X15M;
    }

    /**
     * 代入に伴うタイプ設定 - 変数への数値代入有り.
     */
    static Node.YyVariable incdecNegateType(Node.YyVariable e) {
        return updateType(e, Flags.T01NUM, Flags.T20ASSIGN);
    }

    /**
     * タイプ属性の更新 ++, --, in, NAME[], delete, getline var, split(_,arr,_).
     */
    static Node.YyVariable updateType(Node.YyVariable e,
                                      int defaults, int orType) {
        int t1 = Symbols.getType(e.name);
        int t2 = (Flags.T26NIL == t1) ? defaults : t1;
        if (Flags.T26NIL != t2) {
            int w = orType;
            if (!Type.isEmpty(e.index)) {
                w |= Flags.T16ARRAY;
            }
            Symbols.setType(e.name, t2, w);
        }
        return e;
    }

    /**
     * 代入に伴うタイプ設定 return, assign.
     */
    Object assignType(String cid, String name,
                      Object e, int optTyp) {
        int w = Type.getNodeType(e);
        int t1 = ((Flags.T25UNDEFIN == w) ? Flags.T11ANY : w)
                & ~Flags.T16ARRAY;
        int t2 = Symbols.setType(name, t1, Flags.T20ASSIGN | optTyp);
        int m1 = t1 & Flags.T12X15M;
        int m2 = t2 & Flags.T12X15M;
        if ((m1 != m2) && Flags.isAnyType(t2) && "RETURN".equals(cid)) {
            // return で異なる復帰値タイプが指定された場合
            int t3 = checkType(t2, t1);
            if (t2 != t3) {
                Symbols.resetFunctionType(name);
                Symbols.setType(name, t3, Flags.T20ASSIGN);
            }
        }
        if (Flags.isAnyType(t1) && !Flags.isVoid(t2) && (0 == (m1 & m2))) { // 代入可能か? (Anyの代入はチェックしない)
            if (Flags.T01NUM >= m2) { // 数値正規化が必要
                yyWARNING(cid + " ? " + name + " <" + Integer.toHexString(m2)
                        + "> : " + e + " <" + Integer.toHexString(m1) + '>');
                return new Node.IncDec(Node.NUMBER_OP, e);
            }
            return e;
        }
        return e;
    }

    /**
     * * オプションの変数属性を返す.
     */
    T4Types optType(String name, boolean wantColon) {
        if (!":".equals(super.tok)
                || (wantColon && !super.yyText.startsWith(":"))) {
            return new T4Types(Flags.T26NIL, "");
        }
        while (":".equals(super.tok)) {
            eat(super.tok);
        }
        nl();
        int typ;
        String sType;
        if (super.tok instanceof Node.NAME x) { // 文字列でタイプ指定
            if (TypeForGroovy.hasVarType(x.name)) {
                typ = TypeForGroovy.getVarType(x.name);
            } else {
                typ = Flags.T13OBJ;
            }
            sType = x.name;
        } else if (super.tok instanceof Term.BOXING x) {
            typ = Flags.T13OBJ;
            sType = x.id;
        } else {
            // i%3?i%5?i: X :"oops"
            // i%3?(i%5?i:X):"OK"
            throw new IllegalStateException(String.valueOf(super.tok)); // BUG
        }
        advance();
        if ("[".equals(super.tok)) {
            sType += subType(); // サブタイプ
        }
        while ("*".equals(super.tok)) {
            typ |= Flags.T15VARG; // 可変長引数
            advance();
        }
        return (0 > name.indexOf('.')) ? // Type宣言の初期登録
                new T4Types(Symbols.setType(name, typ, 0), sType) :
                new T4Types(typ, sType);
    }

    /**
     * サブタイプを返す.
     */
    private String subType() {
        LexArray<String> buf = new LexArray<>();
        eat("[");
        while (!"]".equals(super.tok)) {
            buf.add(Objects.toString(super.tok, ""));
            eat(super.tok);
        }
        eat("]");
        return buf.mkString("[", ",", "]");
    }

    /**
     * タプル戻り値.
     */
    static class T4Types { // TODO

        public final int nType;
        public final String sType;

        T4Types(int nType, String sType) {
            this.nType = nType;
            this.sType = sType;
        }
    }
}