/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.gen

import groovy.transform.CompileStatic
import plus.BiIO
import plus.lex.*
import plus.runtime.BuiltInVar

import java.util.regex.Pattern

/**
 * Code Generator.
 *
 * @author kunio himei.
 */
@CompileStatic
abstract class GenBase extends BiIO {

    abstract Object eval(Object e)

    //* 文字列(single quotes)を表わす正規表現. '...', "..."
    private static final Pattern rxQuoteString = Pattern.compile(
            /^('([^'\\]*(?:\\.[^'\\]*)*)')|("([^"\\]*(?:\\.[^"\\]*)*)")$/)

    static Parser.SyntaxTree sTree
    static Stack currentFunc = new Stack()  // 現在処理中の関数
    static Map<String, Integer> globalDupMap = [:] // 2重登録チェック
    static List<String> globalBuf = [] // グローバルバッファ
    static List<String> codeBuf = [] // コード生成バッファ
    static Map<String, Integer[]> callMap = [:] // 定義した関数を呼び出した
    static Map<String, Integer> excludeMap = [:] // 関数生成除外フラグ
    static Map<String, Node.Func> functionMap = [:] // ユーザ定義関数

    static final int NIL = 0 // REMIND グローバル変数の初期値.
    static int yyIndent // 字下げ
    static int scriptLineNumber // 行番号
    static int varSequence // 一時変数の連番
    static String finalValDef = '' // Val変数は、final属性を持つか？
    static boolean isVarValDef // var,val定義中か？
    static boolean isGlobalDef // グローバル定義中か？

    /** 連続代入連鎖マップ - a = b = value -> a = value; b = value
     */
    static Map<String, String> chainMap = [:]

    //* 組み込み関数エイリアスの判定
    private static Map<String, Integer> aliasMap =
            ['double': 0, 'float': 0, 'long': 0, 'int': 0,
             'printf': 0, 'sprintf': 0]

    //* 組み込み関数のエイリアスを返す
    static String builtInFuncName(String name) {
        return aliasMap.containsKey(name) ? '_' + name : name
    }

    //* 変数名のエイリアスを返す.(組込み変数の名前を、groovy仕様に変換 $->_$)
    static String getAlias(String name) {
        String alias = BuiltInVar.getAlias(name)
        if (alias != name) {
            // REMIND エイリアス'$'をここで外す.
            alias = alias.replaceFirst(/^[$]/, '')
            Symbols.markGlobal(alias, Flags.T11ANY) // 'def'の付加判定に使用する
        }
        return alias
    }

    //* 正式な代入演算子を返す
    static String getOperator(String op) {
        if (2 <= op.length()) return op
        return ('=' == op) ? op : op + '='
    }

    //* 代入文で使用する、属性、'def' または、''を返す
    static String getAttribute(String name, String sType) {
        return !isVarValDef && isDefine(name) ? '' : // 定義済み
                exists(sType) ? sType + ' ' : 'def '
    }

    //* 変数は、定義済みか？
    static boolean isDefine(String name) {
        return isGlobal(name) || Symbols.isLocal(name)
    }

    //* Global定義済みか？
    static boolean isGlobal(String name) {
        return globalDupMap.containsKey(name)
                || Symbols.isGlobal(name)
                || exists(BuiltInVar.forName(name))
    }

    //* 文字リテラル - NOTE GStringを無効にするため、"'"を使用する.
    static String addQuote(Object e) {
        String xx = e
        if (isNil(e)) return "''"
        if (e instanceof Term.YyValue) // 数値 文字列, 正規表現, BOXING
            return (e as Term.YyValue).toString()
        if (rxQuoteString.matcher(xx).find()) // "..." or '...'
            return xx
        return "'" + xx + "'"
    }

    // ------------------------------------------------------------------------
    // コード生成
    // ------------------------------------------------------------------------
    static void gen(Object x) {
        String w = x as String
        if ('' != w) gen(scriptLineNumber, w)
    }

    //* groval変数は、genGlobalで出力する
    static void gen(int lno, String x) {
        if (isGlobalDef) {
            String name = x.replaceAll(
                    /^\s*(final )?(\w+ )|( = ).+$/, '')
            genGlobal(name, x)
        } else {
            codeBuf.add(('/*' + lno + '*/\t' + genIndent(yyIndent) + x).trim())
        }
    }

    static void genTop(String x) {
        globalBuf.add('/*' + scriptLineNumber + ('*/\t' + genIndent(0) + x).trim())
    }

    static def genGlobal(String name, String x) {
        if (!globalDupMap.containsKey(name)) { // 2重登録チェック
            globalBuf.add(('/*' + scriptLineNumber + '*/\t' + x).trim())
            globalDupMap.put(name, 1)
        }
    }

    //* コード生成(字下げ)
    static String genIndent(int n) {
        StringBuilder sb = new StringBuilder() // 0, 4, 6, 8, ...
        if (0 < n) {
            for (int i = n; 0 <= i; i--)
                sb.append('  ')
        }
        return sb.toString()
    }

    // ------------------------------------------------------------------------
    // Utils.
    // ------------------------------------------------------------------------
    static String mkStringNil(Object[] arr, String a, String b, String c) {
        if (isNil(arr)) return ''
        return mkString(arr, a, b, c)
    }

    static String mkString(Object[] arr, String a, String b, String c) {
        if (isNil(arr)) return a + c
        StringBuilder sb = new StringBuilder(a)
        for (x in arr)
            sb.append(x).append(b)
        sb.setLength(sb.length() - b.length())
        return sb.append(c).toString()
    }

    /**
     * 必要なら、()を付加する.
     */
    static String paren(String paren) {
        String x = paren // 文字列を削除する.
                .replaceAll(/'([^'\\]*(?:\\.[^'\\]*)*)'/, '')
                .replaceAll(/"([^"\\]*(?:\\.[^"\\]*)*)"/, '')
                .replaceAll(/\/([^\/\\]*(?:\\.[^\/\\]*)*)\//, '')
                .replaceAll(/[^()]+/, '')
        int count = 0, i = 1
        for (; x.length() > i; i++) { // [(]...) 2番目からで初めて、最後に')'が残れば、OK.
            char c = x.charAt(i)
            if ('(' as char == c) count++
            else if (')' as char == c) count--
            if (0 > count) break
        }
        return ((i == x.length() - 1) &&
                paren.matches(/^[(].*[)]$/)) ? // (...)
                paren : '(' + paren + ')'
    }

    // ------------------------------------------------------------------------
    // Message.
    // ------------------------------------------------------------------------
    @SuppressWarnings("unused")
    static void message(int level, Object name, Object... args) {
        if (messageLevel <= level)
            System.err.println("${name} ${args.toString()}")
    }
    static int messageLevel = 1
}