/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;

/**
 * Lexical Analyzer - Operators (演算子).
 *
 * @author kunio himei.
 */
public final class Operator {

    /**
     * NAME as クラス名 - 強制型変換.
     */
    public static final String YY_AS = "as";
    /**
     * NAME is クラス名 - 型判定.
     */
    public static final String YY_IS = "is";
    /**
     * INDEX in NAME - 配列要素の存在検査.
     */
    public static final String YY_OPIN = "in";
    /**
     * 読み替えテーブル.
     */
    private static final Map<String, String> CHANGE_OPERATOR = new HashMap<>();
    /**
     * 演算子(予約語でない).
     */
    private static final Collection<String> OPERATORS = new HashSet<>();

    /* 演算子(予約語でない). */
    static {
        OPERATORS.add(YY_AS); // 強制型変換
        OPERATORS.add(YY_IS); // is (instanceof)
        OPERATORS.add(YY_OPIN); // in
    }

    //* 読み替えテーブル初期化.
    static {
        CHANGE_OPERATOR.put("<>", "!="); // REMIND 読み替えテーブル ☆
    }

    /**
     * Don't let anyone instantiate this class.
     */
    private Operator() {
        super();
    }

    /**
     * 読み替えテーブルに存在すれば、読み替えたトークンを返す.
     */
    public static String changeOperator(String x) {
        String s = CHANGE_OPERATOR.get(x);
        return (null == s) ? x : s;
    }

    /**
     * 演算子かどうかを返す.
     */
    public static boolean isOperator(String x) {
        return OPERATORS.contains(x);
    }
}