"""Define the unit tests for the :mod:`colour.colorimetry.spectrum` module."""

from __future__ import annotations

import pickle

import numpy as np
import pytest

import colour
from colour.algebra import CubicSplineInterpolator
from colour.colorimetry.spectrum import (
    SPECTRAL_SHAPE_DEFAULT,
    MultiSpectralDistributions,
    SpectralDistribution,
    SpectralShape,
    reshape_msds,
    reshape_sd,
    sds_and_msds_to_msds,
    sds_and_msds_to_sds,
)
from colour.constants import TOLERANCE_ABSOLUTE_TESTS
from colour.utilities import is_caching_enabled, is_scipy_installed, tstack

__author__ = "Colour Developers"
__copyright__ = "Copyright 2013 Colour Developers"
__license__ = "BSD-3-Clause - https://opensource.org/licenses/BSD-3-Clause"
__maintainer__ = "Colour Developers"
__email__ = "colour-developers@colour-science.org"
__status__ = "Production"

__all__ = [
    "DATA_SAMPLE",
    "DATA_SAMPLE_NON_UNIFORM",
    "DATA_SAMPLE_INTERPOLATED",
    "DATA_SAMPLE_INTERPOLATED_NON_UNIFORM",
    "DATA_SAMPLE_NORMALISED",
    "DATA_STANDARD_OBSERVER_2_DEGREE_CIE1931",
    "DATA_CMFS",
    "DATA_SAMPLE_ABRIDGED",
    "DATA_MULTI_SAMPLE_ABRIDGED",
    "TestSpectralShape",
    "TestSpectralDistribution",
    "TestMultiSpectralDistributions",
    "TestReshapeSd",
    "TestSdsAndMdsToSds",
    "TestSdsAndMsdsToMsds",
]

DATA_SAMPLE: dict = {
    340: 0.0000,
    360: 0.0000,
    380: 0.0000,
    400: 0.0641,
    420: 0.0645,
    440: 0.0562,
    460: 0.0537,
    480: 0.0559,
    500: 0.0651,
    520: 0.0705,
    540: 0.0772,
    560: 0.0870,
    580: 0.1128,
    600: 0.1360,
    620: 0.1511,
    640: 0.1688,
    660: 0.1996,
    680: 0.2397,
    700: 0.2852,
    720: 0.0000,
    740: 0.0000,
    760: 0.0000,
    780: 0.0000,
    800: 0.0000,
    820: 0.0000,
}

DATA_SAMPLE_NON_UNIFORM: dict = {
    391.898: 16.331740,
    392.069: 16.333122,
    405.606: 40.197224,
    406.794: 39.923366,
    406.891: 39.924098,
    407.026: 39.925138,
    416.286: 40.064293,
    418.690: 40.009950,
    426.700: 18.045809,
    426.726: 18.045986,
    432.556: 38.435883,
    464.742: 29.534647,
    465.025: 29.534647,
    465.147: 29.534647,
    466.586: 38.226470,
    477.175: 7.487795,
    493.205: 7.684766,
    505.217: 7.684766,
    513.294: 20.701285,
    513.328: 20.704211,
    514.349: 20.704211,
    514.516: 20.709788,
    515.109: 20.709788,
    538.034: 7.684766,
    564.807: 20.704211,
    566.247: 20.709788,
    569.592: 32.103387,
    580.133: 37.548490,
    581.198: 37.548490,
    582.642: 40.197224,
    588.977: 18.045986,
    589.159: 18.045809,
    600.113: 8.643020,
    600.603: 8.647157,
    600.718: 8.640394,
    601.068: 8.640394,
    601.322: 8.647157,
    601.484: 8.643020,
    657.805: 14.448826,
    658.288: 14.448826,
    658.761: 8.537097,
    674.438: 38.22647,
    678.390: 20.709788,
    703.725: 38.435883,
    711.318: 8.647157,
    711.519: 8.640394,
    711.563: 22.532398,
    711.699: 8.647157,
    711.990: 22.536906,
    723.132: 16.33174,
    723.642: 16.333122,
    761.265: 41.342187,
    786.089: 8.850659,
    805.862: 8.850659,
}

DATA_SAMPLE_INTERPOLATED: tuple = (
    0.000000000000000,
    0.000230709627131,
    0.000384144814593,
    0.000507137093115,
    0.000632114832536,
    0.000778810112328,
    0.000955965592105,
    0.001163041382140,
    0.001391921913876,
    0.001628622810444,
    0.001854997757177,
    0.002050445372122,
    0.002193616076555,
    0.002264118965498,
    0.002244228678230,
    0.002120592268802,
    0.001885936076555,
    0.001540772596628,
    0.001095107350478,
    0.000570145756392,
    0.000000000000000,
    -0.000581325882457,
    -0.001183945630981,
    -0.001820338942229,
    -0.002492278660287,
    -0.003192248184715,
    -0.003904858878589,
    -0.004608267476544,
    -0.005275593492823,
    -0.005876336629317,
    -0.006377794183612,
    -0.006746478457031,
    -0.006949534162679,
    -0.006956155833489,
    -0.006739005230263,
    -0.006275628749720,
    -0.005549874832536,
    -0.004553311371393,
    -0.003286643119019,
    -0.001761129096236,
    0.000000000000000,
    0.001979832128906,
    0.004221516875000,
    0.006762744980469,
    0.009617760000000,
    0.012780285644531,
    0.016226453125000,
    0.019917728496094,
    0.023803840000000,
    0.027825705410156,
    0.031918359375000,
    0.036013880761719,
    0.040044320000000,
    0.043944626425781,
    0.047655575625000,
    0.051126696777344,
    0.054319200000000,
    0.057208903691406,
    0.059789161875000,
    0.062073791542969,
    0.064100000000000,
    0.065908998066406,
    0.067477265625000,
    0.068776060136719,
    0.069791040000000,
    0.070520520019531,
    0.070973726875000,
    0.071169054589844,
    0.071132320000000,
    0.070895018222656,
    0.070492578125000,
    0.069962617792969,
    0.069343200000000,
    0.068671087675781,
    0.067979999375000,
    0.067298864746094,
    0.066650080000000,
    0.066047763378906,
    0.065496010625000,
    0.064987150449219,
    0.064500000000000,
    0.064007805449219,
    0.063510900625000,
    0.063016138378906,
    0.062528640000000,
    0.062051989746094,
    0.061588429375000,
    0.061139052675781,
    0.060704000000000,
    0.060282652792969,
    0.059873828125000,
    0.059475973222656,
    0.059087360000000,
    0.058706279589844,
    0.058331236875000,
    0.057961145019531,
    0.057595520000000,
    0.057234675136719,
    0.056879915625000,
    0.056533733066406,
    0.056200000000000,
    0.055883811757812,
    0.055589516250000,
    0.055320009023437,
    0.055076960000000,
    0.054860961914063,
    0.054671678750000,
    0.054507994179687,
    0.054368160000000,
    0.054249944570312,
    0.054150781250000,
    0.054067916835938,
    0.053998560000000,
    0.053940029726562,
    0.053889903750000,
    0.053846166992187,
    0.053807360000000,
    0.053772727382813,
    0.053742366250000,
    0.053717374648438,
    0.053700000000000,
    0.053692483144531,
    0.053693175625000,
    0.053700270058594,
    0.053712960000000,
    0.053731311035156,
    0.053756131875000,
    0.053788845449219,
    0.053831360000000,
    0.053885940175781,
    0.053955078125000,
    0.054041364589844,
    0.054147360000000,
    0.054275465566406,
    0.054427794375000,
    0.054606042480469,
    0.054811360000000,
    0.055044222207031,
    0.055304300625000,
    0.055590334121094,
    0.055900000000000,
    0.056231407851562,
    0.056587597500000,
    0.056971279335937,
    0.057383360000000,
    0.057823168945313,
    0.058288685000000,
    0.058776762929688,
    0.059283360000000,
    0.059803762539063,
    0.060332812500000,
    0.060865134023438,
    0.061395360000000,
    0.061918358632813,
    0.062429460000000,
    0.062924682617188,
    0.063400960000000,
    0.063856367226563,
    0.064290347500000,
    0.064703938710938,
    0.065100000000000,
    0.065481647265625,
    0.065846786250000,
    0.066192982265625,
    0.066519040000000,
    0.066824853515625,
    0.067111256250000,
    0.067379871015625,
    0.067632960000000,
    0.067873274765625,
    0.068103906250000,
    0.068328134765625,
    0.068549280000000,
    0.068770551015625,
    0.068994896250000,
    0.069224853515625,
    0.069462400000000,
    0.069708802265625,
    0.069964466250000,
    0.070228787265625,
    0.070500000000000,
    0.070776270703125,
    0.071059446250000,
    0.071351597578125,
    0.071653920000000,
    0.071966845703125,
    0.072290156250000,
    0.072623095078125,
    0.072964480000000,
    0.073312815703125,
    0.073666406250000,
    0.074023467578125,
    0.074382240000000,
    0.074741100703125,
    0.075098676250000,
    0.075453955078125,
    0.075806400000000,
    0.076156060703125,
    0.076503686250000,
    0.076850837578125,
    0.077200000000000,
    0.077552701992188,
    0.077904262500000,
    0.078250762070313,
    0.078590880000000,
    0.078925561523438,
    0.079257685000000,
    0.079591729101562,
    0.079933440000000,
    0.080289498554688,
    0.080667187500000,
    0.081074058632813,
    0.081517600000000,
    0.082004903085938,
    0.082542330000000,
    0.083135180664063,
    0.083787360000000,
    0.084501045117188,
    0.085276352500000,
    0.086111005195313,
    0.087000000000000,
    0.087938453242188,
    0.088930890000000,
    0.089981833007812,
    0.091092960000000,
    0.092263452148438,
    0.093490342500000,
    0.094768864414062,
    0.096092800000000,
    0.097454828554687,
    0.098846875000000,
    0.100260458320312,
    0.101687040000000,
    0.103118372460937,
    0.104546847500000,
    0.105965844726563,
    0.107370080000000,
    0.108755953867188,
    0.110121900000000,
    0.111468733632812,
    0.112800000000000,
    0.114120657988281,
    0.115431176875000,
    0.116730532871094,
    0.118017600000000,
    0.119291174316406,
    0.120549998125000,
    0.121792784199219,
    0.123018240000000,
    0.124225091894531,
    0.125412109375000,
    0.126578129277344,
    0.127722080000000,
    0.128843005722656,
    0.129940090625000,
    0.131012683105469,
    0.132060320000000,
    0.133082750800781,
    0.134079961875000,
    0.135052200683594,
    0.136000000000000,
    0.136923531484375,
    0.137820920000000,
    0.138690739765625,
    0.139532640000000,
    0.140347216796875,
    0.141135885000000,
    0.141900750078125,
    0.142644480000000,
    0.143370177109375,
    0.144081250000000,
    0.144781285390625,
    0.145473920000000,
    0.146162712421875,
    0.146851015000000,
    0.147541845703125,
    0.148237760000000,
    0.148940722734375,
    0.149651980000000,
    0.150371931015625,
    0.151100000000000,
    0.151834687363281,
    0.152574745625000,
    0.153319862089844,
    0.154070560000000,
    0.154828088378906,
    0.155594311875000,
    0.156371600605469,
    0.157162720000000,
    0.157970720644531,
    0.158798828125000,
    0.159650332871094,
    0.160528480000000,
    0.161436359160156,
    0.162376794375000,
    0.163352233886719,
    0.164364640000000,
    0.165415378925781,
    0.166505110625000,
    0.167633678652344,
    0.168800000000000,
    0.170002988242187,
    0.171244585000000,
    0.172526722382812,
    0.173850400000000,
    0.175215795898437,
    0.176622377500000,
    0.178069012539063,
    0.179554080000000,
    0.181075581054688,
    0.182631250000000,
    0.184218665195313,
    0.185835360000000,
    0.187478933710938,
    0.189147162500000,
    0.190838110351563,
    0.192550240000000,
    0.194282523867188,
    0.196034555000000,
    0.197806658007813,
    0.199600000000000,
    0.201405046894531,
    0.203174116875000,
    0.204868198964844,
    0.206468000000000,
    0.207971350097656,
    0.209390608125000,
    0.210750067167969,
    0.212083360000000,
    0.213430864550781,
    0.214837109375000,
    0.216348179121094,
    0.218009120000000,
    0.219861345253906,
    0.221940040625000,
    0.224271569824219,
    0.226870880000000,
    0.229738907207031,
    0.232859981875000,
    0.236199234277344,
    0.239700000000000,
    0.243337282929688,
    0.247262056250000,
    0.251598942851562,
    0.256394240000000,
    0.261625952148438,
    0.267213823750000,
    0.273029372070313,
    0.278905920000000,
    0.284648628867188,
    0.290044531250000,
    0.294872563789062,
    0.298913600000000,
    0.301960483085938,
    0.303828058750000,
    0.304363208007813,
    0.303454880000000,
    0.301044124804688,
    0.297134126250000,
    0.291800234726562,
    0.285200000000000,
    0.277470396855469,
    0.268408756875000,
    0.257826504003906,
    0.245651040000000,
    0.231911267089844,
    0.216723110625000,
    0.200275041738281,
    0.182813600000000,
    0.164628916074219,
    0.146040234375000,
    0.127381435722656,
    0.108986560000000,
    0.091175328808594,
    0.074238668125000,
    0.058424230957031,
    0.043921920000000,
    0.030849410292969,
    0.019237671875000,
    0.009016492441406,
    0.000000000000000,
    -0.008014721386719,
    -0.014901410625000,
    -0.020510217441406,
    -0.024767360000000,
    -0.027665856933594,
    -0.029256259375000,
    -0.029637382988281,
    -0.028947040000000,
    -0.027352771230469,
    -0.025042578125000,
    -0.022215654785156,
    -0.019073120000000,
    -0.015808749277344,
    -0.012599706875000,
    -0.009597277832031,
    -0.006917600000000,
    -0.004632396074219,
    -0.002759705625000,
    -0.001254617128906,
    0.000000000000000,
    0.001146203203125,
    0.002165737500000,
    0.003010363984375,
    0.003650560000000,
    0.004073291015625,
    0.004279782500000,
    0.004283291796875,
    0.004106880000000,
    0.003781183828125,
    0.003342187500000,
    0.002828994609375,
    0.002281600000000,
    0.001738661640625,
    0.001235272500000,
    0.000800732421875,
    0.000456320000000,
    0.000213064453125,
    0.000069517500000,
    0.000009525234375,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
)

DATA_SAMPLE_INTERPOLATED_NON_UNIFORM: tuple = (
    16.33174000,
    16.64993716,
    17.64548368,
    19.20219560,
    21.20388894,
    23.53437976,
    26.07748407,
    28.71701792,
    31.33679734,
    33.82063837,
    36.05235704,
    37.91576938,
    39.29469143,
    40.07293922,
    40.14077612,
    39.92419302,
    39.89684762,
    39.81049683,
    39.69333189,
    39.57332622,
    39.47845324,
    39.43668638,
    39.47599905,
    39.62436468,
    39.90975669,
    40.33304002,
    40.50491197,
    39.74501724,
    37.67902112,
    34.62018880,
    30.97314049,
    27.14249640,
    23.53287675,
    20.54890177,
    18.59519166,
    18.08081760,
    19.38373812,
    22.23096797,
    26.16305357,
    30.72054135,
    35.44397775,
    39.87706471,
    43.74602436,
    47.05613715,
    49.83624442,
    52.11518748,
    53.92180764,
    55.28494623,
    56.23344457,
    56.79614397,
    57.00188576,
    56.87951125,
    56.45786176,
    55.76577861,
    54.83210312,
    53.68567662,
    52.35534040,
    50.86993581,
    49.25830415,
    47.54928674,
    45.77172491,
    43.95445997,
    42.12633325,
    40.31618605,
    38.55285970,
    36.86519552,
    35.28203483,
    33.83221894,
    32.54458918,
    31.44798686,
    30.57125331,
    29.94322984,
    29.59275777,
    29.54381964,
    32.79286157,
    40.36558235,
    45.21160576,
    47.09673165,
    46.46577457,
    43.76354913,
    39.43486990,
    33.92455146,
    27.67740839,
    21.13825529,
    14.75190672,
    8.96317728,
    4.18354245,
    0.49841631,
    -2.19120584,
    -3.98642200,
    -4.98833019,
    -5.29802840,
    -5.01661465,
    -4.24518693,
    -3.08484325,
    -1.63668162,
    -0.00180003,
    1.71870349,
    3.42373096,
    5.01218436,
    6.38296569,
    7.43497695,
    8.07693863,
    8.32474693,
    8.26027419,
    7.96635835,
    7.52583734,
    7.02154908,
    6.53633153,
    6.15302259,
    5.95446022,
    6.02348233,
    6.44292687,
    7.29563177,
    8.65167043,
    10.43511482,
    12.47696318,
    14.60671211,
    16.65385820,
    18.44789806,
    19.81832829,
    20.59464547,
    20.70989576,
    20.71490257,
    20.63782267,
    20.42824992,
    20.09863019,
    19.66155390,
    19.12961147,
    18.51539332,
    17.83148987,
    17.09049153,
    16.30498873,
    15.48757188,
    14.65083142,
    13.80735774,
    12.96974128,
    12.15057246,
    11.36244169,
    10.61793939,
    9.92965598,
    9.31018188,
    8.77210752,
    8.32802330,
    7.99051965,
    7.77218699,
    7.68561574,
    7.74083122,
    7.93173795,
    8.24599287,
    8.67123920,
    9.19512022,
    9.80527915,
    10.48935925,
    11.23500376,
    12.02985593,
    12.86155901,
    13.71775624,
    14.58609087,
    15.45420615,
    16.30974532,
    17.14035164,
    17.93366833,
    18.67733867,
    19.35900588,
    19.96631322,
    20.48690393,
    20.90842126,
    21.21850846,
    21.40480878,
    21.45496545,
    21.35662174,
    21.09742087,
    20.66514352,
    20.43751858,
    21.91097080,
    25.20288294,
    29.29167835,
    33.13704964,
    36.09348577,
    38.20199674,
    39.57231714,
    40.31418157,
    40.53732465,
    40.35148096,
    39.86638512,
    39.19177172,
    38.43737537,
    37.71293067,
    37.32215353,
    38.85348208,
    40.28480326,
    38.66906876,
    34.70409378,
    29.52631184,
    24.27215647,
    20.07806121,
    18.08045957,
    17.97606450,
    17.51718098,
    16.71767084,
    15.66588470,
    14.45017318,
    13.15888689,
    11.88037645,
    10.70299247,
    9.71508557,
    9.00500637,
    8.66110548,
    8.63646961,
    8.61886171,
    8.49018794,
    8.26908563,
    7.96116808,
    7.57204861,
    7.10734052,
    6.57265713,
    5.97361172,
    5.31581763,
    4.60488814,
    3.84643656,
    3.04607621,
    2.20942040,
    1.34208241,
    0.44967558,
    -0.46218681,
    -1.38789144,
    -2.32182500,
    -3.25837419,
    -4.19192570,
    -5.11686623,
    -6.02758247,
    -6.91846110,
    -7.78388883,
    -8.61825235,
    -9.41593835,
    -10.17133352,
    -10.87882456,
    -11.53279815,
    -12.12764100,
    -12.65773980,
    -13.11748123,
    -13.50125200,
    -13.80343879,
    -14.01842830,
    -14.14060723,
    -14.16436225,
    -14.08408008,
    -13.89414739,
    -13.58895089,
    -13.16287727,
    -12.61031321,
    -11.92564542,
    -11.10326059,
    -10.13754541,
    -9.02288657,
    -7.75367076,
    -6.32428468,
    -4.72911503,
    -2.96254849,
    -1.01897176,
    1.10722847,
    3.42166550,
    5.92995265,
    8.63770321,
    11.55053050,
    14.66926935,
    6.86179709,
    -5.47290466,
    -13.83246355,
    -18.66023243,
    -20.39956416,
    -19.49381157,
    -16.38632753,
    -11.52046487,
    -5.33957647,
    1.71298484,
    9.19386620,
    16.65971477,
    23.66717768,
    29.77290209,
    34.53353516,
    37.50572402,
    38.25154792,
    36.50596565,
    32.20857414,
    25.31081084,
    15.77278173,
    3.77074445,
    -10.29313973,
    -26.00608936,
    -42.95532304,
    -60.72805933,
    -78.91151682,
    -97.09291408,
    -114.85946968,
    -131.79840221,
    -147.49693024,
    -161.54227235,
    -173.52164712,
    -183.02227312,
    -189.63136892,
    -192.93615311,
    -192.52384426,
    -187.98166096,
    -178.89682177,
    -164.85654527,
    -145.44805004,
    -120.25855466,
    -88.87527770,
    -50.88543775,
    -5.87625337,
    46.56007116,
    104.43504592,
    159.52668786,
    202.61238918,
    224.46954207,
    215.87553872,
    167.60777130,
    70.44363202,
    5.46110638,
    141.75857149,
    230.42498382,
    276.55540391,
    287.68451731,
    271.34700958,
    235.07756626,
    186.41087289,
    132.88161504,
    82.02447824,
    41.37414805,
    18.46531002,
    16.85424129,
    19.09321438,
    21.24241296,
    23.30206825,
    25.27241149,
    27.15367391,
    28.94608674,
    30.64988121,
    32.26528857,
    33.79254002,
    35.23186682,
    36.58350019,
    37.84767137,
    39.02461158,
    40.11455206,
    41.11772404,
    42.03435875,
    42.86468742,
    43.60894129,
    44.26735159,
    44.84014956,
    45.32756641,
    45.72983339,
    46.04718172,
    46.27984264,
    46.42804739,
    46.49202718,
    46.47201326,
    46.36823686,
    46.18092921,
    45.91032154,
    45.55664508,
    45.12013106,
    44.60101073,
    43.99951530,
    43.31587601,
    42.55032410,
    41.70309079,
    40.77461777,
    39.76848476,
    38.69068156,
    37.54725968,
    36.34427059,
    35.08776577,
    33.78379673,
    32.43841493,
    31.05767187,
    29.64761904,
    28.21430791,
    26.76378998,
    25.30211674,
    23.83533966,
    22.36951024,
    20.91067996,
    19.46490031,
    18.03822277,
    16.63669883,
    15.26637998,
    13.93331770,
    12.64356348,
    11.40316880,
    10.21818516,
    9.09466403,
    8.03865691,
    7.05621528,
    6.15339062,
    5.33623443,
    4.61079818,
    3.98313337,
    3.45929148,
    3.04532400,
    2.74728241,
    2.57121821,
    2.52318286,
    2.60922788,
    2.83540473,
    3.20776490,
    3.73235989,
    4.41524117,
    5.26246024,
    6.28006857,
    7.47411766,
    8.85065900,
)

DATA_SAMPLE_NORMALISED: tuple = (
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    22.475455820476860,
    22.615708274894811,
    19.705469845722302,
    18.828892005610097,
    19.600280504908834,
    22.826086956521742,
    24.719495091164092,
    27.068723702664798,
    30.504908835904626,
    39.551192145862551,
    47.685834502103788,
    52.980364656381497,
    59.186535764375883,
    69.985974754558200,
    84.046283309957929,
    100.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
)

DATA_STANDARD_OBSERVER_2_DEGREE_CIE1931: dict = {
    380: (0.001368, 0.000039, 0.006450),
    385: (0.002236, 0.000064, 0.010550),
    390: (0.004243, 0.000120, 0.020050),
    395: (0.007650, 0.000217, 0.036210),
    400: (0.014310, 0.000396, 0.067850),
    405: (0.023190, 0.000640, 0.110200),
    410: (0.043510, 0.001210, 0.207400),
    415: (0.077630, 0.002180, 0.371300),
    420: (0.134380, 0.004000, 0.645600),
    425: (0.214770, 0.007300, 1.039050),
    430: (0.283900, 0.011600, 1.385600),
    435: (0.328500, 0.016840, 1.622960),
    440: (0.348280, 0.023000, 1.747060),
    445: (0.348060, 0.029800, 1.782600),
    450: (0.336200, 0.038000, 1.772110),
    455: (0.318700, 0.048000, 1.744100),
    460: (0.290800, 0.060000, 1.669200),
    465: (0.251100, 0.073900, 1.528100),
    470: (0.195360, 0.090980, 1.287640),
    475: (0.142100, 0.112600, 1.041900),
    480: (0.095640, 0.139020, 0.812950),
    485: (0.057950, 0.169300, 0.616200),
    490: (0.032010, 0.208020, 0.465180),
    495: (0.014700, 0.258600, 0.353300),
    500: (0.004900, 0.323000, 0.272000),
    505: (0.002400, 0.407300, 0.212300),
    510: (0.009300, 0.503000, 0.158200),
    515: (0.029100, 0.608200, 0.111700),
    520: (0.063270, 0.710000, 0.078250),
    525: (0.109600, 0.793200, 0.057250),
    530: (0.165500, 0.862000, 0.042160),
    535: (0.225750, 0.914850, 0.029840),
    540: (0.290400, 0.954000, 0.020300),
    545: (0.359700, 0.980300, 0.013400),
    550: (0.433450, 0.994950, 0.008750),
    555: (0.512050, 1.000000, 0.005750),
    560: (0.594500, 0.995000, 0.003900),
    565: (0.678400, 0.978600, 0.002750),
    570: (0.762100, 0.952000, 0.002100),
    575: (0.842500, 0.915400, 0.001800),
    580: (0.916300, 0.870000, 0.001650),
    585: (0.978600, 0.816300, 0.001400),
    590: (1.026300, 0.757000, 0.001100),
    595: (1.056700, 0.694900, 0.001000),
    600: (1.062200, 0.631000, 0.000800),
    605: (1.045600, 0.566800, 0.000600),
    610: (1.002600, 0.503000, 0.000340),
    615: (0.938400, 0.441200, 0.000240),
    620: (0.854450, 0.381000, 0.000190),
    625: (0.751400, 0.321000, 0.000100),
    630: (0.642400, 0.265000, 0.000050),
    635: (0.541900, 0.217000, 0.000030),
    640: (0.447900, 0.175000, 0.000020),
    645: (0.360800, 0.138200, 0.000010),
    650: (0.283500, 0.107000, 0.000000),
    655: (0.218700, 0.081600, 0.000000),
    660: (0.164900, 0.061000, 0.000000),
    665: (0.121200, 0.044580, 0.000000),
    670: (0.087400, 0.032000, 0.000000),
    675: (0.063600, 0.023200, 0.000000),
    680: (0.046770, 0.017000, 0.000000),
    685: (0.032900, 0.011920, 0.000000),
    690: (0.022700, 0.008210, 0.000000),
    695: (0.015840, 0.005723, 0.000000),
    700: (0.011359, 0.004102, 0.000000),
    705: (0.008111, 0.002929, 0.000000),
    710: (0.005790, 0.002091, 0.000000),
    715: (0.004109, 0.001484, 0.000000),
    720: (0.002899, 0.001047, 0.000000),
    725: (0.002049, 0.000740, 0.000000),
    730: (0.001440, 0.000520, 0.000000),
    735: (0.001000, 0.000361, 0.000000),
    740: (0.000690, 0.000249, 0.000000),
    745: (0.000476, 0.000172, 0.000000),
    750: (0.000332, 0.000120, 0.000000),
    755: (0.000235, 0.000085, 0.000000),
    760: (0.000166, 0.000060, 0.000000),
    765: (0.000117, 0.000042, 0.000000),
    770: (0.000083, 0.000030, 0.000000),
    775: (0.000059, 0.000021, 0.000000),
    780: (0.000042, 0.000015, 0.000000),
}

DATA_CMFS: dict = {
    380: np.array([0.001368, 3.90e-05, 0.006450]),
    385: np.array([0.002236, 6.40e-05, 0.010550]),
    390: np.array([0.004243, 0.000120, 0.020050]),
    395: np.array([0.007650, 0.000217, 0.036210]),
    400: np.array([0.014310, 0.000396, 0.067850]),
    405: np.array([0.023190, 0.000640, 0.110200]),
    410: np.array([0.043510, 0.001210, 0.207400]),
    415: np.array([0.077630, 0.002180, 0.371300]),
    420: np.array([0.134380, 0.004000, 0.645600]),
    425: np.array([0.214770, 0.007300, 1.039050]),
    430: np.array([0.283900, 0.011600, 1.385600]),
    435: np.array([0.328500, 0.016840, 1.622960]),
    440: np.array([0.348280, 0.023000, 1.747060]),
    445: np.array([0.348060, 0.029800, 1.782600]),
    450: np.array([0.336200, 0.038000, 1.772110]),
    455: np.array([0.318700, 0.048000, 1.744100]),
    460: np.array([0.290800, 0.060000, 1.669200]),
    465: np.array([0.251100, 0.073900, 1.528100]),
    470: np.array([0.195360, 0.090980, 1.287640]),
    475: np.array([0.142100, 0.112600, 1.041900]),
    480: np.array([0.095640, 0.139020, 0.812950]),
    485: np.array([0.057950, 0.169300, 0.616200]),
    490: np.array([0.032010, 0.208020, 0.465180]),
    495: np.array([0.014700, 0.258600, 0.353300]),
    500: np.array([0.004900, 0.323000, 0.272000]),
    505: np.array([0.002400, 0.407300, 0.212300]),
    510: np.array([0.009300, 0.503000, 0.158200]),
    515: np.array([0.029100, 0.608200, 0.111700]),
    520: np.array([0.063270, 0.710000, 0.078250]),
    525: np.array([0.109600, 0.793200, 0.057250]),
    530: np.array([0.165500, 0.862000, 0.042160]),
    535: np.array([0.225750, 0.914850, 0.029840]),
    540: np.array([0.290400, 0.954000, 0.020300]),
    545: np.array([0.359700, 0.980300, 0.013400]),
    550: np.array([0.433450, 0.994950, 0.008750]),
    555: np.array([0.512050, 1.000000, 0.005750]),
    560: np.array([0.594500, 0.995000, 0.003900]),
    565: np.array([0.678400, 0.978600, 0.002750]),
    570: np.array([0.762100, 0.952000, 0.002100]),
    575: np.array([0.842500, 0.915400, 0.001800]),
    580: np.array([0.916300, 0.870000, 0.001650]),
    585: np.array([0.978600, 0.816300, 0.001400]),
    590: np.array([1.026300, 0.757000, 0.001100]),
    595: np.array([1.056700, 0.694900, 0.001000]),
    600: np.array([1.062200, 0.631000, 0.000800]),
    605: np.array([1.045600, 0.566800, 0.000600]),
    610: np.array([1.002600, 0.503000, 0.000340]),
    615: np.array([0.938400, 0.441200, 0.000240]),
    620: np.array([0.854450, 0.381000, 0.000190]),
    625: np.array([0.751400, 0.321000, 0.000100]),
    630: np.array([0.642400, 0.265000, 5.00e-05]),
    635: np.array([0.541900, 0.217000, 3.00e-05]),
    640: np.array([0.447900, 0.175000, 2.00e-05]),
    645: np.array([0.360800, 0.138200, 1.00e-05]),
    650: np.array([0.283500, 0.107000, 0.000000]),
    655: np.array([0.218700, 0.081600, 0.000000]),
    660: np.array([0.164900, 0.061000, 0.000000]),
    665: np.array([0.121200, 0.044580, 0.000000]),
    670: np.array([0.087400, 0.032000, 0.000000]),
    675: np.array([0.063600, 0.023200, 0.000000]),
    680: np.array([0.046770, 0.017000, 0.000000]),
    685: np.array([0.032900, 0.011920, 0.000000]),
    690: np.array([0.022700, 0.008210, 0.000000]),
    695: np.array([0.015840, 0.005723, 0.000000]),
    700: np.array([0.011359, 0.004102, 0.000000]),
    705: np.array([0.008111, 0.002929, 0.000000]),
    710: np.array([0.005790, 0.002091, 0.000000]),
    715: np.array([0.004109, 0.001484, 0.000000]),
    720: np.array([0.002899, 0.001047, 0.000000]),
    725: np.array([0.002049, 0.000740, 0.000000]),
    730: np.array([0.001440, 0.000520, 0.000000]),
    735: np.array([0.001000, 0.000361, 0.000000]),
    740: np.array([0.000690, 0.000249, 0.000000]),
    745: np.array([0.000476, 0.000172, 0.000000]),
    750: np.array([0.000332, 0.000120, 0.000000]),
    755: np.array([0.000235, 8.50e-05, 0.000000]),
    760: np.array([0.000166, 6.00e-05, 0.000000]),
    765: np.array([0.000117, 4.20e-05, 0.000000]),
    770: np.array([8.30e-05, 3.00e-05, 0.000000]),
    775: np.array([5.90e-05, 2.10e-05, 0.000000]),
    780: np.array([4.20e-05, 1.50e-05, 0.000000]),
}

DATA_SAMPLE_ABRIDGED: dict = {
    500: 0.0651,
    520: 0.0705,
    540: 0.0772,
    560: 0.0870,
    580: 0.1128,
    600: 0.1360,
}

DATA_MULTI_SAMPLE_ABRIDGED: dict = {
    500: (0.004900, 0.323000, 0.272000),
    510: (0.009300, 0.503000, 0.158200),
    520: (0.063270, 0.710000, 0.078250),
    530: (0.165500, 0.862000, 0.042160),
    540: (0.290400, 0.954000, 0.020300),
    550: (0.433450, 0.994950, 0.008750),
    560: (0.594500, 0.995000, 0.003900),
}


class TestSpectralShape:
    """
    Define :class:`colour.colorimetry.spectrum.SpectralShape` class unit tests
    methods.
    """

    def test_required_attributes(self) -> None:
        """Test the presence of required attributes."""

        required_attributes = (
            "start",
            "end",
            "interval",
            "boundaries",
            "wavelengths",
        )

        for attribute in required_attributes:
            assert attribute in dir(SpectralShape)

    def test_required_methods(self) -> None:
        """Test the presence of required methods."""

        required_methods = (
            "__init__",
            "__str__",
            "__repr__",
            "__hash__",
            "__iter__",
            "__contains__",
            "__len__",
            "__eq__",
            "__ne__",
            "range",
        )

        for method in required_methods:
            assert method in dir(SpectralShape)

    def test_pickling(self) -> None:
        """
        Test whether the :class:`colour.colorimetry.spectrum.SpectralShape`
        class can be pickled.
        """

        shape = SpectralShape(360, 830, 1)
        data = pickle.dumps(shape)
        data = pickle.loads(data)  # noqa: S301
        assert shape == data

    def test_start(self) -> None:
        """
        Test :attr:`colour.colorimetry.spectrum.SpectralShape.start`
        attribute.
        """

        assert SpectralShape(360, 830, 1).start == 360

        pytest.raises(AssertionError, lambda: SpectralShape(360, 360, 1))

        pytest.raises(AssertionError, lambda: SpectralShape(360, 0, 1))

    def test_end(self) -> None:
        """Test :attr:`colour.colorimetry.spectrum.SpectralShape.end` property."""

        assert SpectralShape(360, 830, 1).end == 830

        pytest.raises(AssertionError, lambda: SpectralShape(830, 830, 1))

        pytest.raises(AssertionError, lambda: SpectralShape(830, 0, 1))

    def test_interval(self) -> None:
        """
        Test :attr:`colour.colorimetry.spectrum.SpectralShape.interval`
        property.
        """

        assert SpectralShape(360, 830, 1).interval == 1

    def test_boundaries(self) -> None:
        """
        Test :attr:`colour.colorimetry.spectrum.SpectralShape.boundaries`
        property.
        """

        shape = SpectralShape(400, 700, 1)
        shape.boundaries = (360, 830)

        assert shape.start == 360
        assert shape.end == 830

    def test_wavelengths(self) -> None:
        """
        Test :attr:`colour.colorimetry.spectrum.SpectralShape.wavelengths`
        property.
        """

        np.testing.assert_array_equal(
            SpectralShape(0, 10, 0.1).wavelengths,
            np.arange(0, 10 + 0.1, 0.1),
        )

    def test__hash__(self) -> None:
        """
        Test :meth:`colour.colorimetry.spectrum.SpectralShape.__hash__`
        method.
        """

        assert isinstance(hash(SpectralShape(0, 10, 0.1)), int)

    def test__iter__(self) -> None:
        """
        Test :meth:`colour.colorimetry.spectrum.SpectralShape.__iter__`
        method.
        """

        np.testing.assert_array_equal(
            list(SpectralShape(0, 10, 0.1)),
            np.arange(0, 10 + 0.1, 0.1),
        )

    def test__contains__(self) -> None:
        """
        Test :meth:`colour.colorimetry.spectrum.SpectralShape.__contains__`
        method.
        """

        assert 360.1 in SpectralShape(360, 830, 0.1)

        assert 360.11 not in SpectralShape(360, 830, 0.1)

        assert np.array([0.5, 0.6]) in SpectralShape(0, 10, 0.1)

        assert np.array([0.5, 0.61]) not in SpectralShape(0, 10, 0.1)

    def test__len__(self) -> None:
        """
        Test :meth:`colour.colorimetry.spectrum.SpectralShape.__len__`
        method.
        """

        assert len(SpectralShape(0, 10, 0.1)) == 101

    def test__eq__(self) -> None:
        """
        Test :meth:`colour.colorimetry.spectrum.SpectralShape.__eq__`
        method.
        """

        assert SpectralShape(0, 10, 0.1) == SpectralShape(0, 10, 0.1)
        assert SpectralShape(0, 10, 0.1) != ()

    def test__ne__(self) -> None:
        """
        Test :meth:`colour.colorimetry.spectrum.SpectralShape.__ne__`
        method.
        """

        assert SpectralShape(0, 10, 0.1) != SpectralShape(1, 10, 0.1)

    def test_range(self) -> None:
        """Test :func:`colour.colorimetry.spectrum.SpectralShape.range` method."""

        np.testing.assert_array_equal(
            list(SpectralShape(0, 10, 0.1)),
            np.arange(0, 10 + 0.1, 0.1),
        )


class TestSpectralDistribution:
    """
    Define :class:`colour.colorimetry.spectrum.SpectralDistribution`
    class unit tests methods.
    """

    def setup_method(self) -> None:
        """Initialise the common tests attributes."""

        self._sd = SpectralDistribution(DATA_SAMPLE, name="Sample")

        self._non_uniform_sd = SpectralDistribution(
            DATA_SAMPLE_NON_UNIFORM,
            name="Non Uniform Sample",
            display_name="Display Non Uniform Sample",
        )

        self._phi = (1 + np.sqrt(5)) / 2

    def test_required_attributes(self) -> None:
        """Test the presence of required attributes."""

        required_attributes = (
            "display_name",
            "wavelengths",
            "values",
            "shape",
        )

        for attribute in required_attributes:
            assert attribute in dir(SpectralDistribution)

    def test_required_methods(self) -> None:
        """Test the presence of required methods."""

        required_methods = (
            "__init__",
            "interpolate",
            "extrapolate",
            "align",
            "trim",
            "normalise",
        )

        for method in required_methods:
            assert method in dir(SpectralDistribution)

    def test_pickling(self) -> None:
        """
        Test whether the :class:`colour.colorimetry.spectrum.\
SpectralDistribution` class can be pickled.
        """

        data = pickle.dumps(self._sd)
        data = pickle.loads(data)  # noqa: S301
        assert self._sd == data

    def test_display_name(self) -> None:
        """
        Test :attr:`colour.colorimetry.spectrum.SpectralDistribution.display_name`
        property.
        """

        assert self._sd.display_name == "Sample"
        assert self._non_uniform_sd.display_name == "Display Non Uniform Sample"

    def test_wavelengths(self) -> None:
        """
        Test :attr:`colour.colorimetry.spectrum.SpectralDistribution.wavelengths`
        property.
        """

        np.testing.assert_array_equal(self._sd.wavelengths, self._sd.domain)

        sd = self._sd.copy()
        sd.wavelengths = sd.wavelengths + 10
        np.testing.assert_array_equal(sd.wavelengths, sd.domain)

    def test_values(self) -> None:
        """
        Test :attr:`colour.colorimetry.spectrum.SpectralDistribution.values`
        property.
        """

        np.testing.assert_array_equal(self._sd.values, self._sd.range)

        sd = self._sd.copy()
        sd.values = sd.values + 10
        np.testing.assert_array_equal(sd.values, sd.range)

    def test_shape(self) -> None:
        """
        Test :attr:`colour.colorimetry.spectrum.SpectralDistribution.shape`
        property.
        """

        assert self._sd.shape == SpectralShape(340, 820, 20)

    def test__init__(self) -> None:
        """
        Test :meth:`colour.colorimetry.spectrum.SpectralDistribution.__init__`
        method.
        """

        np.testing.assert_allclose(
            SpectralDistribution(DATA_SAMPLE).wavelengths,
            SpectralDistribution(
                DATA_SAMPLE.values(),
                SpectralShape(340, 820, 20),
            ).wavelengths,
            atol=TOLERANCE_ABSOLUTE_TESTS,
        )

    def test_interpolate(self) -> None:
        """
        Test :func:`colour.colorimetry.spectrum.\
SpectralDistribution.interpolate` method.
        """

        if not is_scipy_installed():  # pragma: no cover
            return

        shape = SpectralShape(self._sd.shape.start, self._sd.shape.end, 1)
        sd = reshape_sd(self._sd, shape, "Interpolate")
        np.testing.assert_allclose(
            sd.values, DATA_SAMPLE_INTERPOLATED, atol=TOLERANCE_ABSOLUTE_TESTS
        )
        assert sd.shape == shape

        shape = SpectralShape(
            self._non_uniform_sd.shape.start,
            self._non_uniform_sd.shape.end,
            1,
        )
        sd = reshape_sd(
            self._non_uniform_sd,
            shape,
            "Interpolate",
        )
        np.testing.assert_allclose(
            sd.values,
            DATA_SAMPLE_INTERPOLATED_NON_UNIFORM,
            atol=TOLERANCE_ABSOLUTE_TESTS,
        )
        assert sd.shape == SpectralShape(
            self._non_uniform_sd.shape.start,
            self._non_uniform_sd.shape.end,
            1,
        )

    def test_extrapolate(self) -> None:
        """
        Test :func:`colour.colorimetry.spectrum.\
SpectralDistribution.extrapolate` method.
        """

        if not is_scipy_installed():  # pragma: no cover
            return

        data = dict(zip(range(25, 35), [0] * 5 + [1] * 5, strict=True))
        sd = SpectralDistribution(data)
        sd.extrapolate(SpectralShape(10, 50, 5))

        np.testing.assert_allclose(sd[10], 0, atol=TOLERANCE_ABSOLUTE_TESTS)
        np.testing.assert_allclose(sd[50], 1, atol=TOLERANCE_ABSOLUTE_TESTS)

        sd = SpectralDistribution(np.linspace(0, 1, 10), np.linspace(25, 35, 10))
        shape = SpectralShape(10, 50, 10)
        sd.extrapolate(
            shape,
            extrapolator_kwargs={
                "method": "Linear",
                "left": None,
                "right": None,
            },
        )

        np.testing.assert_allclose(
            sd[10], -1.5000000000000004, atol=TOLERANCE_ABSOLUTE_TESTS
        )
        np.testing.assert_allclose(
            sd[50], 2.4999999999999964, atol=TOLERANCE_ABSOLUTE_TESTS
        )

    def test_align(self) -> None:
        """
        Test :func:`colour.colorimetry.spectrum.\
SpectralDistribution.align` method.
        """

        shape = SpectralShape(100, 900, 5)
        assert self._sd.copy().align(shape).shape == shape

        shape = SpectralShape(600, 650, 1)
        assert self._sd.copy().align(shape).shape == shape

    def test_trim(self) -> None:
        """
        Test :func:`colour.colorimetry.spectrum.\
SpectralDistribution.trim` method.
        """

        shape = SpectralShape(400, 700, 20)
        assert self._sd.copy().trim(shape).shape == shape

        shape = SpectralShape(200, 900, 1)
        assert self._sd.copy().trim(shape).shape == self._sd.shape

    def test_normalise(self) -> None:
        """
        Test :func:`colour.colorimetry.spectrum.\
SpectralDistribution.normalise` method.
        """

        np.testing.assert_allclose(
            self._sd.copy().normalise(100).values,
            DATA_SAMPLE_NORMALISED,
            atol=TOLERANCE_ABSOLUTE_TESTS,
        )

    def test_callback_on_domain_changed(self) -> None:
        """
        Test :class:`colour.colorimetry.spectrum.\
SpectralDistribution` *on_domain_changed* callback.
        """

        sd = self._sd.copy()
        assert sd.shape == SpectralShape(340, 820, 20)
        sd[840] = 0
        assert sd.shape == SpectralShape(340, 840, 20)


class TestMultiSpectralDistributions:
    """
    Define :class:`colour.colorimetry.spectrum.MultiSpectralDistributions`
    class unit tests methods.
    """

    def setup_method(self) -> None:
        """Initialise the common tests attributes."""

        self._labels = ("x_bar", "y_bar", "z_bar")
        self._display_labels = (
            "Display x_bar",
            "Display y_bar",
            "Display z_bar",
        )

        self._msds = MultiSpectralDistributions(
            DATA_STANDARD_OBSERVER_2_DEGREE_CIE1931,
            name="Observer",
            labels=self._labels,
        )

        sd = SpectralDistribution(DATA_SAMPLE)
        domain = sd.domain
        range_ = tstack([sd.values, sd.values, sd.values])
        self._sample_msds = MultiSpectralDistributions(
            range_,
            domain,
            name="Sample Observer",
            labels=self._labels,
        )

        sd = SpectralDistribution(DATA_SAMPLE_NON_UNIFORM)
        domain = sd.domain
        range_ = tstack([sd.values, sd.values, sd.values])
        self._non_uniform_sample_msds = MultiSpectralDistributions(
            range_,
            domain,
            name="Non Uniform Sample Observer",
            display_name="Display Non Uniform Sample Observer",
            labels=self._labels,
            display_labels=("Display x_bar", "Display y_bar", "Display z_bar"),
        )

        self._phi = (1 + np.sqrt(5)) / 2

    def test_required_attributes(self) -> None:
        """Test the presence of required attributes."""

        required_attributes = (
            "display_name",
            "display_labels",
            "wavelengths",
            "values",
            "shape",
        )

        for attribute in required_attributes:
            assert attribute in dir(MultiSpectralDistributions)

    def test_required_methods(self) -> None:
        """Test the presence of required methods."""

        required_methods = (
            "__init__",
            "interpolate",
            "extrapolate",
            "align",
            "trim",
            "normalise",
            "to_sds",
        )

        for method in required_methods:
            assert method in dir(MultiSpectralDistributions)

    def test_pickling(self) -> None:
        """
        Test whether the :class:`colour.colorimetry.spectrum.\
MultiSpectralDistributions` class can be pickled.
        """

        data = pickle.dumps(self._msds)
        data = pickle.loads(data)  # noqa: S301
        assert self._msds == data

    def test_display_name(self) -> None:
        """
        Test :attr:`colour.colorimetry.spectrum.MultiSpectralDistributions.display_name`
        property.
        """

        assert self._sample_msds.display_name == "Sample Observer"
        assert (
            self._non_uniform_sample_msds.display_name
            == "Display Non Uniform Sample Observer"
        )

    def test_wavelengths(self) -> None:
        """
        Test :attr:`colour.colorimetry.spectrum.MultiSpectralDistributions.wavelengths`
        property.
        """

        np.testing.assert_array_equal(self._msds.wavelengths, self._msds.domain)

        msds = self._msds.copy()
        msds.wavelengths = msds.wavelengths + 10
        np.testing.assert_array_equal(msds.wavelengths, msds.domain)

    def test_values(self) -> None:
        """
        Test :attr:`colour.colorimetry.spectrum.MultiSpectralDistributions.values`
        property.
        """

        np.testing.assert_array_equal(self._msds.values, self._msds.range)

        msds = self._msds.copy()
        msds.values = msds.values + 10
        np.testing.assert_array_equal(msds.values, msds.range)

    def test_display_labels(self) -> None:
        """
        Test :attr:`colour.colorimetry.spectrum.MultiSpectralDistributions.\
display_labels` property.
        """

        assert tuple(self._sample_msds.display_labels) == self._labels
        assert tuple(self._non_uniform_sample_msds.display_labels) == (
            "Display x_bar",
            "Display y_bar",
            "Display z_bar",
        )

    def test_shape(self) -> None:
        """
        Test :attr:`colour.colorimetry.spectrum.MultiSpectralDistributions.shape`
        property.
        """

        assert self._msds.shape == SpectralShape(380, 780, 5)

    def test__init__(self) -> None:
        """
        Test :func:`colour.colorimetry.spectrum.\
MultiSpectralDistributions.__init__` method.
        """

        np.testing.assert_allclose(
            MultiSpectralDistributions(DATA_CMFS).wavelengths,
            MultiSpectralDistributions(
                DATA_CMFS.values(),
                SpectralShape(380, 780, 5),
            ).wavelengths,
            atol=TOLERANCE_ABSOLUTE_TESTS,
        )

    def test_interpolate(self) -> None:
        """
        Test :func:`colour.colorimetry.spectrum.\
MultiSpectralDistributions.interpolate` method.
        """

        if not is_scipy_installed():  # pragma: no cover
            return

        shape = SpectralShape(
            self._sample_msds.shape.start, self._sample_msds.shape.end, 1
        )
        msds = reshape_msds(self._sample_msds, shape, "Interpolate")
        for signal in msds.signals.values():
            np.testing.assert_allclose(
                signal.values,  # pyright: ignore
                DATA_SAMPLE_INTERPOLATED,
                atol=TOLERANCE_ABSOLUTE_TESTS,
            )
        assert msds.shape == shape

        shape = SpectralShape(
            self._non_uniform_sample_msds.shape.start,
            self._non_uniform_sample_msds.shape.end,
            1,
        )
        msds = reshape_msds(
            self._non_uniform_sample_msds,
            shape,
            "Interpolate",
        )
        for signal in msds.signals.values():
            np.testing.assert_allclose(
                signal.values,  # pyright: ignore
                DATA_SAMPLE_INTERPOLATED_NON_UNIFORM,
                atol=TOLERANCE_ABSOLUTE_TESTS,
            )
        assert msds.shape == SpectralShape(
            self._non_uniform_sample_msds.shape.start,
            self._non_uniform_sample_msds.shape.end,
            1,
        )

    def test_extrapolate(self) -> None:
        """
        Test :func:`colour.colorimetry.spectrum.\
MultiSpectralDistributions.extrapolate` method.
        """

        if not is_scipy_installed():  # pragma: no cover
            return

        data = dict(zip(range(25, 35), tstack([[0] * 5 + [1] * 5] * 3), strict=True))
        msds = MultiSpectralDistributions(data)
        msds.extrapolate(SpectralShape(10, 50, 5))

        np.testing.assert_allclose(
            msds[10], np.array([0.0, 0.0, 0.0]), atol=TOLERANCE_ABSOLUTE_TESTS
        )
        np.testing.assert_allclose(
            msds[50], np.array([1.0, 1.0, 1.0]), atol=TOLERANCE_ABSOLUTE_TESTS
        )

        msds = MultiSpectralDistributions(
            tstack([np.linspace(0, 1, 10)] * 3), np.linspace(25, 35, 10)
        )
        msds.extrapolate(
            SpectralShape(10, 50, 10),
            extrapolator_kwargs={
                "method": "Linear",
                "left": None,
                "right": None,
            },
        )
        np.testing.assert_allclose(
            msds[10],
            np.array([-1.5, -1.5, -1.5]),
            atol=TOLERANCE_ABSOLUTE_TESTS,
        )
        np.testing.assert_allclose(
            msds[50], np.array([2.5, 2.5, 2.5]), atol=TOLERANCE_ABSOLUTE_TESTS
        )

    def test_align(self) -> None:
        """
        Test :func:`colour.colorimetry.spectrum.\
MultiSpectralDistributions.align` method.
        """

        msds = self._sample_msds.copy()

        shape = SpectralShape(100, 900, 5)
        assert msds.align(shape).shape == shape

        shape = SpectralShape(600, 650, 1)
        assert msds.align(shape).shape == shape

    def test_trim(self) -> None:
        """
        Test :func:`colour.colorimetry.spectrum.\
MultiSpectralDistributions.trim` method.
        """

        shape = SpectralShape(400, 700, 5)
        assert self._msds.copy().trim(shape).shape == shape

        shape = SpectralShape(200, 900, 1)
        assert self._msds.copy().trim(shape).shape == self._msds.shape

    def test_normalise(self) -> None:
        """
        Test :func:`colour.colorimetry.spectrum.
        MultiSpectralDistributions.normalise` method.
        """

        np.testing.assert_allclose(
            self._sample_msds.copy().normalise(100).values,
            tstack([DATA_SAMPLE_NORMALISED] * 3),
            atol=TOLERANCE_ABSOLUTE_TESTS,
        )

    def test_to_sds(self) -> None:
        """
        Test :func:`colour.colorimetry.spectrum.\
MultiSpectralDistributions.to_sds` method.
        """

        if not is_scipy_installed():  # pragma: no cover
            return

        sds = self._non_uniform_sample_msds.to_sds()
        assert len(sds) == 3

        for i, sd in enumerate(sds):
            assert sd.name == self._labels[i]
            assert sd.display_name == self._display_labels[i]

    def test_callback_on_domain_changed(self) -> None:
        """
        Test underlying :class:`colour.colorimetry.spectrum.\
SpectralDistribution` *on_domain_changed* callback when used with
        :class:`colour.colorimetry.spectrum.MultiSpectralDistributions` class.
        """

        msds = self._msds.copy()
        assert msds.shape == SpectralShape(380, 780, 5)
        msds[785] = 0
        assert msds.shape == SpectralShape(380, 785, 5)


class TestReshapeSd:
    """
    Define :func:`colour.colorimetry.spectrum.reshape_sd` definition unit
    tests methods.
    """

    def test_reshape_sd(self) -> None:
        """Test :func:`colour.colorimetry.spectrum.reshape_sd` definition."""

        if not is_scipy_installed():  # pragma: no cover
            return

        sd = SpectralDistribution(DATA_SAMPLE_ABRIDGED)
        sd_reshaped = reshape_sd(sd)
        assert sd_reshaped == sd.copy().align(SPECTRAL_SHAPE_DEFAULT)
        assert reshape_sd(sd) == sd_reshaped

        shape = colour.SpectralShape(100, 900, 1)
        extrapolator_kwargs = {
            "method": "Constant",
            "left": 0.05,
            "right": 0.15,
        }
        sd_reshaped = reshape_sd(
            sd,
            shape,
            method="Extrapolate",
            extrapolator_kwargs=extrapolator_kwargs,
        )
        assert sd_reshaped == sd.copy().extrapolate(
            shape, extrapolator_kwargs=extrapolator_kwargs
        )

        shape = colour.SpectralShape(400, 700, 1)
        interpolator_kwargs = {"fill_value": 0}
        sd_reshaped = reshape_sd(
            sd,
            shape,
            method="Interpolate",
            interpolator=CubicSplineInterpolator,
            interpolator_kwargs=interpolator_kwargs,
        )
        assert sd_reshaped == sd.copy().interpolate(
            shape,
            interpolator=CubicSplineInterpolator,
            interpolator_kwargs=interpolator_kwargs,
        )

        sd = SpectralDistribution(DATA_SAMPLE)
        shape = colour.SpectralShape(500, 600, 1)
        sd_reshaped = reshape_sd(sd, shape, method="Trim")
        assert sd_reshaped == sd.copy().trim(shape)

        if is_caching_enabled():
            assert reshape_sd(sd, shape, method="Trim", copy=False) is sd_reshaped


class TestSdsAndMdsToSds:
    """
    Define :func:`colour.colorimetry.spectrum.sds_and_msds_to_sds` definition
    unit tests methods.
    """

    def test_sds_and_msds_to_sds(self) -> None:
        """
        Test :func:`colour.colorimetry.spectrum.sds_and_msds_to_sds`
        definition.
        """

        sd_1 = SpectralDistribution(DATA_SAMPLE_ABRIDGED)
        sd_2 = SpectralDistribution(DATA_SAMPLE_ABRIDGED)
        multi_sds_1 = MultiSpectralDistributions(DATA_MULTI_SAMPLE_ABRIDGED)
        multi_sds_2 = MultiSpectralDistributions(DATA_MULTI_SAMPLE_ABRIDGED)

        assert sds_and_msds_to_sds(sd_1) == [sd_1]
        assert (
            len(
                sds_and_msds_to_sds(
                    [
                        sd_1,
                        sd_2,
                        multi_sds_1,
                        multi_sds_2,
                    ]
                )
            )
            == 8
        )

        assert len(sds_and_msds_to_sds(multi_sds_1)) == 3


class TestSdsAndMsdsToMsds:
    """
    Define :func:`colour.colorimetry.spectrum.sds_and_msds_to_msds`
    definition unit tests methods.
    """

    def test_sds_and_msds_to_msds(self) -> None:
        """
        Test :func:`colour.colorimetry.spectrum.sds_and_msds_to_msds`
        definition.
        """

        sd_1 = SpectralDistribution(DATA_SAMPLE_ABRIDGED)
        sd_2 = SpectralDistribution(DATA_SAMPLE_ABRIDGED)
        multi_sds_1 = MultiSpectralDistributions(DATA_MULTI_SAMPLE_ABRIDGED)
        multi_sds_2 = MultiSpectralDistributions(DATA_MULTI_SAMPLE_ABRIDGED)

        assert len(sds_and_msds_to_msds(sd_1)) == 6

        assert sds_and_msds_to_msds(multi_sds_1) == multi_sds_1

        multi_sds_0 = sds_and_msds_to_msds([multi_sds_1])
        np.testing.assert_array_equal(multi_sds_0.range, multi_sds_1.range)
        assert sds_and_msds_to_msds([multi_sds_1]) == multi_sds_1

        shape = SpectralShape(500, 560, 10)
        assert (
            sds_and_msds_to_msds([sd_1, sd_2, multi_sds_1, multi_sds_2]).shape == shape
        )

        np.testing.assert_allclose(
            sds_and_msds_to_msds([sd_1, sd_2, multi_sds_1, multi_sds_2]).wavelengths,
            shape.wavelengths,
            atol=TOLERANCE_ABSOLUTE_TESTS,
        )

        np.testing.assert_allclose(
            sds_and_msds_to_msds([sd_1, sd_2, multi_sds_1, multi_sds_2]).values,
            tstack(
                [sd_1.align(shape).values, sd_2.align(shape).values]
                + [sd.values for sd in sds_and_msds_to_sds(multi_sds_1.align(shape))]
                + [sd.values for sd in sds_and_msds_to_sds(multi_sds_2.align(shape))]
            ),
            atol=TOLERANCE_ABSOLUTE_TESTS,
        )
