///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoTableObject.cc 
// ------------------
// Cego table object class implementation
//     
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2019 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoTableObject
// 
// Description: Cego table object structure
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// cego includes
#include "CegoTableObject.h"
#include "CegoXMLHelper.h"
#include "CegoTypeConverter.h"
#include "CegoXMLdef.h"

#include <string.h>
#include <stdlib.h>

CegoTableObject::CegoTableObject()
{
    _subCOList.Insert(this);
}

CegoTableObject::CegoTableObject(Element *pTO)
{
    putElement(pTO);
    _subCOList.Insert(this);
}

CegoTableObject::CegoTableObject(const CegoTableObject& oe) : CegoContentObject(oe)
{
    _pageId = oe._pageId;
    _lastPageId = oe._lastPageId;
    _pLastPageId = oe._pLastPageId;
    _maxFid = oe._maxFid;
    _subCOList.Insert(this);
}

CegoTableObject::CegoTableObject(int tabSetId, CegoObject::ObjectType type, const Chain& objName ) : CegoContentObject(tabSetId, type, objName)
{
    _pageId = 0;
    _lastPageId = 0;
    _pLastPageId=0;
    _maxFid = 0;
    _subCOList.Insert(this);
}

CegoTableObject::CegoTableObject(int tabSetId, CegoObject::ObjectType type,  const Chain& objName, const ListT<CegoField>& schema, const Chain& tabName) : CegoContentObject(tabSetId, type, objName, tabName, schema)
{
    _pageId = 0;    
    _lastPageId = 0;
    _pLastPageId=0;
    _maxFid = 0;
    _subCOList.Insert(this);
}

CegoTableObject::~CegoTableObject()
{   
}

void CegoTableObject::setSchema(const ListT<CegoField>& schema)
{
    _schema = schema;
}

void CegoTableObject::setDataPageId(PageIdType pageId)
{
    _pageId = pageId;
}

PageIdType CegoTableObject::getDataPageId() const
{
    return _pageId;
}

void CegoTableObject::setLastDataPageId(PageIdType pageId)
{
    _lastPageId = pageId;
}

PageIdType CegoTableObject::getLastDataPageId() const
{
    return _lastPageId;
}

void CegoTableObject::setRefLastDataPageId(PageIdType pageId)
{
    _lastPageId = pageId;
    if (_pLastPageId)
    {
	memcpy(_pLastPageId, &pageId, sizeof(PageIdType));
    }
}

PageIdType CegoTableObject::getRefLastDataPageId() const
{
    if (_pLastPageId)
    {
	PageIdType pageId;
	memcpy(&pageId, _pLastPageId, sizeof(PageIdType));
	return pageId;
    }
    else
    {
	return _lastPageId;
    }
}

void CegoTableObject::setMaxFid(int maxFid)
{
    _maxFid = maxFid;
}

int CegoTableObject::getMaxFid() const
{
    return _maxFid;
}

int CegoTableObject::getEntrySize() const
{    
    int entrySize = CegoContentObject::getBaseContentSize();
    entrySize = entrySize + sizeof(PageIdType); // pageId       
    entrySize = entrySize + sizeof(PageIdType); // last pageId
    entrySize = entrySize + sizeof(int); // maxFid    
    return entrySize;
}

void CegoTableObject::encode(char *buf)
{
    char* bufPtr = buf;

    int entrySize = getEntrySize();

    CegoContentObject::encodeBaseContent(bufPtr, entrySize);
    bufPtr += CegoContentObject::getBaseContentSize();

    memcpy(bufPtr, &_pageId, sizeof(PageIdType));
    bufPtr = bufPtr + sizeof(PageIdType);

    _pLastPageId = (char*)bufPtr;
    memcpy(bufPtr, &_lastPageId, sizeof(PageIdType));
    bufPtr = bufPtr + sizeof(PageIdType);

    memcpy(bufPtr, &_maxFid, sizeof(int));
    bufPtr = bufPtr + sizeof(int);
}

void CegoTableObject::decode(char *buf)
{
    char* bufPtr = buf;

    int size;

    CegoContentObject::decodeBaseContent(bufPtr, size);
    bufPtr += CegoContentObject::getBaseContentSize();
    
    memcpy(&_pageId, bufPtr, sizeof(PageIdType));
    bufPtr = bufPtr + sizeof(PageIdType);

    _pLastPageId = (char*)bufPtr;
    memcpy(&_lastPageId, bufPtr, sizeof(PageIdType));
    bufPtr = bufPtr + sizeof(PageIdType);

    memcpy(&_maxFid, bufPtr, sizeof(int));
    bufPtr = bufPtr + sizeof(int);
}

bool CegoTableObject::isValid()
{
    return _pageId != 0;
}

CegoTableObject& CegoTableObject::operator = ( const CegoTableObject& oe) 
{
    CegoContentObject::operator=(oe);

    _pageId = oe._pageId;
    _lastPageId = oe._pageId;
    _pLastPageId = oe._pLastPageId;    
    _maxFid = oe._maxFid;

    return (*this);
}

bool CegoTableObject::operator == ( const CegoTableObject& oe)
{
    return CegoContentObject::operator==(oe);
}

Chain CegoTableObject::getId() const
{
    return getTabName() + Chain("@") + getTableSet() + Chain("@") + getName();
}

Chain CegoTableObject::toChain() const
{
    return getTabName() + Chain("@") + getTableSet() + Chain(" ") + getName();
}

Element* CegoTableObject::getElement() const
{
    Element* pRoot = new Element(XML_OBJ_ELEMENT);

    pRoot->setAttribute(XML_TSID_ATTR, Chain(getTabSetId()));
    switch ( getType() )
    {
    case CegoObject::SYSTEM:
	pRoot->setAttribute(XML_OBJTYPE_ATTR, XML_SYSOBJ_VALUE);	
	break;
    case CegoObject::TABLE:
	pRoot->setAttribute(XML_OBJTYPE_ATTR, XML_TABOBJ_VALUE);	
	break;
    case CegoObject::PAVLTREE:
	pRoot->setAttribute(XML_OBJTYPE_ATTR, XML_PIXOBJ_VALUE);	
	break;
    case CegoObject::UAVLTREE:
	pRoot->setAttribute(XML_OBJTYPE_ATTR, XML_UIXOBJ_VALUE);	
	break;
    case CegoObject::AVLTREE:
	pRoot->setAttribute(XML_OBJTYPE_ATTR, XML_IDXOBJ_VALUE);	
	break;
    case CegoObject::VIEW:
    case CegoObject::RBSEG:
    case CegoObject::CHECK:
    case CegoObject::BTREE:
    case CegoObject::PBTREE:
    case CegoObject::UBTREE:
    case CegoObject::ALIAS:
    case CegoObject::TRIGGER:
    case CegoObject::FKEY:
    case CegoObject::PROCEDURE:
    case CegoObject::JOIN:
    case CegoObject::UNDEFINED:	
	throw Exception(EXLOC, "Invalid object type");
	break;
    }

    pRoot->setAttribute(XML_OBJNAME_ATTR, getName());

    CegoField *pF = _schema.First();
    while ( pF ) 
    {
	Element *pColElement = new Element(XML_SCHEMA_ELEMENT);

	CegoXMLHelper xh;
	xh.setColInfo(pColElement, pF);

	pRoot->addContent(pColElement);
	pF = _schema.Next();
    }

    return pRoot;
}

void CegoTableObject::putElement(Element* pElement)
{
    Element *pRoot = pElement;
    
    if ( pRoot )
    {
	
	Chain objName = pRoot->getAttributeValue(XML_OBJNAME_ATTR); 
	int tabSetId = pRoot->getAttributeValue(XML_TSID_ATTR).asInteger();
	
	setName(objName);
	setTabName(objName);
	setTabSetId(tabSetId);

	Chain objType = pRoot->getAttributeValue(XML_OBJTYPE_ATTR);
	if ( objType == Chain(XML_SYSOBJ_VALUE))
	    setType(CegoObject::SYSTEM);
	else if ( objType == Chain(XML_TABOBJ_VALUE))
	    setType(CegoObject::TABLE);
	else if ( objType == Chain(XML_PIXOBJ_VALUE))
	    setType(CegoObject::PAVLTREE);
	else if ( objType == Chain(XML_UIXOBJ_VALUE))
	    setType(CegoObject::UAVLTREE);
	else if ( objType == Chain(XML_IDXOBJ_VALUE))
	    setType(CegoObject::AVLTREE);
	else if ( objType == Chain(XML_VIEWOBJ_VALUE))
	    setType(CegoObject::VIEW);
	else if ( objType == Chain(XML_FKEYOBJ_VALUE))
	    setType(CegoObject::FKEY);
	else if ( objType == Chain(XML_PROCOBJ_VALUE))
	    setType(CegoObject::PROCEDURE);
	
	ListT<Element*> colList = pRoot->getChildren(XML_SCHEMA_ELEMENT);
	
	ListT<CegoField> fl;
	Element **pCol = colList.First();
	while ( pCol ) 
	{
	    Chain colName = (*pCol)->getAttributeValue(XML_COLNAME_ATTR);
	    Chain colType = (*pCol)->getAttributeValue(XML_COLTYPE_ATTR);
	    Chain colSize = (*pCol)->getAttributeValue(XML_COLSIZE_ATTR);
	    Chain colDim = (*pCol)->getAttributeValue(XML_COLDIM_ATTR);

	    Chain colNullable = (*pCol)->getAttributeValue(XML_COLNULLABLE_ATTR);
	    Chain colDefValue = (*pCol)->getAttributeValue(XML_COLDEFVALUE_ATTR);

	    bool isNullable;
	    if ( colNullable == Chain(XML_TRUE_VALUE) ) 
		isNullable = true;
	    else
		isNullable = false;
	   
	    CegoDataType dataType = CegoTypeConverter::getTypeId(colType);

	    CegoFieldValue defValue;
	    if ( colDefValue != Chain("") )
	    {
		defValue = CegoFieldValue(dataType, colDefValue);
	    }

	    CegoField f(CegoField(objName, objName, colName, dataType, colSize.asInteger(), colDim.asInteger(), defValue, isNullable));		
	    fl.Insert(f);

	    pCol = colList.Next();
	}
	
	setSchema(fl);
    }
}

Chain CegoTableObject::getFormatted() const
{
    Chain s;
	
    int maxAttrLen = 12;
    int maxDefLen = 10;

    CegoField* pF = _schema.First();
    while (pF)
    {
	if (maxAttrLen < pF->getAttrName().length())
	    maxAttrLen = pF->getAttrName().length();
	if (maxDefLen < pF->getValue().valAsChain().length())
	    maxDefLen = pF->getValue().valAsChain().length();
	
	pF = _schema.Next();
    }
    
    s += Chain("+") + fill("-", maxAttrLen) + fill(Chain("-"), maxDefLen + 30) + Chain("+\n");
    s += Chain("| ObjectName : ") + getName() + fill(" ", maxAttrLen + maxDefLen + 17 - getName().length()) + Chain("|\n");
	
    s += Chain("| ObjectType : ");
    switch ( getType() )
    {
    case CegoObject::SYSTEM:
	s += Chain("system table    "); 
	break;
    case CegoObject::TABLE:
	s += Chain("table           "); 
	break;
    case CegoObject::AVLTREE:
	s += Chain("avltree         ");
	break;
    case CegoObject::UAVLTREE:
	s += Chain("unique avltree  ");
	break;
    case CegoObject::PAVLTREE:
	s += Chain("primary avltree ");
	break;
    case CegoObject::VIEW:
    case CegoObject::RBSEG:
    case CegoObject::CHECK:
    case CegoObject::BTREE:
    case CegoObject::PBTREE:
    case CegoObject::UBTREE:
    case CegoObject::ALIAS:
    case CegoObject::TRIGGER:
    case CegoObject::FKEY:
    case CegoObject::PROCEDURE:
    case CegoObject::JOIN:
    case CegoObject::UNDEFINED:	
	throw Exception(EXLOC, "Invalid object type");
	break;
    }

    s += fill(" ", maxAttrLen + maxDefLen + 1 ) + Chain("|\n");
    
    s += Chain("+-----------") + fill("-", maxAttrLen-10) + Chain("+---------------+") + fill("-", maxDefLen + 1) + Chain("+----------+\n");
    s += Chain("| Attribute ") + fill(" ", maxAttrLen-10) + Chain("| Type          | Default ") + fill(" ", maxDefLen - 8) + Chain("| Nullable |\n");
    s += Chain("+-----------") + fill("-", maxAttrLen-10) + Chain("+---------------+") + fill("-", maxDefLen + 1) + Chain("+----------+\n");
	
    pF = _schema.First();
    while (pF)
    {	
	int num = maxAttrLen - pF->getAttrName().length() ;
	s += Chain("| ") + pF->getAttrName() + fill(" ", num) + Chain(" |");
	    	    
	switch (pF->getType())
	{
	case INT_TYPE:
	{
	    s += Chain("  int          |");
	    break;
	}
	case LONG_TYPE:
	{
	    s +=  Chain("  long         |");
	    break;
	}
	case BOOL_TYPE:
	{
	    s += Chain("  bool         |");
	    break;
	}
	case DATETIME_TYPE:
	{
	    s += Chain("  datetime     |");
	    break;
	}
	case VARCHAR_TYPE:
	{
	    Chain l(pF->getLength());
	    s += Chain("  string[") + l + Chain("]") + fill(" ", 6 - l.length()) + Chain("|");
	    break;
	}
	case BIGINT_TYPE:
	{
	    Chain l(pF->getLength());
	    s += Chain("  bigint[") + l + Chain("]") + fill(" ", 6 - l.length()) + Chain("|");
	    break;
	}
	case SMALLINT_TYPE:
	{
	    s += Chain("  smallint     |");
	    break;
	}
	case TINYINT_TYPE:
	{
	    s += Chain("  tinyint      |");
	    break;
	}	
	case DECIMAL_TYPE:
	{
	    Chain l(pF->getLength());
	    s += Chain("  decimal[") + l + Chain("]") + fill(" ", 5 - l.length()) + Chain("|");
	    break;
	}
	case FIXED_TYPE:
	{
	    Chain l(pF->getLength());
	    s += Chain("  fixed[") + l + Chain("]") + fill(" ", 7 - l.length()) + Chain("|");
	    break;
	}
	case FLOAT_TYPE:
	{
	    s += Chain(" float         |");
	    break;
	}
	case DOUBLE_TYPE:
	{
	    s += Chain(" double        |");
	    break;
	}
	case BLOB_TYPE:
	{
	    s += Chain(" blob          |");
	    break;
	}
	case CLOB_TYPE:
	{
	    s += Chain(" clob          |");
	    break;
	}
	case NULL_TYPE:
	case PAGEID_TYPE:
	    break;
	}
	
	Chain defVal = pF->getValue().valAsChain();
	
	s += Chain(" ") + defVal + fill(" ", maxDefLen - defVal.length()) + Chain(" |");
	
	if ( pF->isNullable() )
	    s += Chain(" y        |");
	else
	    s += Chain(" n        |");
	
	pF = _schema.Next();
	    
	s += "\n";
    }
    s += Chain("+") + fill("-", maxAttrLen + 1) + Chain("+---------------+") + fill("-", maxDefLen + 1) + Chain("+----------+\n");

    return s;
}

Chain CegoTableObject::fill(const Chain& s, int num) const
{
    Chain fs = Chain("");
    while (num > 0)
    {
	fs = fs + s;
	num--;
    }

    return fs;
}

CegoContentObject* CegoTableObject::clone(bool isAttrRef)
{
    CegoTableObject *pClone = new CegoTableObject(*this);
    return (pClone);
}
