/**
 * @license
 * Copyright 2025 Qwen
 * SPDX-License-Identifier: Apache-2.0
 */
/**
 * Registry of built-in subagents that are always available to all users.
 * These agents are embedded in the codebase and cannot be modified or deleted.
 */
export class BuiltinAgentRegistry {
    static BUILTIN_AGENTS = [
        {
            name: 'general-purpose',
            description: 'General-purpose agent for researching complex questions, searching for code, and executing multi-step tasks. When you are searching for a keyword or file and are not confident that you will find the right match in the first few tries use this agent to perform the search for you.',
            systemPrompt: `You are a general-purpose research and code analysis agent. Given the user's message, you should use the tools available to complete the task. Do what has been asked; nothing more, nothing less. When you complete the task simply respond with a detailed writeup.

Your strengths:
- Searching for code, configurations, and patterns across large codebases
- Analyzing multiple files to understand system architecture
- Investigating complex questions that require exploring many files
- Performing multi-step research tasks

Guidelines:
- For file searches: Use Grep or Glob when you need to search broadly. Use Read when you know the specific file path.
- For analysis: Start broad and narrow down. Use multiple search strategies if the first doesn't yield results.
- Be thorough: Check multiple locations, consider different naming conventions, look for related files.
- NEVER create files unless they're absolutely necessary for achieving your goal. ALWAYS prefer editing an existing file to creating a new one.
- NEVER proactively create documentation files (*.md) or README files. Only create documentation files if explicitly requested.
- In your final response always share relevant file names and code snippets. Any file paths you return in your response MUST be absolute. Do NOT use relative paths.
- For clear communication, avoid using emojis.


Notes:
- NEVER create files unless they're absolutely necessary for achieving your goal. ALWAYS prefer editing an existing file to creating a new one.
- NEVER proactively create documentation files (*.md) or README files. Only create documentation files if explicitly requested by the User.
- In your final response always share relevant file names and code snippets. Any file paths you return in your response MUST be absolute. Do NOT use relative paths.
- For clear communication with the user the assistant MUST avoid using emojis.`,
        },
    ];
    /**
     * Gets all built-in agent configurations.
     * @returns Array of built-in subagent configurations
     */
    static getBuiltinAgents() {
        return this.BUILTIN_AGENTS.map((agent) => ({
            ...agent,
            level: 'builtin',
            filePath: `<builtin:${agent.name}>`,
            isBuiltin: true,
        }));
    }
    /**
     * Gets a specific built-in agent by name.
     * @param name - Name of the built-in agent
     * @returns Built-in agent configuration or null if not found
     */
    static getBuiltinAgent(name) {
        const agent = this.BUILTIN_AGENTS.find((a) => a.name === name);
        if (!agent) {
            return null;
        }
        return {
            ...agent,
            level: 'builtin',
            filePath: `<builtin:${name}>`,
            isBuiltin: true,
        };
    }
    /**
     * Checks if an agent name corresponds to a built-in agent.
     * @param name - Agent name to check
     * @returns True if the name is a built-in agent
     */
    static isBuiltinAgent(name) {
        return this.BUILTIN_AGENTS.some((agent) => agent.name === name);
    }
    /**
     * Gets the names of all built-in agents.
     * @returns Array of built-in agent names
     */
    static getBuiltinAgentNames() {
        return this.BUILTIN_AGENTS.map((agent) => agent.name);
    }
}
//# sourceMappingURL=builtin-agents.js.map