/* This entire file is licensed under GNU General Public License v3.0
 *
 * Copyright 2020- sfwbar maintainers
 */

#include "sfwbar.h"
#include "flowgrid.h"
#include "taskbaritem.h"
#include "taskbarpopup.h"
#include "taskbarpager.h"
#include "taskbar.h"
#include "wintree.h"
#include "config.h"

G_DEFINE_TYPE_WITH_CODE (Taskbar, taskbar, BASE_WIDGET_TYPE, G_ADD_PRIVATE (Taskbar));

static GList *taskbars;

static GtkWidget *taskbar_get_child ( GtkWidget *self )
{
  TaskbarPrivate *priv;

  g_return_val_if_fail(IS_TASKBAR(self),NULL);
  priv = taskbar_get_instance_private(TASKBAR(self));

  return priv->taskbar;
}

static GtkWidget *taskbar_mirror ( GtkWidget *src )
{
  GtkWidget *self;
  TaskbarPrivate *dpriv, *spriv;

  g_return_val_if_fail(IS_TASKBAR(src),NULL);
  spriv = taskbar_get_instance_private(TASKBAR(src));

  self = taskbar_new(TRUE);
  dpriv = taskbar_get_instance_private(TASKBAR(self));
  dpriv->filter = spriv->filter;
  dpriv->floating_filter = spriv->floating_filter;
  dpriv->grouping = spriv->grouping;
  g_object_set_data(G_OBJECT(self),"title_width",
      g_object_get_data(G_OBJECT(src),"title_width"));
  g_object_set_data(G_OBJECT(self),"g_cols",
      g_object_get_data(G_OBJECT(src),"g_cols"));
  g_object_set_data(G_OBJECT(self),"g_rows",
      g_object_get_data(G_OBJECT(src),"g_rows"));
  g_object_set_data(G_OBJECT(self),"g_icons",
      g_object_get_data(G_OBJECT(src),"g_icons"));
  g_object_set_data(G_OBJECT(self),"g_labels",
      g_object_get_data(G_OBJECT(src),"g_labels"));
  g_object_set_data_full(G_OBJECT(self),"g_css",
      g_strdup(g_object_get_data(G_OBJECT(src),"g_css")),g_free);
  g_object_set_data_full(G_OBJECT(self),"g_style",
      g_strdup(g_object_get_data(G_OBJECT(src),"g_style")),g_free);
  g_object_set_data(G_OBJECT(self),"g_title_width",
      g_object_get_data(G_OBJECT(src),"g_title_width"));
  g_object_set_data(G_OBJECT(self),"g_sort",
      g_object_get_data(G_OBJECT(src),"g_sort"));
  g_object_set_data(G_OBJECT(self),"title_width",
      g_object_get_data(G_OBJECT(src),"title_width"));
  flow_grid_copy_properties(self, src);
  base_widget_copy_properties(self,src);
  taskbar_populate();

  return self;
}

static void taskbar_destroy ( GtkWidget *self )
{
  taskbars = g_list_remove(taskbars,self);
  GTK_WIDGET_CLASS(taskbar_parent_class)->destroy(self);
}

static void taskbar_class_init ( TaskbarClass *kclass )
{
  BASE_WIDGET_CLASS(kclass)->get_child = taskbar_get_child;
  BASE_WIDGET_CLASS(kclass)->mirror = taskbar_mirror;
  BASE_WIDGET_CLASS(kclass)->action_exec = NULL;
  GTK_WIDGET_CLASS(kclass)->destroy = taskbar_destroy;
}

static void taskbar_init ( Taskbar *self )
{
  TaskbarPrivate *priv;
  action_t *action;

  priv = taskbar_get_instance_private(TASKBAR(self));
  priv->taskbar = flow_grid_new(TRUE);
  gtk_container_add(GTK_CONTAINER(self),priv->taskbar);
  flow_grid_invalidate(priv->taskbar);
  action = action_new();
  action->quark = g_quark_from_static_string("taskbaritemdefault");
  base_widget_set_action(GTK_WIDGET(self), 1, 0, action);
}

GtkWidget *taskbar_new ( gboolean toplevel )
{
  GtkWidget *self;

  self = GTK_WIDGET(g_object_new(taskbar_get_type(), NULL));
  if(toplevel)
    taskbars = g_list_prepend(taskbars, self);

  return self;
}

gpointer taskbar_group_id ( GtkWidget *self, window_t *win )
{
  TaskbarPrivate *priv;

  g_return_val_if_fail(IS_TASKBAR(self), NULL);
  priv = taskbar_get_instance_private(TASKBAR(self));

  switch(priv->grouping)
  {
    case TASKBAR_POPUP:
      return win->appid;
    case TASKBAR_DESK:
      return workspace_from_id(win->workspace);
  }
  return NULL;
}

void taskbar_set_grouping ( GtkWidget *self, gint grouping )
{
  TaskbarPrivate *priv;

  g_return_if_fail(IS_TASKBAR(self));
  priv = taskbar_get_instance_private(TASKBAR(self));

  priv->grouping = grouping;
}

gpointer taskbar_holder_new ( GtkWidget *self, window_t *win )
{
  TaskbarPrivate *priv;

  g_return_val_if_fail(IS_TASKBAR(self), NULL);
  priv = taskbar_get_instance_private(TASKBAR(self));

  switch(priv->grouping)
  {
    case TASKBAR_POPUP:
      return taskbar_popup_new(win->appid, self);
    case TASKBAR_DESK:
      return taskbar_pager_new(workspace_from_id(win->workspace), self);
  }
  return NULL;
}

GtkWidget *taskbar_holder_get ( GtkWidget *self, window_t *win, gboolean new )
{
  TaskbarPrivate *priv;
  GtkWidget *taskbar, *holder;

  g_return_val_if_fail(IS_TASKBAR(self), NULL);
  priv = taskbar_get_instance_private(TASKBAR(self));
  if(!priv->grouping)
    return self;

  holder = flow_grid_find_child(self, taskbar_group_id(self, win));
  taskbar = holder?base_widget_get_child(holder):NULL;
  if(!taskbar && new)
    taskbar = taskbar_holder_new(self, win);

  return taskbar;
}

void taskbar_set_filter ( GtkWidget *self, gint filter )
{
  TaskbarPrivate *priv;

  g_return_if_fail(IS_TASKBAR(self));
  priv = taskbar_get_instance_private(TASKBAR(self));

  if(filter == G_TOKEN_FLOATING)
    priv->floating_filter = TRUE;
  else
    priv->filter = filter;
}

gint taskbar_get_filter ( GtkWidget *self, gboolean *floating )
{
  TaskbarPrivate *priv;

  g_return_val_if_fail(IS_TASKBAR(self),0);
  priv = taskbar_get_instance_private(TASKBAR(self));

  *floating = priv->floating_filter;
  return priv->filter;
}

void taskbar_invalidate_item ( window_t *win )
{
  GList *iter;
  GtkWidget *taskbar;

  for(iter=taskbars; iter; iter=g_list_next(iter))
  {
    taskbar = taskbar_holder_get(iter->data, win, FALSE);
    if(taskbar)
      flow_item_invalidate(flow_grid_find_child(taskbar, win));
    if(taskbar!=iter->data)
      flow_item_invalidate(
          flow_grid_find_child(iter->data, taskbar_group_id(iter->data, win)));
  }
}

void taskbar_invalidate_all ( void )
{
  GList *iter;

  for(iter=wintree_get_list();iter;iter=g_list_next(iter))
    taskbar_invalidate_item(iter->data);
}

void taskbar_init_item ( window_t *win )
{
  GList *iter;

  for(iter=taskbars; iter; iter=g_list_next(iter))
    taskbar_item_new(win, taskbar_holder_get(iter->data, win, TRUE));
}

void taskbar_destroy_item ( window_t *win )
{
  GList *iter;
  GtkWidget *taskbar;

  for(iter=taskbars; iter; iter=g_list_next(iter))
    if( (taskbar = taskbar_holder_get(iter->data, win, FALSE)) )
    {
      flow_grid_delete_child(taskbar, win);
      if(!flow_grid_n_children(taskbar) && taskbar!=iter->data)
        flow_grid_delete_child(iter->data, taskbar_group_id(iter->data, win));
      taskbar_invalidate_item(win);
    }
}

void taskbar_populate ( void )
{
  GList *iter;

  for(iter=wintree_get_list(); iter; iter=g_list_next(iter))
    taskbar_init_item (iter->data);
}

void taskbar_update_all ( void )
{
  GList *iter;

  for(iter=taskbars; iter; iter=g_list_next(iter))
    flow_grid_update(iter->data);
}
