/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.arrow.vector;

import java.util.concurrent.TimeUnit;
import org.apache.arrow.memory.BoundsChecking;
import org.apache.arrow.memory.BufferAllocator;
import org.apache.arrow.memory.RootAllocator;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.TearDown;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

/** Benchmarks for {@link Float8Vector}. */
@State(Scope.Benchmark)
public class Float8Benchmarks {
  // checkstyle:off: MissingJavadocMethod

  private static final int VECTOR_LENGTH = 1024;

  private static final int ALLOCATOR_CAPACITY = 1024 * 1024;

  private BufferAllocator allocator;

  private Float8Vector vector;

  private Float8Vector fromVector;

  /** Setup benchmarks. */
  @Setup
  public void prepare() {
    allocator = new RootAllocator(ALLOCATOR_CAPACITY);
    vector = new Float8Vector("vector", allocator);
    vector.allocateNew(VECTOR_LENGTH);

    fromVector = new Float8Vector("vector", allocator);
    fromVector.allocateNew(VECTOR_LENGTH);

    for (int i = 0; i < VECTOR_LENGTH; i++) {
      if (i % 3 == 0) {
        fromVector.setNull(i);
      } else {
        fromVector.set(i, i * i);
      }
    }
    fromVector.setValueCount(VECTOR_LENGTH);
  }

  /** Tear down benchmarks. */
  @TearDown
  public void tearDown() {
    vector.close();
    fromVector.close();
    allocator.close();
  }

  /**
   * Test reading/writing on {@link Float8Vector}. The performance of this benchmark is influenced
   * by the states of two flags: 1. The flag for boundary checking. For details, please see {@link
   * BoundsChecking}. 2. The flag for null checking in get methods. For details, please see {@link
   * NullCheckingForGet}.
   *
   * @return useless. To avoid DCE by JIT.
   */
  @Benchmark
  @BenchmarkMode(Mode.AverageTime)
  @OutputTimeUnit(TimeUnit.MICROSECONDS)
  public double readWriteBenchmark() {
    double sum = 0;
    for (int i = 0; i < VECTOR_LENGTH; i++) {
      vector.set(i, i + 10.0);
      sum += vector.get(i);
    }
    return sum;
  }

  @Benchmark
  @BenchmarkMode(Mode.AverageTime)
  @OutputTimeUnit(TimeUnit.MICROSECONDS)
  public void copyFromBenchmark() {
    for (int i = 0; i < VECTOR_LENGTH; i++) {
      vector.copyFrom(i, i, (Float8Vector) fromVector);
    }
  }

  public static void main(String[] args) throws RunnerException {
    Options opt =
        new OptionsBuilder().include(Float8Benchmarks.class.getSimpleName()).forks(1).build();

    new Runner(opt).run();
  }
  // checkstyle:on: MissingJavadocMethod
}
