"""
Munsell Renotation System Dataset - 1929 Munsell Colours
========================================================

Define the *Munsell Renotation System* *1929* datasets. Those are the colours
appearing in the 1929 *Munsell Book of Color*. These data has been used in the
scaling experiments leading to the 1943 renotation.

Notes
-----
-   The Munsell Renotation data commonly available within the *all.dat*,
    *experimental.dat* and *real.dat* files features *CIE xyY* colourspace
    values that are scaled by a :math:`1 / 0.975 \\simeq 1.02568` factor. If
    you are performing conversions using *Munsell* *Colorlab* specification,
    e.g., *2.5R 9/2*, according to *ASTM D1535-08e1* method, you should not
    scale the output :math:`Y` Luminance. However, if you use directly the
    *CIE xyY* colourspace values from the Munsell Renotation data, you should
    scale the :math:`Y` Luminance before conversions by a :math:`0.975` factor.

    *ASTM D1535-08e1* states that::

        The coefficients of this equation are obtained from the 1943 equation
        by multiplying each coefficient by 0.975, the reflectance factor of
        magnesium oxide with respect to the perfect reflecting diffuser, and
        rounding to ve digits of precision.

-   Chromaticities assume *CIE Illuminant C*, approximately 6700K, as neutral
    origin for both the hue and chroma loci.

References
----------
-   :cite:`MunsellColorSciencec` : Munsell Color Science. (n.d.). Munsell
    Colours Data. Retrieved August 20, 2014, from
    http://www.cis.rit.edu/research/mcsl2/online/munsell.php
"""

from __future__ import annotations

import numpy as np

__author__ = "Colour Developers"
__copyright__ = "Copyright 2013 Colour Developers"
__license__ = "BSD-3-Clause - https://opensource.org/licenses/BSD-3-Clause"
__maintainer__ = "Colour Developers"
__email__ = "colour-developers@colour-science.org"
__status__ = "Production"

__all__ = [
    "MUNSELL_COLOURS_1929",
]


MUNSELL_COLOURS_1929: tuple = (
    (("10RP", 2.0, 2.0), np.array([0.3532, 0.2957, 3.1260])),
    (("10RP", 2.0, 4.0), np.array([0.3850, 0.2778, 3.1260])),
    (("10RP", 2.0, 6.0), np.array([0.4139, 0.2608, 3.1260])),
    (("10RP", 2.0, 8.0), np.array([0.4428, 0.2419, 3.1260])),
    (("2.5R", 2.0, 2.0), np.array([0.3614, 0.3033, 3.1260])),
    (("2.5R", 2.0, 4.0), np.array([0.4021, 0.2900, 3.1260])),
    (("2.5R", 2.0, 6.0), np.array([0.4390, 0.2760, 3.1260])),
    (("5R", 2.0, 2.0), np.array([0.3692, 0.3111, 3.1260])),
    (("5R", 2.0, 4.0), np.array([0.4184, 0.3032, 3.1260])),
    (("5R", 2.0, 6.0), np.array([0.4642, 0.2934, 3.1260])),
    (("7.5R", 2.0, 2.0), np.array([0.3751, 0.3181, 3.1260])),
    (("7.5R", 2.0, 4.0), np.array([0.4335, 0.3169, 3.1260])),
    (("10R", 2.0, 2.0), np.array([0.3811, 0.3274, 3.1260])),
    (("2.5YR", 2.0, 2.0), np.array([0.3852, 0.3365, 3.1260])),
    (("5YR", 2.0, 2.0), np.array([0.3880, 0.3476, 3.1260])),
    (("7.5YR", 2.0, 2.0), np.array([0.3889, 0.3590, 3.1260])),
    (("10YR", 2.0, 2.0), np.array([0.3872, 0.3688, 3.1260])),
    (("2.5Y", 2.0, 2.0), np.array([0.3825, 0.3785, 3.1260])),
    (("5Y", 2.0, 2.0), np.array([0.3757, 0.3839, 3.1260])),
    (("7.5Y", 2.0, 2.0), np.array([0.3660, 0.3858, 3.1260])),
    (("10Y", 2.0, 2.0), np.array([0.3556, 0.3848, 3.1260])),
    (("2.5GY", 2.0, 2.0), np.array([0.3421, 0.3803, 3.1260])),
    (("5GY", 2.0, 2.0), np.array([0.3309, 0.3743, 3.1260])),
    (("7.5GY", 2.0, 2.0), np.array([0.3165, 0.3650, 3.1260])),
    (("10GY", 2.0, 2.0), np.array([0.3069, 0.3580, 3.1260])),
    (("2.5G", 2.0, 2.0), np.array([0.2978, 0.3507, 3.1260])),
    (("5G", 2.0, 2.0), np.array([0.2918, 0.3450, 3.1260])),
    (("7.5G", 2.0, 2.0), np.array([0.2869, 0.3400, 3.1260])),
    (("10G", 2.0, 2.0), np.array([0.2820, 0.3341, 3.1260])),
    (("2.5BG", 2.0, 2.0), np.array([0.2765, 0.3271, 3.1260])),
    (("2.5BG", 2.0, 4.0), np.array([0.2343, 0.3378, 3.1260])),
    (("5BG", 2.0, 2.0), np.array([0.2697, 0.3175, 3.1260])),
    (("5BG", 2.0, 4.0), np.array([0.2234, 0.3150, 3.1260])),
    (("7.5BG", 2.0, 2.0), np.array([0.2651, 0.3098, 3.1260])),
    (("7.5BG", 2.0, 4.0), np.array([0.2162, 0.2981, 3.1260])),
    (("10BG", 2.0, 2.0), np.array([0.2606, 0.3010, 3.1260])),
    (("2.5B", 2.0, 2.0), np.array([0.2578, 0.2940, 3.1260])),
    (("5B", 2.0, 2.0), np.array([0.2559, 0.2874, 3.1260])),
    (("7.5B", 2.0, 2.0), np.array([0.2545, 0.2799, 3.1260])),
    (("7.5B", 2.0, 4.0), np.array([0.2063, 0.2400, 3.1260])),
    (("10B", 2.0, 2.0), np.array([0.2558, 0.2725, 3.1260])),
    (("10B", 2.0, 4.0), np.array([0.2102, 0.2313, 3.1260])),
    (("2.5PB", 2.0, 2.0), np.array([0.2592, 0.2675, 3.1260])),
    (("2.5PB", 2.0, 4.0), np.array([0.2175, 0.2245, 3.1260])),
    (("2.5PB", 2.0, 6.0), np.array([0.1825, 0.1857, 3.1260])),
    (("5PB", 2.0, 2.0), np.array([0.2638, 0.2624, 3.1260])),
    (("5PB", 2.0, 4.0), np.array([0.2263, 0.2192, 3.1260])),
    (("5PB", 2.0, 6.0), np.array([0.1942, 0.1811, 3.1260])),
    (("7.5PB", 2.0, 2.0), np.array([0.2712, 0.2582, 3.1260])),
    (("7.5PB", 2.0, 4.0), np.array([0.2420, 0.2148, 3.1260])),
    (("7.5PB", 2.0, 6.0), np.array([0.2189, 0.1790, 3.1260])),
    (("10PB", 2.0, 2.0), np.array([0.2803, 0.2567, 3.1260])),
    (("10PB", 2.0, 4.0), np.array([0.2600, 0.2162, 3.1260])),
    (("10PB", 2.0, 6.0), np.array([0.2440, 0.1840, 3.1260])),
    (("2.5P", 2.0, 2.0), np.array([0.2892, 0.2583, 3.1260])),
    (("2.5P", 2.0, 4.0), np.array([0.2758, 0.2208, 3.1260])),
    (("2.5P", 2.0, 6.0), np.array([0.2661, 0.1921, 3.1260])),
    (("5P", 2.0, 2.0), np.array([0.2984, 0.2612, 3.1260])),
    (("5P", 2.0, 4.0), np.array([0.2908, 0.2261, 3.1260])),
    (("5P", 2.0, 6.0), np.array([0.2850, 0.1992, 3.1260])),
    (("7.5P", 2.0, 2.0), np.array([0.3071, 0.2647, 3.1260])),
    (("7.5P", 2.0, 4.0), np.array([0.3048, 0.2321, 3.1260])),
    (("7.5P", 2.0, 6.0), np.array([0.3025, 0.2058, 3.1260])),
    (("10P", 2.0, 2.0), np.array([0.3161, 0.2691, 3.1260])),
    (("10P", 2.0, 4.0), np.array([0.3189, 0.2390, 3.1260])),
    (("10P", 2.0, 6.0), np.array([0.3207, 0.2132, 3.1260])),
    (("2.5RP", 2.0, 2.0), np.array([0.3279, 0.2754, 3.1260])),
    (("2.5RP", 2.0, 4.0), np.array([0.3382, 0.2496, 3.1260])),
    (("2.5RP", 2.0, 6.0), np.array([0.3470, 0.2259, 3.1260])),
    (("5RP", 2.0, 2.0), np.array([0.3383, 0.2829, 3.1260])),
    (("5RP", 2.0, 4.0), np.array([0.3558, 0.2597, 3.1260])),
    (("5RP", 2.0, 6.0), np.array([0.3708, 0.2380, 3.1260])),
    (("7.5RP", 2.0, 2.0), np.array([0.3459, 0.2892, 3.1260])),
    (("7.5RP", 2.0, 4.0), np.array([0.3702, 0.2683, 3.1260])),
    (("7.5RP", 2.0, 6.0), np.array([0.3918, 0.2490, 3.1260])),
    (("10RP", 3.0, 2.0), np.array([0.3526, 0.3068, 6.5550])),
    (("10RP", 3.0, 4.0), np.array([0.3889, 0.2969, 6.5550])),
    (("10RP", 3.0, 6.0), np.array([0.4218, 0.2864, 6.5550])),
    (("10RP", 3.0, 8.0), np.array([0.4552, 0.2741, 6.5550])),
    (("10RP", 3.0, 10.0), np.array([0.4851, 0.2618, 6.5550])),
    (("2.5R", 3.0, 2.0), np.array([0.3591, 0.3130, 6.5550])),
    (("2.5R", 3.0, 4.0), np.array([0.4021, 0.3076, 6.5550])),
    (("2.5R", 3.0, 6.0), np.array([0.4409, 0.3009, 6.5550])),
    (("2.5R", 3.0, 8.0), np.array([0.4821, 0.2918, 6.5550])),
    (("2.5R", 3.0, 10.0), np.array([0.5191, 0.2811, 6.5550])),
    (("5R", 3.0, 2.0), np.array([0.3645, 0.3190, 6.5550])),
    (("5R", 3.0, 4.0), np.array([0.4148, 0.3190, 6.5550])),
    (("5R", 3.0, 6.0), np.array([0.4592, 0.3168, 6.5550])),
    (("5R", 3.0, 8.0), np.array([0.5064, 0.3114, 6.5550])),
    (("5R", 3.0, 10.0), np.array([0.5500, 0.3024, 6.5550])),
    (("7.5R", 3.0, 2.0), np.array([0.3690, 0.3248, 6.5550])),
    (("7.5R", 3.0, 4.0), np.array([0.4240, 0.3302, 6.5550])),
    (("7.5R", 3.0, 6.0), np.array([0.4738, 0.3316, 6.5550])),
    (("7.5R", 3.0, 8.0), np.array([0.5251, 0.3297, 6.5550])),
    (("10R", 3.0, 2.0), np.array([0.3728, 0.3314, 6.5550])),
    (("10R", 3.0, 4.0), np.array([0.4308, 0.3412, 6.5550])),
    (("10R", 3.0, 6.0), np.array([0.4854, 0.3467, 6.5550])),
    (("2.5YR", 3.0, 2.0), np.array([0.3757, 0.3391, 6.5550])),
    (("2.5YR", 3.0, 4.0), np.array([0.4360, 0.3563, 6.5550])),
    (("2.5YR", 3.0, 6.0), np.array([0.4954, 0.3692, 6.5550])),
    (("5YR", 3.0, 2.0), np.array([0.3771, 0.3476, 6.5550])),
    (("5YR", 3.0, 4.0), np.array([0.4376, 0.3715, 6.5550])),
    (("7.5YR", 3.0, 2.0), np.array([0.3771, 0.3549, 6.5550])),
    (("7.5YR", 3.0, 4.0), np.array([0.4378, 0.3865, 6.5550])),
    (("10YR", 3.0, 2.0), np.array([0.3747, 0.3630, 6.5550])),
    (("10YR", 3.0, 4.0), np.array([0.4341, 0.4018, 6.5550])),
    (("2.5Y", 3.0, 2.0), np.array([0.3703, 0.3700, 6.5550])),
    (("2.5Y", 3.0, 4.0), np.array([0.4277, 0.4166, 6.5550])),
    (("5Y", 3.0, 2.0), np.array([0.3646, 0.3748, 6.5550])),
    (("7.5Y", 3.0, 2.0), np.array([0.3589, 0.3778, 6.5550])),
    (("10Y", 3.0, 2.0), np.array([0.3513, 0.3789, 6.5550])),
    (("2.5GY", 3.0, 2.0), np.array([0.3412, 0.3768, 6.5550])),
    (("2.5GY", 3.0, 4.0), np.array([0.3772, 0.4484, 6.5550])),
    (("5GY", 3.0, 2.0), np.array([0.3319, 0.3729, 6.5550])),
    (("5GY", 3.0, 4.0), np.array([0.3554, 0.4429, 6.5550])),
    (("7.5GY", 3.0, 2.0), np.array([0.3180, 0.3644, 6.5550])),
    (("7.5GY", 3.0, 4.0), np.array([0.3270, 0.4288, 6.5550])),
    (("10GY", 3.0, 2.0), np.array([0.3088, 0.3578, 6.5550])),
    (("10GY", 3.0, 4.0), np.array([0.3053, 0.4123, 6.5550])),
    (("2.5G", 3.0, 2.0), np.array([0.2999, 0.3500, 6.5550])),
    (("2.5G", 3.0, 4.0), np.array([0.2836, 0.3915, 6.5550])),
    (("5G", 3.0, 2.0), np.array([0.2935, 0.3439, 6.5550])),
    (("5G", 3.0, 4.0), np.array([0.2711, 0.3780, 6.5550])),
    (("7.5G", 3.0, 2.0), np.array([0.2890, 0.3391, 6.5550])),
    (("7.5G", 3.0, 4.0), np.array([0.2618, 0.3667, 6.5550])),
    (("10G", 3.0, 2.0), np.array([0.2844, 0.3337, 6.5550])),
    (("10G", 3.0, 4.0), np.array([0.2525, 0.3537, 6.5550])),
    (("2.5BG", 3.0, 2.0), np.array([0.2799, 0.3271, 6.5550])),
    (("2.5BG", 3.0, 4.0), np.array([0.2437, 0.3386, 6.5550])),
    (("2.5BG", 3.0, 6.0), np.array([0.2132, 0.3468, 6.5550])),
    (("5BG", 3.0, 2.0), np.array([0.2742, 0.3192, 6.5550])),
    (("5BG", 3.0, 4.0), np.array([0.2343, 0.3200, 6.5550])),
    (("5BG", 3.0, 6.0), np.array([0.2020, 0.3188, 6.5550])),
    (("7.5BG", 3.0, 2.0), np.array([0.2699, 0.3120, 6.5550])),
    (("7.5BG", 3.0, 4.0), np.array([0.2272, 0.3041, 6.5550])),
    (("7.5BG", 3.0, 6.0), np.array([0.1928, 0.2958, 6.5550])),
    (("10BG", 3.0, 2.0), np.array([0.2660, 0.3050, 6.5550])),
    (("10BG", 3.0, 4.0), np.array([0.2221, 0.2886, 6.5550])),
    (("10BG", 3.0, 6.0), np.array([0.1861, 0.2722, 6.5550])),
    (("2.5B", 3.0, 2.0), np.array([0.2636, 0.2983, 6.5550])),
    (("2.5B", 3.0, 4.0), np.array([0.2183, 0.2748, 6.5550])),
    (("2.5B", 3.0, 6.0), np.array([0.1826, 0.2536, 6.5550])),
    (("5B", 3.0, 2.0), np.array([0.2617, 0.2921, 6.5550])),
    (("5B", 3.0, 4.0), np.array([0.2176, 0.2632, 6.5550])),
    (("5B", 3.0, 6.0), np.array([0.1835, 0.2375, 6.5550])),
    (("7.5B", 3.0, 2.0), np.array([0.2616, 0.2857, 6.5550])),
    (("7.5B", 3.0, 4.0), np.array([0.2200, 0.2536, 6.5550])),
    (("7.5B", 3.0, 6.0), np.array([0.1875, 0.2258, 6.5550])),
    (("7.5B", 3.0, 8.0), np.array([0.1583, 0.1987, 6.5550])),
    (("10B", 3.0, 2.0), np.array([0.2631, 0.2801, 6.5550])),
    (("10B", 3.0, 4.0), np.array([0.2246, 0.2467, 6.5550])),
    (("10B", 3.0, 6.0), np.array([0.1933, 0.2173, 6.5550])),
    (("10B", 3.0, 8.0), np.array([0.1658, 0.1905, 6.5550])),
    (("2.5PB", 3.0, 2.0), np.array([0.2663, 0.2756, 6.5550])),
    (("2.5PB", 3.0, 4.0), np.array([0.2312, 0.2405, 6.5550])),
    (("2.5PB", 3.0, 6.0), np.array([0.2022, 0.2101, 6.5550])),
    (("2.5PB", 3.0, 8.0), np.array([0.1780, 0.1833, 6.5550])),
    (("2.5PB", 3.0, 10.0), np.array([0.1576, 0.1600, 6.5550])),
    (("5PB", 3.0, 2.0), np.array([0.2708, 0.2719, 6.5550])),
    (("5PB", 3.0, 4.0), np.array([0.2393, 0.2361, 6.5550])),
    (("5PB", 3.0, 6.0), np.array([0.2122, 0.2052, 6.5550])),
    (("5PB", 3.0, 8.0), np.array([0.1908, 0.1799, 6.5550])),
    (("5PB", 3.0, 10.0), np.array([0.1718, 0.1562, 6.5550])),
    (("5PB", 3.0, 12.0), np.array([0.1557, 0.1356, 6.5550])),
    (("7.5PB", 3.0, 2.0), np.array([0.2777, 0.2687, 6.5550])),
    (("7.5PB", 3.0, 4.0), np.array([0.2520, 0.2319, 6.5550])),
    (("7.5PB", 3.0, 6.0), np.array([0.2311, 0.2010, 6.5550])),
    (("7.5PB", 3.0, 8.0), np.array([0.2149, 0.1761, 6.5550])),
    (("7.5PB", 3.0, 10.0), np.array([0.2005, 0.1536, 6.5550])),
    (("10PB", 3.0, 2.0), np.array([0.2847, 0.2670, 6.5550])),
    (("10PB", 3.0, 4.0), np.array([0.2660, 0.2319, 6.5550])),
    (("10PB", 3.0, 6.0), np.array([0.2511, 0.2031, 6.5550])),
    (("10PB", 3.0, 8.0), np.array([0.2387, 0.1786, 6.5550])),
    (("10PB", 3.0, 10.0), np.array([0.2278, 0.1565, 6.5550])),
    (("2.5P", 3.0, 2.0), np.array([0.2922, 0.2680, 6.5550])),
    (("2.5P", 3.0, 4.0), np.array([0.2792, 0.2342, 6.5550])),
    (("2.5P", 3.0, 6.0), np.array([0.2691, 0.2072, 6.5550])),
    (("2.5P", 3.0, 8.0), np.array([0.2615, 0.1845, 6.5550])),
    (("2.5P", 3.0, 10.0), np.array([0.2548, 0.1638, 6.5550])),
    (("5P", 3.0, 2.0), np.array([0.2997, 0.2700, 6.5550])),
    (("5P", 3.0, 4.0), np.array([0.2928, 0.2386, 6.5550])),
    (("5P", 3.0, 6.0), np.array([0.2870, 0.2135, 6.5550])),
    (("5P", 3.0, 8.0), np.array([0.2819, 0.1910, 6.5550])),
    (("5P", 3.0, 10.0), np.array([0.2772, 0.1707, 6.5550])),
    (("7.5P", 3.0, 2.0), np.array([0.3088, 0.2740, 6.5550])),
    (("7.5P", 3.0, 4.0), np.array([0.3072, 0.2448, 6.5550])),
    (("7.5P", 3.0, 6.0), np.array([0.3057, 0.2208, 6.5550])),
    (("7.5P", 3.0, 8.0), np.array([0.3037, 0.1981, 6.5550])),
    (("7.5P", 3.0, 10.0), np.array([0.3020, 0.1794, 6.5550])),
    (("10P", 3.0, 2.0), np.array([0.3170, 0.2790, 6.5550])),
    (("10P", 3.0, 4.0), np.array([0.3214, 0.2517, 6.5550])),
    (("10P", 3.0, 6.0), np.array([0.3243, 0.2293, 6.5550])),
    (("10P", 3.0, 8.0), np.array([0.3269, 0.2075, 6.5550])),
    (("10P", 3.0, 10.0), np.array([0.3286, 0.1889, 6.5550])),
    (("2.5RP", 3.0, 2.0), np.array([0.3272, 0.2861, 6.5550])),
    (("2.5RP", 3.0, 4.0), np.array([0.3400, 0.2624, 6.5550])),
    (("2.5RP", 3.0, 6.0), np.array([0.3501, 0.2425, 6.5550])),
    (("2.5RP", 3.0, 8.0), np.array([0.3598, 0.2233, 6.5550])),
    (("2.5RP", 3.0, 10.0), np.array([0.3681, 0.2054, 6.5550])),
    (("5RP", 3.0, 2.0), np.array([0.3370, 0.2940, 6.5550])),
    (("5RP", 3.0, 4.0), np.array([0.3586, 0.2742, 6.5550])),
    (("5RP", 3.0, 6.0), np.array([0.3765, 0.2569, 6.5550])),
    (("5RP", 3.0, 8.0), np.array([0.3930, 0.2395, 6.5550])),
    (("5RP", 3.0, 10.0), np.array([0.4073, 0.2235, 6.5550])),
    (("7.5RP", 3.0, 2.0), np.array([0.3450, 0.3001, 6.5550])),
    (("7.5RP", 3.0, 4.0), np.array([0.3739, 0.2851, 6.5550])),
    (("7.5RP", 3.0, 6.0), np.array([0.3990, 0.2708, 6.5550])),
    (("7.5RP", 3.0, 8.0), np.array([0.4234, 0.2556, 6.5550])),
    (("7.5RP", 3.0, 10.0), np.array([0.4445, 0.2419, 6.5550])),
    (("10RP", 4.0, 2.0), np.array([0.3417, 0.3106, 12.0000])),
    (("10RP", 4.0, 4.0), np.array([0.3715, 0.3042, 12.0000])),
    (("10RP", 4.0, 6.0), np.array([0.3999, 0.2972, 12.0000])),
    (("10RP", 4.0, 8.0), np.array([0.4282, 0.2890, 12.0000])),
    (("10RP", 4.0, 10.0), np.array([0.4528, 0.2811, 12.0000])),
    (("2.5R", 4.0, 2.0), np.array([0.3461, 0.3150, 12.0000])),
    (("2.5R", 4.0, 4.0), np.array([0.3806, 0.3125, 12.0000])),
    (("2.5R", 4.0, 6.0), np.array([0.4141, 0.3085, 12.0000])),
    (("2.5R", 4.0, 8.0), np.array([0.4472, 0.3031, 12.0000])),
    (("2.5R", 4.0, 10.0), np.array([0.4774, 0.2969, 12.0000])),
    (("2.5R", 4.0, 12.0), np.array([0.5072, 0.2897, 12.0000])),
    (("5R", 4.0, 2.0), np.array([0.3508, 0.3200, 12.0000])),
    (("5R", 4.0, 4.0), np.array([0.3916, 0.3223, 12.0000])),
    (("5R", 4.0, 6.0), np.array([0.4299, 0.3226, 12.0000])),
    (("5R", 4.0, 8.0), np.array([0.4690, 0.3209, 12.0000])),
    (("5R", 4.0, 10.0), np.array([0.5043, 0.3176, 12.0000])),
    (("5R", 4.0, 12.0), np.array([0.5385, 0.3129, 12.0000])),
    (("5R", 4.0, 14.0), np.array([0.5734, 0.3057, 12.0000])),
    (("7.5R", 4.0, 2.0), np.array([0.3538, 0.3236, 12.0000])),
    (("7.5R", 4.0, 4.0), np.array([0.3990, 0.3300, 12.0000])),
    (("7.5R", 4.0, 6.0), np.array([0.4415, 0.3340, 12.0000])),
    (("7.5R", 4.0, 8.0), np.array([0.4850, 0.3359, 12.0000])),
    (("7.5R", 4.0, 10.0), np.array([0.5235, 0.3351, 12.0000])),
    (("7.5R", 4.0, 12.0), np.array([0.5603, 0.3321, 12.0000])),
    (("10R", 4.0, 2.0), np.array([0.3582, 0.3294, 12.0000])),
    (("10R", 4.0, 4.0), np.array([0.4078, 0.3412, 12.0000])),
    (("10R", 4.0, 6.0), np.array([0.4535, 0.3500, 12.0000])),
    (("10R", 4.0, 8.0), np.array([0.4995, 0.3557, 12.0000])),
    (("10R", 4.0, 10.0), np.array([0.5418, 0.3580, 12.0000])),
    (("2.5YR", 4.0, 2.0), np.array([0.3624, 0.3367, 12.0000])),
    (("2.5YR", 4.0, 4.0), np.array([0.4141, 0.3539, 12.0000])),
    (("2.5YR", 4.0, 6.0), np.array([0.4612, 0.3674, 12.0000])),
    (("2.5YR", 4.0, 8.0), np.array([0.5071, 0.3777, 12.0000])),
    (("2.5YR", 4.0, 10.0), np.array([0.5475, 0.3856, 12.0000])),
    (("5YR", 4.0, 2.0), np.array([0.3651, 0.3442, 12.0000])),
    (("5YR", 4.0, 4.0), np.array([0.4187, 0.3679, 12.0000])),
    (("5YR", 4.0, 6.0), np.array([0.4651, 0.3859, 12.0000])),
    (("5YR", 4.0, 8.0), np.array([0.5070, 0.3994, 12.0000])),
    (("7.5YR", 4.0, 2.0), np.array([0.3662, 0.3504, 12.0000])),
    (("7.5YR", 4.0, 4.0), np.array([0.4208, 0.3809, 12.0000])),
    (("7.5YR", 4.0, 6.0), np.array([0.4655, 0.4029, 12.0000])),
    (("7.5YR", 4.0, 8.0), np.array([0.5038, 0.4204, 12.0000])),
    (("10YR", 4.0, 2.0), np.array([0.3660, 0.3590, 12.0000])),
    (("10YR", 4.0, 4.0), np.array([0.4189, 0.3948, 12.0000])),
    (("10YR", 4.0, 6.0), np.array([0.4618, 0.4213, 12.0000])),
    (("2.5Y", 4.0, 2.0), np.array([0.3633, 0.3654, 12.0000])),
    (("2.5Y", 4.0, 4.0), np.array([0.4138, 0.4076, 12.0000])),
    (("2.5Y", 4.0, 6.0), np.array([0.4542, 0.4391, 12.0000])),
    (("5Y", 4.0, 2.0), np.array([0.3590, 0.3701, 12.0000])),
    (("5Y", 4.0, 4.0), np.array([0.4069, 0.4188, 12.0000])),
    (("7.5Y", 4.0, 2.0), np.array([0.3542, 0.3727, 12.0000])),
    (("7.5Y", 4.0, 4.0), np.array([0.3982, 0.4272, 12.0000])),
    (("10Y", 4.0, 2.0), np.array([0.3436, 0.3732, 12.0000])),
    (("10Y", 4.0, 4.0), np.array([0.3871, 0.4321, 12.0000])),
    (("2.5GY", 4.0, 2.0), np.array([0.3382, 0.3706, 12.0000])),
    (("2.5GY", 4.0, 4.0), np.array([0.3708, 0.4329, 12.0000])),
    (("2.5GY", 4.0, 6.0), np.array([0.3968, 0.4857, 12.0000])),
    (("5GY", 4.0, 2.0), np.array([0.3312, 0.3678, 12.0000])),
    (("5GY", 4.0, 4.0), np.array([0.3538, 0.4284, 12.0000])),
    (("5GY", 4.0, 6.0), np.array([0.3718, 0.4852, 12.0000])),
    (("7.5GY", 4.0, 2.0), np.array([0.3185, 0.3604, 12.0000])),
    (("7.5GY", 4.0, 4.0), np.array([0.3281, 0.4157, 12.0000])),
    (("7.5GY", 4.0, 6.0), np.array([0.3355, 0.4739, 12.0000])),
    (("10GY", 4.0, 2.0), np.array([0.3109, 0.3550, 12.0000])),
    (("10GY", 4.0, 4.0), np.array([0.3100, 0.4018, 12.0000])),
    (("10GY", 4.0, 6.0), np.array([0.3069, 0.4550, 12.0000])),
    (("2.5G", 4.0, 2.0), np.array([0.3012, 0.3470, 12.0000])),
    (("2.5G", 4.0, 4.0), np.array([0.2891, 0.3821, 12.0000])),
    (("2.5G", 4.0, 6.0), np.array([0.2735, 0.4215, 12.0000])),
    (("5G", 4.0, 2.0), np.array([0.2959, 0.3417, 12.0000])),
    (("5G", 4.0, 4.0), np.array([0.2781, 0.3704, 12.0000])),
    (("7.5G", 4.0, 2.0), np.array([0.2919, 0.3371, 12.0000])),
    (("7.5G", 4.0, 4.0), np.array([0.2702, 0.3602, 12.0000])),
    (("10G", 4.0, 2.0), np.array([0.2880, 0.3327, 12.0000])),
    (("10G", 4.0, 4.0), np.array([0.2628, 0.3498, 12.0000])),
    (("2.5BG", 4.0, 2.0), np.array([0.2840, 0.3270, 12.0000])),
    (("2.5BG", 4.0, 4.0), np.array([0.2552, 0.3375, 12.0000])),
    (("2.5BG", 4.0, 6.0), np.array([0.2278, 0.3463, 12.0000])),
    (("5BG", 4.0, 2.0), np.array([0.2799, 0.3208, 12.0000])),
    (("5BG", 4.0, 4.0), np.array([0.2480, 0.3232, 12.0000])),
    (("5BG", 4.0, 6.0), np.array([0.2182, 0.3240, 12.0000])),
    (("7.5BG", 4.0, 2.0), np.array([0.2764, 0.3148, 12.0000])),
    (("7.5BG", 4.0, 4.0), np.array([0.2429, 0.3108, 12.0000])),
    (("7.5BG", 4.0, 6.0), np.array([0.2113, 0.3052, 12.0000])),
    (("10BG", 4.0, 2.0), np.array([0.2740, 0.3091, 12.0000])),
    (("10BG", 4.0, 4.0), np.array([0.2384, 0.2984, 12.0000])),
    (("10BG", 4.0, 6.0), np.array([0.2065, 0.2863, 12.0000])),
    (("2.5B", 4.0, 2.0), np.array([0.2727, 0.3038, 12.0000])),
    (("2.5B", 4.0, 4.0), np.array([0.2360, 0.2872, 12.0000])),
    (("2.5B", 4.0, 6.0), np.array([0.2048, 0.2708, 12.0000])),
    (("2.5B", 4.0, 8.0), np.array([0.1737, 0.2524, 12.0000])),
    (("5B", 4.0, 2.0), np.array([0.2723, 0.2992, 12.0000])),
    (("5B", 4.0, 4.0), np.array([0.2363, 0.2782, 12.0000])),
    (("5B", 4.0, 6.0), np.array([0.2060, 0.2572, 12.0000])),
    (("5B", 4.0, 8.0), np.array([0.1759, 0.2345, 12.0000])),
    (("7.5B", 4.0, 2.0), np.array([0.2733, 0.2947, 12.0000])),
    (("7.5B", 4.0, 4.0), np.array([0.2388, 0.2704, 12.0000])),
    (("7.5B", 4.0, 6.0), np.array([0.2102, 0.2470, 12.0000])),
    (("7.5B", 4.0, 8.0), np.array([0.1821, 0.2232, 12.0000])),
    (("10B", 4.0, 2.0), np.array([0.2753, 0.2910, 12.0000])),
    (("10B", 4.0, 4.0), np.array([0.2429, 0.2648, 12.0000])),
    (("10B", 4.0, 6.0), np.array([0.2157, 0.2407, 12.0000])),
    (("10B", 4.0, 8.0), np.array([0.1893, 0.2160, 12.0000])),
    (("2.5PB", 4.0, 2.0), np.array([0.2782, 0.2876, 12.0000])),
    (("2.5PB", 4.0, 4.0), np.array([0.2487, 0.2597, 12.0000])),
    (("2.5PB", 4.0, 6.0), np.array([0.2235, 0.2343, 12.0000])),
    (("2.5PB", 4.0, 8.0), np.array([0.1995, 0.2094, 12.0000])),
    (("2.5PB", 4.0, 10.0), np.array([0.1805, 0.1888, 12.0000])),
    (("5PB", 4.0, 2.0), np.array([0.2816, 0.2842, 12.0000])),
    (("5PB", 4.0, 4.0), np.array([0.2562, 0.2560, 12.0000])),
    (("5PB", 4.0, 6.0), np.array([0.2325, 0.2300, 12.0000])),
    (("5PB", 4.0, 8.0), np.array([0.2103, 0.2050, 12.0000])),
    (("5PB", 4.0, 10.0), np.array([0.1925, 0.1843, 12.0000])),
    (("7.5PB", 4.0, 2.0), np.array([0.2861, 0.2819, 12.0000])),
    (("7.5PB", 4.0, 4.0), np.array([0.2657, 0.2528, 12.0000])),
    (("7.5PB", 4.0, 6.0), np.array([0.2471, 0.2266, 12.0000])),
    (("7.5PB", 4.0, 8.0), np.array([0.2304, 0.2023, 12.0000])),
    (("7.5PB", 4.0, 10.0), np.array([0.2158, 0.1811, 12.0000])),
    (("10PB", 4.0, 2.0), np.array([0.2911, 0.2804, 12.0000])),
    (("10PB", 4.0, 4.0), np.array([0.2759, 0.2522, 12.0000])),
    (("10PB", 4.0, 6.0), np.array([0.2618, 0.2263, 12.0000])),
    (("10PB", 4.0, 8.0), np.array([0.2497, 0.2038, 12.0000])),
    (("10PB", 4.0, 10.0), np.array([0.2388, 0.1837, 12.0000])),
    (("2.5P", 4.0, 2.0), np.array([0.2962, 0.2807, 12.0000])),
    (("2.5P", 4.0, 4.0), np.array([0.2855, 0.2531, 12.0000])),
    (("2.5P", 4.0, 6.0), np.array([0.2763, 0.2300, 12.0000])),
    (("2.5P", 4.0, 8.0), np.array([0.2685, 0.2089, 12.0000])),
    (("2.5P", 4.0, 10.0), np.array([0.2619, 0.1903, 12.0000])),
    (("2.5P", 4.0, 12.0), np.array([0.2559, 0.1730, 12.0000])),
    (("5P", 4.0, 2.0), np.array([0.3022, 0.2825, 12.0000])),
    (("5P", 4.0, 4.0), np.array([0.2958, 0.2565, 12.0000])),
    (("5P", 4.0, 6.0), np.array([0.2903, 0.2347, 12.0000])),
    (("5P", 4.0, 8.0), np.array([0.2855, 0.2150, 12.0000])),
    (("5P", 4.0, 10.0), np.array([0.2814, 0.1967, 12.0000])),
    (("5P", 4.0, 12.0), np.array([0.2778, 0.1808, 12.0000])),
    (("7.5P", 4.0, 2.0), np.array([0.3093, 0.2859, 12.0000])),
    (("7.5P", 4.0, 4.0), np.array([0.3084, 0.2622, 12.0000])),
    (("7.5P", 4.0, 6.0), np.array([0.3076, 0.2416, 12.0000])),
    (("7.5P", 4.0, 8.0), np.array([0.3066, 0.2228, 12.0000])),
    (("7.5P", 4.0, 10.0), np.array([0.3056, 0.2060, 12.0000])),
    (("7.5P", 4.0, 12.0), np.array([0.3045, 0.1905, 12.0000])),
    (("10P", 4.0, 2.0), np.array([0.3162, 0.2902, 12.0000])),
    (("10P", 4.0, 4.0), np.array([0.3210, 0.2686, 12.0000])),
    (("10P", 4.0, 6.0), np.array([0.3248, 0.2493, 12.0000])),
    (("10P", 4.0, 8.0), np.array([0.3280, 0.2318, 12.0000])),
    (("10P", 4.0, 10.0), np.array([0.3306, 0.2162, 12.0000])),
    (("2.5RP", 4.0, 2.0), np.array([0.3231, 0.2951, 12.0000])),
    (("2.5RP", 4.0, 4.0), np.array([0.3340, 0.2770, 12.0000])),
    (("2.5RP", 4.0, 6.0), np.array([0.3442, 0.2595, 12.0000])),
    (("2.5RP", 4.0, 8.0), np.array([0.3533, 0.2438, 12.0000])),
    (("2.5RP", 4.0, 10.0), np.array([0.3608, 0.2301, 12.0000])),
    (("2.5RP", 4.0, 12.0), np.array([0.3683, 0.2162, 12.0000])),
    (("5RP", 4.0, 2.0), np.array([0.3310, 0.3010, 12.0000])),
    (("5RP", 4.0, 4.0), np.array([0.3491, 0.2872, 12.0000])),
    (("5RP", 4.0, 6.0), np.array([0.3671, 0.2733, 12.0000])),
    (("5RP", 4.0, 8.0), np.array([0.3833, 0.2600, 12.0000])),
    (("5RP", 4.0, 10.0), np.array([0.3960, 0.2489, 12.0000])),
    (("5RP", 4.0, 12.0), np.array([0.4104, 0.2361, 12.0000])),
    (("7.5RP", 4.0, 2.0), np.array([0.3371, 0.3061, 12.0000])),
    (("7.5RP", 4.0, 4.0), np.array([0.3612, 0.2963, 12.0000])),
    (("7.5RP", 4.0, 6.0), np.array([0.3850, 0.2859, 12.0000])),
    (("7.5RP", 4.0, 8.0), np.array([0.4072, 0.2750, 12.0000])),
    (("7.5RP", 4.0, 10.0), np.array([0.4259, 0.2651, 12.0000])),
    (("7.5RP", 4.0, 12.0), np.array([0.4450, 0.2541, 12.0000])),
    (("10RP", 5.0, 2.0), np.array([0.3332, 0.3131, 19.7700])),
    (("10RP", 5.0, 4.0), np.array([0.3594, 0.3090, 19.7700])),
    (("10RP", 5.0, 6.0), np.array([0.3851, 0.3039, 19.7700])),
    (("10RP", 5.0, 8.0), np.array([0.4105, 0.2980, 19.7700])),
    (("10RP", 5.0, 10.0), np.array([0.4332, 0.2918, 19.7700])),
    (("2.5R", 5.0, 2.0), np.array([0.3360, 0.3158, 19.7700])),
    (("2.5R", 5.0, 4.0), np.array([0.3660, 0.3148, 19.7700])),
    (("2.5R", 5.0, 6.0), np.array([0.3960, 0.3130, 19.7700])),
    (("2.5R", 5.0, 8.0), np.array([0.4252, 0.3101, 19.7700])),
    (("2.5R", 5.0, 10.0), np.array([0.4533, 0.3058, 19.7700])),
    (("2.5R", 5.0, 12.0), np.array([0.4820, 0.3002, 19.7700])),
    (("5R", 5.0, 2.0), np.array([0.3392, 0.3192, 19.7700])),
    (("5R", 5.0, 4.0), np.array([0.3740, 0.3220, 19.7700])),
    (("5R", 5.0, 6.0), np.array([0.4078, 0.3238, 19.7700])),
    (("5R", 5.0, 8.0), np.array([0.4413, 0.3240, 19.7700])),
    (("5R", 5.0, 10.0), np.array([0.4747, 0.3227, 19.7700])),
    (("5R", 5.0, 12.0), np.array([0.5071, 0.3194, 19.7700])),
    (("7.5R", 5.0, 2.0), np.array([0.3425, 0.3229, 19.7700])),
    (("7.5R", 5.0, 4.0), np.array([0.3806, 0.3294, 19.7700])),
    (("7.5R", 5.0, 6.0), np.array([0.4180, 0.3348, 19.7700])),
    (("7.5R", 5.0, 8.0), np.array([0.4563, 0.3387, 19.7700])),
    (("7.5R", 5.0, 10.0), np.array([0.4927, 0.3399, 19.7700])),
    (("7.5R", 5.0, 12.0), np.array([0.5280, 0.3389, 19.7700])),
    (("10R", 5.0, 2.0), np.array([0.3465, 0.3278, 19.7700])),
    (("10R", 5.0, 4.0), np.array([0.3879, 0.3398, 19.7700])),
    (("10R", 5.0, 6.0), np.array([0.4299, 0.3499, 19.7700])),
    (("10R", 5.0, 8.0), np.array([0.4713, 0.3575, 19.7700])),
    (("10R", 5.0, 10.0), np.array([0.5113, 0.3630, 19.7700])),
    (("2.5YR", 5.0, 2.0), np.array([0.3506, 0.3337, 19.7700])),
    (("2.5YR", 5.0, 4.0), np.array([0.3925, 0.3494, 19.7700])),
    (("2.5YR", 5.0, 6.0), np.array([0.4365, 0.3640, 19.7700])),
    (("2.5YR", 5.0, 8.0), np.array([0.4795, 0.3758, 19.7700])),
    (("2.5YR", 5.0, 10.0), np.array([0.5175, 0.3844, 19.7700])),
    (("5YR", 5.0, 2.0), np.array([0.3530, 0.3395, 19.7700])),
    (("5YR", 5.0, 4.0), np.array([0.3968, 0.3614, 19.7700])),
    (("5YR", 5.0, 6.0), np.array([0.4420, 0.3808, 19.7700])),
    (("5YR", 5.0, 8.0), np.array([0.4830, 0.3960, 19.7700])),
    (("5YR", 5.0, 10.0), np.array([0.5161, 0.4064, 19.7700])),
    (("7.5YR", 5.0, 2.0), np.array([0.3540, 0.3445, 19.7700])),
    (("7.5YR", 5.0, 4.0), np.array([0.3991, 0.3714, 19.7700])),
    (("7.5YR", 5.0, 6.0), np.array([0.4440, 0.3954, 19.7700])),
    (("7.5YR", 5.0, 8.0), np.array([0.4820, 0.4141, 19.7700])),
    (("7.5YR", 5.0, 10.0), np.array([0.5108, 0.4276, 19.7700])),
    (("10YR", 5.0, 2.0), np.array([0.3546, 0.3514, 19.7700])),
    (("10YR", 5.0, 4.0), np.array([0.3995, 0.3840, 19.7700])),
    (("10YR", 5.0, 6.0), np.array([0.4428, 0.4128, 19.7700])),
    (("10YR", 5.0, 8.0), np.array([0.4770, 0.4338, 19.7700])),
    (("2.5Y", 5.0, 2.0), np.array([0.3534, 0.3570, 19.7700])),
    (("2.5Y", 5.0, 4.0), np.array([0.3968, 0.3954, 19.7700])),
    (("2.5Y", 5.0, 6.0), np.array([0.4380, 0.4292, 19.7700])),
    (("2.5Y", 5.0, 8.0), np.array([0.4685, 0.4524, 19.7700])),
    (("5Y", 5.0, 2.0), np.array([0.3500, 0.3620, 19.7700])),
    (("5Y", 5.0, 4.0), np.array([0.3915, 0.4057, 19.7700])),
    (("5Y", 5.0, 6.0), np.array([0.4302, 0.4435, 19.7700])),
    (("7.5Y", 5.0, 2.0), np.array([0.3470, 0.3640, 19.7700])),
    (("7.5Y", 5.0, 4.0), np.array([0.3850, 0.4120, 19.7700])),
    (("7.5Y", 5.0, 6.0), np.array([0.4199, 0.4551, 19.7700])),
    (("10Y", 5.0, 2.0), np.array([0.3422, 0.3648, 19.7700])),
    (("10Y", 5.0, 4.0), np.array([0.3762, 0.4158, 19.7700])),
    (("10Y", 5.0, 6.0), np.array([0.4072, 0.4621, 19.7700])),
    (("2.5GY", 5.0, 2.0), np.array([0.3352, 0.3636, 19.7700])),
    (("2.5GY", 5.0, 4.0), np.array([0.3621, 0.4143, 19.7700])),
    (("2.5GY", 5.0, 6.0), np.array([0.3879, 0.4646, 19.7700])),
    (("2.5GY", 5.0, 8.0), np.array([0.4088, 0.5068, 19.7700])),
    (("5GY", 5.0, 2.0), np.array([0.3289, 0.3612, 19.7700])),
    (("5GY", 5.0, 4.0), np.array([0.3482, 0.4097, 19.7700])),
    (("5GY", 5.0, 6.0), np.array([0.3663, 0.4614, 19.7700])),
    (("5GY", 5.0, 8.0), np.array([0.3815, 0.5093, 19.7700])),
    (("7.5GY", 5.0, 2.0), np.array([0.3188, 0.3560, 19.7700])),
    (("7.5GY", 5.0, 4.0), np.array([0.3274, 0.3994, 19.7700])),
    (("7.5GY", 5.0, 6.0), np.array([0.3354, 0.4483, 19.7700])),
    (("7.5GY", 5.0, 8.0), np.array([0.3412, 0.4976, 19.7700])),
    (("10GY", 5.0, 2.0), np.array([0.3110, 0.3508, 19.7700])),
    (("10GY", 5.0, 4.0), np.array([0.3111, 0.3881, 19.7700])),
    (("10GY", 5.0, 6.0), np.array([0.3108, 0.4301, 19.7700])),
    (("10GY", 5.0, 8.0), np.array([0.3080, 0.4759, 19.7700])),
    (("2.5G", 5.0, 2.0), np.array([0.3030, 0.3445, 19.7700])),
    (("2.5G", 5.0, 4.0), np.array([0.2943, 0.3735, 19.7700])),
    (("2.5G", 5.0, 6.0), np.array([0.2841, 0.4045, 19.7700])),
    (("2.5G", 5.0, 8.0), np.array([0.2710, 0.4380, 19.7700])),
    (("5G", 5.0, 2.0), np.array([0.2978, 0.3392, 19.7700])),
    (("5G", 5.0, 4.0), np.array([0.2841, 0.3628, 19.7700])),
    (("5G", 5.0, 6.0), np.array([0.2690, 0.3860, 19.7700])),
    (("5G", 5.0, 8.0), np.array([0.2511, 0.4107, 19.7700])),
    (("7.5G", 5.0, 2.0), np.array([0.2945, 0.3355, 19.7700])),
    (("7.5G", 5.0, 4.0), np.array([0.2775, 0.3545, 19.7700])),
    (("7.5G", 5.0, 6.0), np.array([0.2598, 0.3724, 19.7700])),
    (("7.5G", 5.0, 8.0), np.array([0.2395, 0.3915, 19.7700])),
    (("10G", 5.0, 2.0), np.array([0.2910, 0.3310, 19.7700])),
    (("10G", 5.0, 4.0), np.array([0.2711, 0.3455, 19.7700])),
    (("10G", 5.0, 6.0), np.array([0.2519, 0.3587, 19.7700])),
    (("2.5BG", 5.0, 2.0), np.array([0.2880, 0.3270, 19.7700])),
    (("2.5BG", 5.0, 4.0), np.array([0.2659, 0.3369, 19.7700])),
    (("2.5BG", 5.0, 6.0), np.array([0.2448, 0.3452, 19.7700])),
    (("5BG", 5.0, 2.0), np.array([0.2841, 0.3210, 19.7700])),
    (("5BG", 5.0, 4.0), np.array([0.2591, 0.3246, 19.7700])),
    (("5BG", 5.0, 6.0), np.array([0.2360, 0.3270, 19.7700])),
    (("7.5BG", 5.0, 2.0), np.array([0.2812, 0.3161, 19.7700])),
    (("7.5BG", 5.0, 4.0), np.array([0.2550, 0.3150, 19.7700])),
    (("7.5BG", 5.0, 6.0), np.array([0.2292, 0.3125, 19.7700])),
    (("10BG", 5.0, 2.0), np.array([0.2796, 0.3111, 19.7700])),
    (("10BG", 5.0, 4.0), np.array([0.2512, 0.3040, 19.7700])),
    (("10BG", 5.0, 6.0), np.array([0.2234, 0.2952, 19.7700])),
    (("2.5B", 5.0, 2.0), np.array([0.2791, 0.3071, 19.7700])),
    (("2.5B", 5.0, 4.0), np.array([0.2492, 0.2954, 19.7700])),
    (("2.5B", 5.0, 6.0), np.array([0.2210, 0.2823, 19.7700])),
    (("5B", 5.0, 2.0), np.array([0.2794, 0.3032, 19.7700])),
    (("5B", 5.0, 4.0), np.array([0.2493, 0.2879, 19.7700])),
    (("5B", 5.0, 6.0), np.array([0.2215, 0.2701, 19.7700])),
    (("7.5B", 5.0, 2.0), np.array([0.2803, 0.3000, 19.7700])),
    (("7.5B", 5.0, 4.0), np.array([0.2511, 0.2808, 19.7700])),
    (("7.5B", 5.0, 6.0), np.array([0.2248, 0.2612, 19.7700])),
    (("10B", 5.0, 2.0), np.array([0.2821, 0.2966, 19.7700])),
    (("10B", 5.0, 4.0), np.array([0.2547, 0.2757, 19.7700])),
    (("10B", 5.0, 6.0), np.array([0.2299, 0.2548, 19.7700])),
    (("2.5PB", 5.0, 2.0), np.array([0.2847, 0.2942, 19.7700])),
    (("2.5PB", 5.0, 4.0), np.array([0.2600, 0.2720, 19.7700])),
    (("2.5PB", 5.0, 6.0), np.array([0.2365, 0.2488, 19.7700])),
    (("2.5PB", 5.0, 8.0), np.array([0.2157, 0.2278, 19.7700])),
    (("5PB", 5.0, 2.0), np.array([0.2882, 0.2923, 19.7700])),
    (("5PB", 5.0, 4.0), np.array([0.2662, 0.2687, 19.7700])),
    (("5PB", 5.0, 6.0), np.array([0.2447, 0.2449, 19.7700])),
    (("5PB", 5.0, 8.0), np.array([0.2255, 0.2239, 19.7700])),
    (("5PB", 5.0, 10.0), np.array([0.2080, 0.2041, 19.7700])),
    (("7.5PB", 5.0, 2.0), np.array([0.2918, 0.2908, 19.7700])),
    (("7.5PB", 5.0, 4.0), np.array([0.2739, 0.2666, 19.7700])),
    (("7.5PB", 5.0, 6.0), np.array([0.2563, 0.2417, 19.7700])),
    (("7.5PB", 5.0, 8.0), np.array([0.2417, 0.2204, 19.7700])),
    (("7.5PB", 5.0, 10.0), np.array([0.2285, 0.2020, 19.7700])),
    (("10PB", 5.0, 2.0), np.array([0.2959, 0.2905, 19.7700])),
    (("10PB", 5.0, 4.0), np.array([0.2821, 0.2659, 19.7700])),
    (("10PB", 5.0, 6.0), np.array([0.2686, 0.2412, 19.7700])),
    (("10PB", 5.0, 8.0), np.array([0.2572, 0.2211, 19.7700])),
    (("10PB", 5.0, 10.0), np.array([0.2478, 0.2030, 19.7700])),
    (("2.5P", 5.0, 2.0), np.array([0.3000, 0.2912, 19.7700])),
    (("2.5P", 5.0, 4.0), np.array([0.2898, 0.2667, 19.7700])),
    (("2.5P", 5.0, 6.0), np.array([0.2806, 0.2444, 19.7700])),
    (("2.5P", 5.0, 8.0), np.array([0.2728, 0.2240, 19.7700])),
    (("2.5P", 5.0, 10.0), np.array([0.2665, 0.2075, 19.7700])),
    (("5P", 5.0, 2.0), np.array([0.3045, 0.2928, 19.7700])),
    (("5P", 5.0, 4.0), np.array([0.2986, 0.2699, 19.7700])),
    (("5P", 5.0, 6.0), np.array([0.2932, 0.2487, 19.7700])),
    (("5P", 5.0, 8.0), np.array([0.2885, 0.2296, 19.7700])),
    (("5P", 5.0, 10.0), np.array([0.2845, 0.2137, 19.7700])),
    (("7.5P", 5.0, 2.0), np.array([0.3103, 0.2959, 19.7700])),
    (("7.5P", 5.0, 4.0), np.array([0.3100, 0.2750, 19.7700])),
    (("7.5P", 5.0, 6.0), np.array([0.3093, 0.2555, 19.7700])),
    (("7.5P", 5.0, 8.0), np.array([0.3087, 0.2375, 19.7700])),
    (("7.5P", 5.0, 10.0), np.array([0.3080, 0.2230, 19.7700])),
    (("10P", 5.0, 2.0), np.array([0.3148, 0.2986, 19.7700])),
    (("10P", 5.0, 4.0), np.array([0.3198, 0.2807, 19.7700])),
    (("10P", 5.0, 6.0), np.array([0.3243, 0.2630, 19.7700])),
    (("10P", 5.0, 8.0), np.array([0.3280, 0.2464, 19.7700])),
    (("10P", 5.0, 10.0), np.array([0.3308, 0.2328, 19.7700])),
    (("2.5RP", 5.0, 2.0), np.array([0.3199, 0.3019, 19.7700])),
    (("2.5RP", 5.0, 4.0), np.array([0.3298, 0.2869, 19.7700])),
    (("2.5RP", 5.0, 6.0), np.array([0.3396, 0.2718, 19.7700])),
    (("2.5RP", 5.0, 8.0), np.array([0.3490, 0.2570, 19.7700])),
    (("2.5RP", 5.0, 10.0), np.array([0.3560, 0.2452, 19.7700])),
    (("5RP", 5.0, 2.0), np.array([0.3256, 0.3065, 19.7700])),
    (("5RP", 5.0, 4.0), np.array([0.3421, 0.2954, 19.7700])),
    (("5RP", 5.0, 6.0), np.array([0.3585, 0.2842, 19.7700])),
    (("5RP", 5.0, 8.0), np.array([0.3748, 0.2729, 19.7700])),
    (("5RP", 5.0, 10.0), np.array([0.3880, 0.2630, 19.7700])),
    (("7.5RP", 5.0, 2.0), np.array([0.3296, 0.3098, 19.7700])),
    (("7.5RP", 5.0, 4.0), np.array([0.3515, 0.3024, 19.7700])),
    (("7.5RP", 5.0, 6.0), np.array([0.3726, 0.2941, 19.7700])),
    (("7.5RP", 5.0, 8.0), np.array([0.3932, 0.2852, 19.7700])),
    (("7.5RP", 5.0, 10.0), np.array([0.4108, 0.2773, 19.7700])),
    (("10RP", 6.0, 2.0), np.array([0.3292, 0.3141, 30.0500])),
    (("10RP", 6.0, 4.0), np.array([0.3508, 0.3112, 30.0500])),
    (("10RP", 6.0, 6.0), np.array([0.3740, 0.3074, 30.0500])),
    (("10RP", 6.0, 8.0), np.array([0.3930, 0.3038, 30.0500])),
    (("10RP", 6.0, 10.0), np.array([0.4150, 0.2989, 30.0500])),
    (("2.5R", 6.0, 2.0), np.array([0.3318, 0.3166, 30.0500])),
    (("2.5R", 6.0, 4.0), np.array([0.3566, 0.3163, 30.0500])),
    (("2.5R", 6.0, 6.0), np.array([0.3832, 0.3158, 30.0500])),
    (("2.5R", 6.0, 8.0), np.array([0.4065, 0.3144, 30.0500])),
    (("2.5R", 6.0, 10.0), np.array([0.4320, 0.3118, 30.0500])),
    (("5R", 6.0, 2.0), np.array([0.3343, 0.3190, 30.0500])),
    (("5R", 6.0, 4.0), np.array([0.3628, 0.3221, 30.0500])),
    (("5R", 6.0, 6.0), np.array([0.3921, 0.3244, 30.0500])),
    (("5R", 6.0, 8.0), np.array([0.4187, 0.3251, 30.0500])),
    (("5R", 6.0, 10.0), np.array([0.4480, 0.3250, 30.0500])),
    (("7.5R", 6.0, 2.0), np.array([0.3381, 0.3228, 30.0500])),
    (("7.5R", 6.0, 4.0), np.array([0.3692, 0.3291, 30.0500])),
    (("7.5R", 6.0, 6.0), np.array([0.4000, 0.3340, 30.0500])),
    (("7.5R", 6.0, 8.0), np.array([0.4318, 0.3383, 30.0500])),
    (("7.5R", 6.0, 10.0), np.array([0.4655, 0.3412, 30.0500])),
    (("10R", 6.0, 2.0), np.array([0.3417, 0.3268, 30.0500])),
    (("10R", 6.0, 4.0), np.array([0.3768, 0.3381, 30.0500])),
    (("10R", 6.0, 6.0), np.array([0.4103, 0.3473, 30.0500])),
    (("10R", 6.0, 8.0), np.array([0.4449, 0.3550, 30.0500])),
    (("10R", 6.0, 10.0), np.array([0.4812, 0.3619, 30.0500])),
    (("2.5YR", 6.0, 2.0), np.array([0.3453, 0.3321, 30.0500])),
    (("2.5YR", 6.0, 4.0), np.array([0.3806, 0.3467, 30.0500])),
    (("2.5YR", 6.0, 6.0), np.array([0.4180, 0.3600, 30.0500])),
    (("2.5YR", 6.0, 8.0), np.array([0.4533, 0.3708, 30.0500])),
    (("2.5YR", 6.0, 10.0), np.array([0.4891, 0.3806, 30.0500])),
    (("2.5YR", 6.0, 12.0), np.array([0.5215, 0.3887, 30.0500])),
    (("5YR", 6.0, 2.0), np.array([0.3474, 0.3373, 30.0500])),
    (("5YR", 6.0, 4.0), np.array([0.3840, 0.3564, 30.0500])),
    (("5YR", 6.0, 6.0), np.array([0.4229, 0.3750, 30.0500])),
    (("5YR", 6.0, 8.0), np.array([0.4592, 0.3900, 30.0500])),
    (("5YR", 6.0, 10.0), np.array([0.4921, 0.4022, 30.0500])),
    (("5YR", 6.0, 12.0), np.array([0.5199, 0.4119, 30.0500])),
    (("7.5YR", 6.0, 2.0), np.array([0.3487, 0.3421, 30.0500])),
    (("7.5YR", 6.0, 4.0), np.array([0.3860, 0.3652, 30.0500])),
    (("7.5YR", 6.0, 6.0), np.array([0.4242, 0.3876, 30.0500])),
    (("7.5YR", 6.0, 8.0), np.array([0.4596, 0.4064, 30.0500])),
    (("7.5YR", 6.0, 10.0), np.array([0.4904, 0.4220, 30.0500])),
    (("7.5YR", 6.0, 12.0), np.array([0.5145, 0.4331, 30.0500])),
    (("10YR", 6.0, 2.0), np.array([0.3491, 0.3483, 30.0500])),
    (("10YR", 6.0, 4.0), np.array([0.3861, 0.3767, 30.0500])),
    (("10YR", 6.0, 6.0), np.array([0.4240, 0.4030, 30.0500])),
    (("10YR", 6.0, 8.0), np.array([0.4570, 0.4249, 30.0500])),
    (("10YR", 6.0, 10.0), np.array([0.4843, 0.4416, 30.0500])),
    (("2.5Y", 6.0, 2.0), np.array([0.3480, 0.3540, 30.0500])),
    (("2.5Y", 6.0, 4.0), np.array([0.3840, 0.3867, 30.0500])),
    (("2.5Y", 6.0, 6.0), np.array([0.4203, 0.4176, 30.0500])),
    (("2.5Y", 6.0, 8.0), np.array([0.4517, 0.4421, 30.0500])),
    (("2.5Y", 6.0, 10.0), np.array([0.4760, 0.4607, 30.0500])),
    (("5Y", 6.0, 2.0), np.array([0.3457, 0.3580, 30.0500])),
    (("5Y", 6.0, 4.0), np.array([0.3794, 0.3955, 30.0500])),
    (("5Y", 6.0, 6.0), np.array([0.4140, 0.4305, 30.0500])),
    (("5Y", 6.0, 8.0), np.array([0.4426, 0.4588, 30.0500])),
    (("7.5Y", 6.0, 2.0), np.array([0.3431, 0.3601, 30.0500])),
    (("7.5Y", 6.0, 4.0), np.array([0.3745, 0.4004, 30.0500])),
    (("7.5Y", 6.0, 6.0), np.array([0.4060, 0.4400, 30.0500])),
    (("7.5Y", 6.0, 8.0), np.array([0.4321, 0.4719, 30.0500])),
    (("10Y", 6.0, 2.0), np.array([0.3398, 0.3611, 30.0500])),
    (("10Y", 6.0, 4.0), np.array([0.3679, 0.4033, 30.0500])),
    (("10Y", 6.0, 6.0), np.array([0.3960, 0.4452, 30.0500])),
    (("2.5GY", 6.0, 2.0), np.array([0.3342, 0.3607, 30.0500])),
    (("2.5GY", 6.0, 4.0), np.array([0.3572, 0.4038, 30.0500])),
    (("2.5GY", 6.0, 6.0), np.array([0.3799, 0.4470, 30.0500])),
    (("2.5GY", 6.0, 8.0), np.array([0.4006, 0.4885, 30.0500])),
    (("5GY", 6.0, 2.0), np.array([0.3288, 0.3592, 30.0500])),
    (("5GY", 6.0, 4.0), np.array([0.3461, 0.4008, 30.0500])),
    (("5GY", 6.0, 6.0), np.array([0.3622, 0.4438, 30.0500])),
    (("5GY", 6.0, 8.0), np.array([0.3772, 0.4880, 30.0500])),
    (("7.5GY", 6.0, 2.0), np.array([0.3193, 0.3550, 30.0500])),
    (("7.5GY", 6.0, 4.0), np.array([0.3275, 0.3922, 30.0500])),
    (("7.5GY", 6.0, 6.0), np.array([0.3351, 0.4321, 30.0500])),
    (("7.5GY", 6.0, 8.0), np.array([0.3418, 0.4768, 30.0500])),
    (("7.5GY", 6.0, 10.0), np.array([0.3463, 0.5196, 30.0500])),
    (("10GY", 6.0, 2.0), np.array([0.3112, 0.3496, 30.0500])),
    (("10GY", 6.0, 4.0), np.array([0.3124, 0.3822, 30.0500])),
    (("10GY", 6.0, 6.0), np.array([0.3128, 0.4175, 30.0500])),
    (("10GY", 6.0, 8.0), np.array([0.3116, 0.4563, 30.0500])),
    (("10GY", 6.0, 10.0), np.array([0.3086, 0.4949, 30.0500])),
    (("2.5G", 6.0, 2.0), np.array([0.3039, 0.3437, 30.0500])),
    (("2.5G", 6.0, 4.0), np.array([0.2967, 0.3695, 30.0500])),
    (("2.5G", 6.0, 6.0), np.array([0.2892, 0.3963, 30.0500])),
    (("2.5G", 6.0, 8.0), np.array([0.2799, 0.4239, 30.0500])),
    (("5G", 6.0, 2.0), np.array([0.2988, 0.3382, 30.0500])),
    (("5G", 6.0, 4.0), np.array([0.2868, 0.3595, 30.0500])),
    (("5G", 6.0, 6.0), np.array([0.2748, 0.3795, 30.0500])),
    (("7.5G", 6.0, 2.0), np.array([0.2958, 0.3344, 30.0500])),
    (("7.5G", 6.0, 4.0), np.array([0.2807, 0.3522, 30.0500])),
    (("7.5G", 6.0, 6.0), np.array([0.2662, 0.3672, 30.0500])),
    (("10G", 6.0, 2.0), np.array([0.2929, 0.3303, 30.0500])),
    (("10G", 6.0, 4.0), np.array([0.2749, 0.3443, 30.0500])),
    (("10G", 6.0, 6.0), np.array([0.2591, 0.3558, 30.0500])),
    (("2.5BG", 6.0, 2.0), np.array([0.2902, 0.3268, 30.0500])),
    (("2.5BG", 6.0, 4.0), np.array([0.2702, 0.3369, 30.0500])),
    (("2.5BG", 6.0, 6.0), np.array([0.2526, 0.3448, 30.0500])),
    (("5BG", 6.0, 2.0), np.array([0.2872, 0.3219, 30.0500])),
    (("5BG", 6.0, 4.0), np.array([0.2648, 0.3262, 30.0500])),
    (("5BG", 6.0, 6.0), np.array([0.2441, 0.3290, 30.0500])),
    (("7.5BG", 6.0, 2.0), np.array([0.2849, 0.3172, 30.0500])),
    (("7.5BG", 6.0, 4.0), np.array([0.2604, 0.3169, 30.0500])),
    (("7.5BG", 6.0, 6.0), np.array([0.2384, 0.3155, 30.0500])),
    (("10BG", 6.0, 2.0), np.array([0.2837, 0.3132, 30.0500])),
    (("10BG", 6.0, 4.0), np.array([0.2578, 0.3078, 30.0500])),
    (("10BG", 6.0, 6.0), np.array([0.2335, 0.3015, 30.0500])),
    (("2.5B", 6.0, 2.0), np.array([0.2835, 0.3097, 30.0500])),
    (("2.5B", 6.0, 4.0), np.array([0.2571, 0.3008, 30.0500])),
    (("2.5B", 6.0, 6.0), np.array([0.2312, 0.2899, 30.0500])),
    (("5B", 6.0, 2.0), np.array([0.2842, 0.3063, 30.0500])),
    (("5B", 6.0, 4.0), np.array([0.2579, 0.2938, 30.0500])),
    (("5B", 6.0, 6.0), np.array([0.2320, 0.2789, 30.0500])),
    (("7.5B", 6.0, 2.0), np.array([0.2854, 0.3037, 30.0500])),
    (("7.5B", 6.0, 4.0), np.array([0.2602, 0.2881, 30.0500])),
    (("7.5B", 6.0, 6.0), np.array([0.2352, 0.2708, 30.0500])),
    (("10B", 6.0, 2.0), np.array([0.2871, 0.3012, 30.0500])),
    (("10B", 6.0, 4.0), np.array([0.2637, 0.2840, 30.0500])),
    (("10B", 6.0, 6.0), np.array([0.2399, 0.2650, 30.0500])),
    (("2.5PB", 6.0, 2.0), np.array([0.2897, 0.2991, 30.0500])),
    (("2.5PB", 6.0, 4.0), np.array([0.2684, 0.2804, 30.0500])),
    (("2.5PB", 6.0, 6.0), np.array([0.2465, 0.2599, 30.0500])),
    (("2.5PB", 6.0, 8.0), np.array([0.2274, 0.2406, 30.0500])),
    (("5PB", 6.0, 2.0), np.array([0.2923, 0.2978, 30.0500])),
    (("5PB", 6.0, 4.0), np.array([0.2734, 0.2778, 30.0500])),
    (("5PB", 6.0, 6.0), np.array([0.2533, 0.2558, 30.0500])),
    (("5PB", 6.0, 8.0), np.array([0.2360, 0.2365, 30.0500])),
    (("7.5PB", 6.0, 2.0), np.array([0.2955, 0.2963, 30.0500])),
    (("7.5PB", 6.0, 4.0), np.array([0.2798, 0.2752, 30.0500])),
    (("7.5PB", 6.0, 6.0), np.array([0.2638, 0.2531, 30.0500])),
    (("7.5PB", 6.0, 8.0), np.array([0.2505, 0.2347, 30.0500])),
    (("10PB", 6.0, 2.0), np.array([0.2988, 0.2961, 30.0500])),
    (("10PB", 6.0, 4.0), np.array([0.2863, 0.2747, 30.0500])),
    (("10PB", 6.0, 6.0), np.array([0.2740, 0.2533, 30.0500])),
    (("10PB", 6.0, 8.0), np.array([0.2637, 0.2352, 30.0500])),
    (("2.5P", 6.0, 2.0), np.array([0.3016, 0.2960, 30.0500])),
    (("2.5P", 6.0, 4.0), np.array([0.2932, 0.2759, 30.0500])),
    (("2.5P", 6.0, 6.0), np.array([0.2842, 0.2550, 30.0500])),
    (("2.5P", 6.0, 8.0), np.array([0.2770, 0.2372, 30.0500])),
    (("5P", 6.0, 2.0), np.array([0.3050, 0.2967, 30.0500])),
    (("5P", 6.0, 4.0), np.array([0.3001, 0.2778, 30.0500])),
    (("5P", 6.0, 6.0), np.array([0.2950, 0.2585, 30.0500])),
    (("5P", 6.0, 8.0), np.array([0.2905, 0.2421, 30.0500])),
    (("7.5P", 6.0, 2.0), np.array([0.3107, 0.2993, 30.0500])),
    (("7.5P", 6.0, 4.0), np.array([0.3107, 0.2831, 30.0500])),
    (("7.5P", 6.0, 6.0), np.array([0.3101, 0.2650, 30.0500])),
    (("7.5P", 6.0, 8.0), np.array([0.3099, 0.2502, 30.0500])),
    (("10P", 6.0, 2.0), np.array([0.3146, 0.3018, 30.0500])),
    (("10P", 6.0, 4.0), np.array([0.3181, 0.2871, 30.0500])),
    (("10P", 6.0, 6.0), np.array([0.3226, 0.2716, 30.0500])),
    (("10P", 6.0, 8.0), np.array([0.3259, 0.2584, 30.0500])),
    (("2.5RP", 6.0, 2.0), np.array([0.3188, 0.3048, 30.0500])),
    (("2.5RP", 6.0, 4.0), np.array([0.3272, 0.2929, 30.0500])),
    (("2.5RP", 6.0, 6.0), np.array([0.3362, 0.2799, 30.0500])),
    (("2.5RP", 6.0, 8.0), np.array([0.3437, 0.2688, 30.0500])),
    (("2.5RP", 6.0, 10.0), np.array([0.3509, 0.2578, 30.0500])),
    (("5RP", 6.0, 2.0), np.array([0.3232, 0.3085, 30.0500])),
    (("5RP", 6.0, 4.0), np.array([0.3371, 0.3001, 30.0500])),
    (("5RP", 6.0, 6.0), np.array([0.3520, 0.2904, 30.0500])),
    (("5RP", 6.0, 8.0), np.array([0.3648, 0.2820, 30.0500])),
    (("5RP", 6.0, 10.0), np.array([0.3769, 0.2738, 30.0500])),
    (("7.5RP", 6.0, 2.0), np.array([0.3261, 0.3113, 30.0500])),
    (("7.5RP", 6.0, 4.0), np.array([0.3439, 0.3056, 30.0500])),
    (("7.5RP", 6.0, 6.0), np.array([0.3635, 0.2987, 30.0500])),
    (("7.5RP", 6.0, 8.0), np.array([0.3791, 0.2929, 30.0500])),
    (("7.5RP", 6.0, 10.0), np.array([0.3960, 0.2860, 30.0500])),
    (("10RP", 7.0, 2.0), np.array([0.3258, 0.3148, 43.0600])),
    (("10RP", 7.0, 4.0), np.array([0.3446, 0.3125, 43.0600])),
    (("10RP", 7.0, 6.0), np.array([0.3648, 0.3098, 43.0600])),
    (("10RP", 7.0, 8.0), np.array([0.3851, 0.3067, 43.0600])),
    (("2.5R", 7.0, 2.0), np.array([0.3284, 0.3170, 43.0600])),
    (("2.5R", 7.0, 4.0), np.array([0.3499, 0.3171, 43.0600])),
    (("2.5R", 7.0, 6.0), np.array([0.3728, 0.3170, 43.0600])),
    (("2.5R", 7.0, 8.0), np.array([0.3961, 0.3160, 43.0600])),
    (("5R", 7.0, 2.0), np.array([0.3306, 0.3190, 43.0600])),
    (("5R", 7.0, 4.0), np.array([0.3552, 0.3222, 43.0600])),
    (("5R", 7.0, 6.0), np.array([0.3805, 0.3244, 43.0600])),
    (("5R", 7.0, 8.0), np.array([0.4067, 0.3256, 43.0600])),
    (("7.5R", 7.0, 2.0), np.array([0.3335, 0.3220, 43.0600])),
    (("7.5R", 7.0, 4.0), np.array([0.3611, 0.3282, 43.0600])),
    (("7.5R", 7.0, 6.0), np.array([0.3888, 0.3336, 43.0600])),
    (("7.5R", 7.0, 8.0), np.array([0.4196, 0.3382, 43.0600])),
    (("10R", 7.0, 2.0), np.array([0.3360, 0.3253, 43.0600])),
    (("10R", 7.0, 4.0), np.array([0.3671, 0.3360, 43.0600])),
    (("10R", 7.0, 6.0), np.array([0.3984, 0.3452, 43.0600])),
    (("10R", 7.0, 8.0), np.array([0.4308, 0.3533, 43.0600])),
    (("10R", 7.0, 10.0), np.array([0.4600, 0.3596, 43.0600])),
    (("2.5YR", 7.0, 2.0), np.array([0.3392, 0.3298, 43.0600])),
    (("2.5YR", 7.0, 4.0), np.array([0.3715, 0.3439, 43.0600])),
    (("2.5YR", 7.0, 6.0), np.array([0.4053, 0.3570, 43.0600])),
    (("2.5YR", 7.0, 8.0), np.array([0.4371, 0.3679, 43.0600])),
    (("2.5YR", 7.0, 10.0), np.array([0.4671, 0.3768, 43.0600])),
    (("5YR", 7.0, 2.0), np.array([0.3421, 0.3349, 43.0600])),
    (("5YR", 7.0, 4.0), np.array([0.3750, 0.3530, 43.0600])),
    (("5YR", 7.0, 6.0), np.array([0.4091, 0.3701, 43.0600])),
    (("5YR", 7.0, 8.0), np.array([0.4402, 0.3842, 43.0600])),
    (("5YR", 7.0, 10.0), np.array([0.4711, 0.3972, 43.0600])),
    (("7.5YR", 7.0, 2.0), np.array([0.3437, 0.3397, 43.0600])),
    (("7.5YR", 7.0, 4.0), np.array([0.3772, 0.3613, 43.0600])),
    (("7.5YR", 7.0, 6.0), np.array([0.4107, 0.3820, 43.0600])),
    (("7.5YR", 7.0, 8.0), np.array([0.4415, 0.3996, 43.0600])),
    (("7.5YR", 7.0, 10.0), np.array([0.4704, 0.4151, 43.0600])),
    (("10YR", 7.0, 2.0), np.array([0.3443, 0.3454, 43.0600])),
    (("10YR", 7.0, 4.0), np.array([0.3778, 0.3719, 43.0600])),
    (("10YR", 7.0, 6.0), np.array([0.4102, 0.3960, 43.0600])),
    (("10YR", 7.0, 8.0), np.array([0.4399, 0.4164, 43.0600])),
    (("10YR", 7.0, 10.0), np.array([0.4667, 0.4335, 43.0600])),
    (("2.5Y", 7.0, 2.0), np.array([0.3436, 0.3507, 43.0600])),
    (("2.5Y", 7.0, 4.0), np.array([0.3761, 0.3800, 43.0600])),
    (("2.5Y", 7.0, 6.0), np.array([0.4073, 0.4073, 43.0600])),
    (("2.5Y", 7.0, 8.0), np.array([0.4353, 0.4312, 43.0600])),
    (("2.5Y", 7.0, 10.0), np.array([0.4606, 0.4516, 43.0600])),
    (("5Y", 7.0, 2.0), np.array([0.3419, 0.3540, 43.0600])),
    (("5Y", 7.0, 4.0), np.array([0.3718, 0.3885, 43.0600])),
    (("5Y", 7.0, 6.0), np.array([0.4009, 0.4198, 43.0600])),
    (("5Y", 7.0, 8.0), np.array([0.4271, 0.4462, 43.0600])),
    (("5Y", 7.0, 10.0), np.array([0.4509, 0.4696, 43.0600])),
    (("7.5Y", 7.0, 2.0), np.array([0.3396, 0.3558, 43.0600])),
    (("7.5Y", 7.0, 4.0), np.array([0.3677, 0.3925, 43.0600])),
    (("7.5Y", 7.0, 6.0), np.array([0.3943, 0.4264, 43.0600])),
    (("7.5Y", 7.0, 8.0), np.array([0.4184, 0.4568, 43.0600])),
    (("7.5Y", 7.0, 10.0), np.array([0.4400, 0.4830, 43.0600])),
    (("10Y", 7.0, 2.0), np.array([0.3369, 0.3569, 43.0600])),
    (("10Y", 7.0, 4.0), np.array([0.3624, 0.3951, 43.0600])),
    (("10Y", 7.0, 6.0), np.array([0.3864, 0.4305, 43.0600])),
    (("10Y", 7.0, 8.0), np.array([0.4090, 0.4641, 43.0600])),
    (("2.5GY", 7.0, 2.0), np.array([0.3328, 0.3569, 43.0600])),
    (("2.5GY", 7.0, 4.0), np.array([0.3534, 0.3953, 43.0600])),
    (("2.5GY", 7.0, 6.0), np.array([0.3728, 0.4316, 43.0600])),
    (("2.5GY", 7.0, 8.0), np.array([0.3919, 0.4684, 43.0600])),
    (("2.5GY", 7.0, 10.0), np.array([0.4091, 0.5030, 43.0600])),
    (("5GY", 7.0, 2.0), np.array([0.3284, 0.3559, 43.0600])),
    (("5GY", 7.0, 4.0), np.array([0.3437, 0.3929, 43.0600])),
    (("5GY", 7.0, 6.0), np.array([0.3581, 0.4291, 43.0600])),
    (("5GY", 7.0, 8.0), np.array([0.3722, 0.4669, 43.0600])),
    (("5GY", 7.0, 10.0), np.array([0.3852, 0.5051, 43.0600])),
    (("7.5GY", 7.0, 2.0), np.array([0.3190, 0.3516, 43.0600])),
    (("7.5GY", 7.0, 4.0), np.array([0.3267, 0.3848, 43.0600])),
    (("7.5GY", 7.0, 6.0), np.array([0.3341, 0.4191, 43.0600])),
    (("7.5GY", 7.0, 8.0), np.array([0.3406, 0.4558, 43.0600])),
    (("7.5GY", 7.0, 10.0), np.array([0.3461, 0.4950, 43.0600])),
    (("10GY", 7.0, 2.0), np.array([0.3117, 0.3469, 43.0600])),
    (("10GY", 7.0, 4.0), np.array([0.3133, 0.3764, 43.0600])),
    (("10GY", 7.0, 6.0), np.array([0.3142, 0.4058, 43.0600])),
    (("10GY", 7.0, 8.0), np.array([0.3140, 0.4387, 43.0600])),
    (("2.5G", 7.0, 2.0), np.array([0.3047, 0.3413, 43.0600])),
    (("2.5G", 7.0, 4.0), np.array([0.2992, 0.3644, 43.0600])),
    (("2.5G", 7.0, 6.0), np.array([0.2933, 0.3873, 43.0600])),
    (("2.5G", 7.0, 8.0), np.array([0.2861, 0.4129, 43.0600])),
    (("5G", 7.0, 2.0), np.array([0.3001, 0.3366, 43.0600])),
    (("5G", 7.0, 4.0), np.array([0.2902, 0.3548, 43.0600])),
    (("5G", 7.0, 6.0), np.array([0.2801, 0.3721, 43.0600])),
    (("7.5G", 7.0, 2.0), np.array([0.2972, 0.3333, 43.0600])),
    (("7.5G", 7.0, 4.0), np.array([0.2850, 0.3482, 43.0600])),
    (("7.5G", 7.0, 6.0), np.array([0.2728, 0.3622, 43.0600])),
    (("10G", 7.0, 2.0), np.array([0.2945, 0.3297, 43.0600])),
    (("10G", 7.0, 4.0), np.array([0.2803, 0.3415, 43.0600])),
    (("2.5BG", 7.0, 2.0), np.array([0.2927, 0.3269, 43.0600])),
    (("2.5BG", 7.0, 4.0), np.array([0.2764, 0.3354, 43.0600])),
    (("5BG", 7.0, 2.0), np.array([0.2898, 0.3225, 43.0600])),
    (("5BG", 7.0, 4.0), np.array([0.2712, 0.3269, 43.0600])),
    (("7.5BG", 7.0, 2.0), np.array([0.2878, 0.3182, 43.0600])),
    (("7.5BG", 7.0, 4.0), np.array([0.2671, 0.3189, 43.0600])),
    (("10BG", 7.0, 2.0), np.array([0.2869, 0.3143, 43.0600])),
    (("10BG", 7.0, 4.0), np.array([0.2642, 0.3109, 43.0600])),
    (("2.5B", 7.0, 2.0), np.array([0.2867, 0.3110, 43.0600])),
    (("2.5B", 7.0, 4.0), np.array([0.2629, 0.3038, 43.0600])),
    (("2.5B", 7.0, 6.0), np.array([0.2418, 0.2960, 43.0600])),
    (("5B", 7.0, 2.0), np.array([0.2875, 0.3078, 43.0600])),
    (("5B", 7.0, 4.0), np.array([0.2633, 0.2972, 43.0600])),
    (("5B", 7.0, 6.0), np.array([0.2410, 0.2854, 43.0600])),
    (("7.5B", 7.0, 2.0), np.array([0.2888, 0.3058, 43.0600])),
    (("7.5B", 7.0, 4.0), np.array([0.2651, 0.2927, 43.0600])),
    (("7.5B", 7.0, 6.0), np.array([0.2436, 0.2787, 43.0600])),
    (("10B", 7.0, 2.0), np.array([0.2908, 0.3039, 43.0600])),
    (("10B", 7.0, 4.0), np.array([0.2685, 0.2886, 43.0600])),
    (("10B", 7.0, 6.0), np.array([0.2478, 0.2728, 43.0600])),
    (("2.5PB", 7.0, 2.0), np.array([0.2932, 0.3025, 43.0600])),
    (("2.5PB", 7.0, 4.0), np.array([0.2729, 0.2848, 43.0600])),
    (("2.5PB", 7.0, 6.0), np.array([0.2538, 0.2677, 43.0600])),
    (("5PB", 7.0, 2.0), np.array([0.2952, 0.3011, 43.0600])),
    (("5PB", 7.0, 4.0), np.array([0.2773, 0.2828, 43.0600])),
    (("5PB", 7.0, 6.0), np.array([0.2596, 0.2643, 43.0600])),
    (("7.5PB", 7.0, 2.0), np.array([0.2982, 0.3003, 43.0600])),
    (("7.5PB", 7.0, 4.0), np.array([0.2833, 0.2809, 43.0600])),
    (("7.5PB", 7.0, 6.0), np.array([0.2687, 0.2612, 43.0600])),
    (("10PB", 7.0, 2.0), np.array([0.3005, 0.3000, 43.0600])),
    (("10PB", 7.0, 4.0), np.array([0.2886, 0.2801, 43.0600])),
    (("10PB", 7.0, 6.0), np.array([0.2776, 0.2612, 43.0600])),
    (("2.5P", 7.0, 2.0), np.array([0.3031, 0.3000, 43.0600])),
    (("2.5P", 7.0, 4.0), np.array([0.2950, 0.2810, 43.0600])),
    (("2.5P", 7.0, 6.0), np.array([0.2873, 0.2633, 43.0600])),
    (("5P", 7.0, 2.0), np.array([0.3059, 0.3010, 43.0600])),
    (("5P", 7.0, 4.0), np.array([0.3009, 0.2831, 43.0600])),
    (("5P", 7.0, 6.0), np.array([0.2961, 0.2663, 43.0600])),
    (("7.5P", 7.0, 2.0), np.array([0.3109, 0.3037, 43.0600])),
    (("7.5P", 7.0, 4.0), np.array([0.3111, 0.2880, 43.0600])),
    (("7.5P", 7.0, 6.0), np.array([0.3111, 0.2730, 43.0600])),
    (("7.5P", 7.0, 8.0), np.array([0.3109, 0.2584, 43.0600])),
    (("10P", 7.0, 2.0), np.array([0.3138, 0.3054, 43.0600])),
    (("10P", 7.0, 4.0), np.array([0.3181, 0.2920, 43.0600])),
    (("10P", 7.0, 6.0), np.array([0.3221, 0.2786, 43.0600])),
    (("10P", 7.0, 8.0), np.array([0.3256, 0.2654, 43.0600])),
    (("2.5RP", 7.0, 2.0), np.array([0.3170, 0.3076, 43.0600])),
    (("2.5RP", 7.0, 4.0), np.array([0.3254, 0.2971, 43.0600])),
    (("2.5RP", 7.0, 6.0), np.array([0.3338, 0.2854, 43.0600])),
    (("2.5RP", 7.0, 8.0), np.array([0.3417, 0.2745, 43.0600])),
    (("5RP", 7.0, 2.0), np.array([0.3206, 0.3104, 43.0600])),
    (("5RP", 7.0, 4.0), np.array([0.3332, 0.3032, 43.0600])),
    (("5RP", 7.0, 6.0), np.array([0.3470, 0.2949, 43.0600])),
    (("5RP", 7.0, 8.0), np.array([0.3603, 0.2869, 43.0600])),
    (("7.5RP", 7.0, 2.0), np.array([0.3232, 0.3125, 43.0600])),
    (("7.5RP", 7.0, 4.0), np.array([0.3389, 0.3079, 43.0600])),
    (("7.5RP", 7.0, 6.0), np.array([0.3562, 0.3022, 43.0600])),
    (("7.5RP", 7.0, 8.0), np.array([0.3722, 0.2963, 43.0600])),
    (("10RP", 8.0, 2.0), np.array([0.3218, 0.3152, 59.1000])),
    (("10RP", 8.0, 4.0), np.array([0.3412, 0.3135, 59.1000])),
    (("10RP", 8.0, 6.0), np.array([0.3600, 0.3112, 59.1000])),
    (("2.5R", 8.0, 2.0), np.array([0.3236, 0.3169, 59.1000])),
    (("2.5R", 8.0, 4.0), np.array([0.3460, 0.3177, 59.1000])),
    (("2.5R", 8.0, 6.0), np.array([0.3671, 0.3175, 59.1000])),
    (("5R", 8.0, 2.0), np.array([0.3254, 0.3186, 59.1000])),
    (("5R", 8.0, 4.0), np.array([0.3510, 0.3224, 59.1000])),
    (("7.5R", 8.0, 2.0), np.array([0.3277, 0.3211, 59.1000])),
    (("7.5R", 8.0, 4.0), np.array([0.3564, 0.3279, 59.1000])),
    (("10R", 8.0, 2.0), np.array([0.3301, 0.3237, 59.1000])),
    (("10R", 8.0, 4.0), np.array([0.3621, 0.3349, 59.1000])),
    (("2.5YR", 8.0, 2.0), np.array([0.3334, 0.3276, 59.1000])),
    (("2.5YR", 8.0, 4.0), np.array([0.3667, 0.3429, 59.1000])),
    (("5YR", 8.0, 2.0), np.array([0.3373, 0.3330, 59.1000])),
    (("5YR", 8.0, 4.0), np.array([0.3690, 0.3510, 59.1000])),
    (("7.5YR", 8.0, 2.0), np.array([0.3395, 0.3379, 59.1000])),
    (("7.5YR", 8.0, 4.0), np.array([0.3699, 0.3586, 59.1000])),
    (("7.5YR", 8.0, 6.0), np.array([0.4000, 0.3770, 59.1000])),
    (("10YR", 8.0, 2.0), np.array([0.3407, 0.3434, 59.1000])),
    (("10YR", 8.0, 4.0), np.array([0.3701, 0.3674, 59.1000])),
    (("10YR", 8.0, 6.0), np.array([0.3994, 0.3896, 59.1000])),
    (("2.5Y", 8.0, 2.0), np.array([0.3406, 0.3484, 59.1000])),
    (("2.5Y", 8.0, 4.0), np.array([0.3684, 0.3751, 59.1000])),
    (("2.5Y", 8.0, 6.0), np.array([0.3969, 0.4009, 59.1000])),
    (("2.5Y", 8.0, 8.0), np.array([0.4231, 0.4231, 59.1000])),
    (("5Y", 8.0, 2.0), np.array([0.3394, 0.3518, 59.1000])),
    (("5Y", 8.0, 4.0), np.array([0.3650, 0.3826, 59.1000])),
    (("5Y", 8.0, 6.0), np.array([0.3913, 0.4117, 59.1000])),
    (("5Y", 8.0, 8.0), np.array([0.4158, 0.4378, 59.1000])),
    (("5Y", 8.0, 10.0), np.array([0.4376, 0.4601, 59.1000])),
    (("5Y", 8.0, 12.0), np.array([0.4562, 0.4788, 59.1000])),
    (("7.5Y", 8.0, 2.0), np.array([0.3379, 0.3540, 59.1000])),
    (("7.5Y", 8.0, 4.0), np.array([0.3622, 0.3861, 59.1000])),
    (("7.5Y", 8.0, 6.0), np.array([0.3862, 0.4175, 59.1000])),
    (("7.5Y", 8.0, 8.0), np.array([0.4088, 0.4466, 59.1000])),
    (("7.5Y", 8.0, 10.0), np.array([0.4283, 0.4712, 59.1000])),
    (("10Y", 8.0, 2.0), np.array([0.3359, 0.3552, 59.1000])),
    (("10Y", 8.0, 4.0), np.array([0.3581, 0.3883, 59.1000])),
    (("10Y", 8.0, 6.0), np.array([0.3803, 0.4216, 59.1000])),
    (("10Y", 8.0, 8.0), np.array([0.4008, 0.4520, 59.1000])),
    (("2.5GY", 8.0, 2.0), np.array([0.3327, 0.3555, 59.1000])),
    (("2.5GY", 8.0, 4.0), np.array([0.3504, 0.3887, 59.1000])),
    (("2.5GY", 8.0, 6.0), np.array([0.3690, 0.4230, 59.1000])),
    (("2.5GY", 8.0, 8.0), np.array([0.3858, 0.4550, 59.1000])),
    (("5GY", 8.0, 2.0), np.array([0.3284, 0.3542, 59.1000])),
    (("5GY", 8.0, 4.0), np.array([0.3433, 0.3872, 59.1000])),
    (("5GY", 8.0, 6.0), np.array([0.3573, 0.4214, 59.1000])),
    (("5GY", 8.0, 8.0), np.array([0.3696, 0.4542, 59.1000])),
    (("7.5GY", 8.0, 2.0), np.array([0.3194, 0.3502, 59.1000])),
    (("7.5GY", 8.0, 4.0), np.array([0.3266, 0.3809, 59.1000])),
    (("7.5GY", 8.0, 6.0), np.array([0.3339, 0.4129, 59.1000])),
    (("7.5GY", 8.0, 8.0), np.array([0.3408, 0.4452, 59.1000])),
    (("10GY", 8.0, 2.0), np.array([0.3121, 0.3459, 59.1000])),
    (("10GY", 8.0, 4.0), np.array([0.3140, 0.3727, 59.1000])),
    (("10GY", 8.0, 6.0), np.array([0.3150, 0.4014, 59.1000])),
    (("2.5G", 8.0, 2.0), np.array([0.3053, 0.3404, 59.1000])),
    (("2.5G", 8.0, 4.0), np.array([0.3009, 0.3614, 59.1000])),
    (("2.5G", 8.0, 6.0), np.array([0.2952, 0.3851, 59.1000])),
    (("5G", 8.0, 2.0), np.array([0.3009, 0.3359, 59.1000])),
    (("5G", 8.0, 4.0), np.array([0.2924, 0.3523, 59.1000])),
    (("5G", 8.0, 6.0), np.array([0.2822, 0.3702, 59.1000])),
    (("7.5G", 8.0, 2.0), np.array([0.2981, 0.3326, 59.1000])),
    (("7.5G", 8.0, 4.0), np.array([0.2874, 0.3464, 59.1000])),
    (("10G", 8.0, 2.0), np.array([0.2957, 0.3293, 59.1000])),
    (("2.5BG", 8.0, 2.0), np.array([0.2940, 0.3268, 59.1000])),
    (("5BG", 8.0, 2.0), np.array([0.2919, 0.3228, 59.1000])),
    (("7.5BG", 8.0, 2.0), np.array([0.2900, 0.3183, 59.1000])),
    (("10BG", 8.0, 2.0), np.array([0.2894, 0.3152, 59.1000])),
    (("2.5B", 8.0, 2.0), np.array([0.2897, 0.3124, 59.1000])),
    (("2.5B", 8.0, 4.0), np.array([0.2668, 0.3067, 59.1000])),
    (("5B", 8.0, 2.0), np.array([0.2908, 0.3096, 59.1000])),
    (("5B", 8.0, 4.0), np.array([0.2671, 0.2998, 59.1000])),
    (("7.5B", 8.0, 2.0), np.array([0.2922, 0.3077, 59.1000])),
    (("7.5B", 8.0, 4.0), np.array([0.2688, 0.2956, 59.1000])),
    (("10B", 8.0, 2.0), np.array([0.2935, 0.3062, 59.1000])),
    (("10B", 8.0, 4.0), np.array([0.2718, 0.2911, 59.1000])),
    (("2.5PB", 8.0, 2.0), np.array([0.2957, 0.3047, 59.1000])),
    (("2.5PB", 8.0, 4.0), np.array([0.2758, 0.2879, 59.1000])),
    (("5PB", 8.0, 2.0), np.array([0.2974, 0.3039, 59.1000])),
    (("7.5PB", 8.0, 2.0), np.array([0.3003, 0.3034, 59.1000])),
    (("10PB", 8.0, 2.0), np.array([0.3027, 0.3035, 59.1000])),
    (("2.5P", 8.0, 2.0), np.array([0.3048, 0.3040, 59.1000])),
    (("2.5P", 8.0, 4.0), np.array([0.2962, 0.2850, 59.1000])),
    (("5P", 8.0, 2.0), np.array([0.3065, 0.3047, 59.1000])),
    (("5P", 8.0, 4.0), np.array([0.3012, 0.2868, 59.1000])),
    (("7.5P", 8.0, 2.0), np.array([0.3107, 0.3070, 59.1000])),
    (("7.5P", 8.0, 4.0), np.array([0.3114, 0.2915, 59.1000])),
    (("10P", 8.0, 2.0), np.array([0.3131, 0.3084, 59.1000])),
    (("10P", 8.0, 4.0), np.array([0.3175, 0.2955, 59.1000])),
    (("2.5RP", 8.0, 2.0), np.array([0.3154, 0.3100, 59.1000])),
    (("2.5RP", 8.0, 4.0), np.array([0.3239, 0.3000, 59.1000])),
    (("2.5RP", 8.0, 6.0), np.array([0.3327, 0.2898, 59.1000])),
    (("5RP", 8.0, 2.0), np.array([0.3180, 0.3120, 59.1000])),
    (("5RP", 8.0, 4.0), np.array([0.3308, 0.3052, 59.1000])),
    (("5RP", 8.0, 6.0), np.array([0.3440, 0.2978, 59.1000])),
    (("7.5RP", 8.0, 2.0), np.array([0.3200, 0.3136, 59.1000])),
    (("7.5RP", 8.0, 4.0), np.array([0.3360, 0.3092, 59.1000])),
    (("7.5RP", 8.0, 6.0), np.array([0.3521, 0.3042, 59.1000])),
)
"""
*1929* published *Munsell* colours as a *tuple* as follows::

    (
        (('hue', 'value', 'chroma'), np.array(['x', 'y', 'Y'])),
        ...,
        (('hue', 'value', 'chroma'), np.array(['x', 'y', 'Y'])),
    )

References
----------
:cite:`MunsellColorSciencec`
"""
