import unittest
from test import test_tools

test_tools.skip_if_missing("peg_generator")
with test_tools.imports_under_tool("peg_generator"):
    from pegen.grammar_parser import GeneratedParser as GrammarParser
    from pegen.validator import SubRuleValidator, ValidationError
    from pegen.validator import RaiseRuleValidator, CutValidator
    from pegen.testutil import parse_string
    from pegen.grammar import Grammar


class TestPegen(unittest.TestCase):
    def test_rule_with_no_collision(self) -> None:
        grammar_source = """
        start: bad_rule
        sum:
            | NAME '-' NAME
            | NAME '+' NAME
        """
        grammar: Grammar = parse_string(grammar_source, GrammarParser)
        validator = SubRuleValidator(grammar)
        for rule_name, rule in grammar.rules.items():
            validator.validate_rule(rule_name, rule)

    def test_rule_with_simple_collision(self) -> None:
        grammar_source = """
        start: bad_rule
        sum:
            | NAME '+' NAME
            | NAME '+' NAME ';'
        """
        grammar: Grammar = parse_string(grammar_source, GrammarParser)
        validator = SubRuleValidator(grammar)
        with self.assertRaises(ValidationError):
            for rule_name, rule in grammar.rules.items():
                validator.validate_rule(rule_name, rule)

    def test_rule_with_collision_after_some_other_rules(self) -> None:
        grammar_source = """
        start: bad_rule
        sum:
            | NAME '+' NAME
            | NAME '*' NAME ';'
            | NAME '-' NAME
            | NAME '+' NAME ';'
        """
        grammar: Grammar = parse_string(grammar_source, GrammarParser)
        validator = SubRuleValidator(grammar)
        with self.assertRaises(ValidationError):
            for rule_name, rule in grammar.rules.items():
                validator.validate_rule(rule_name, rule)

    def test_raising_valid_rule(self) -> None:
        grammar_source = """
        start: NAME { RAISE_SYNTAX_ERROR("this is not allowed") }
        """
        grammar: Grammar = parse_string(grammar_source, GrammarParser)
        validator = RaiseRuleValidator(grammar)
        with self.assertRaises(ValidationError):
            for rule_name, rule in grammar.rules.items():
                validator.validate_rule(rule_name, rule)

    def test_cut_validator(self) -> None:
        grammar_source = """
        star: (OP ~ OP)*
        plus: (OP ~ OP)+
        bracket: [OP ~ OP]
        gather: OP.(OP ~ OP)+
        nested: [OP | NAME ~ OP]
        """
        grammar: Grammar = parse_string(grammar_source, GrammarParser)
        validator = CutValidator(grammar)
        for rule_name, rule in grammar.rules.items():
            with self.subTest(rule_name):
                with self.assertRaises(ValidationError):
                    validator.validate_rule(rule_name, rule)
