# Using the SDK
After they are constructed, individual service clients are very similar to other SDKs, such as Java and .NET. This section explains how core works, how to use each feature, and how to construct an individual client.

The aws-cpp-sdk-core is the heart of the system and does the heavy lifting. You can write a client to connect to any AWS service using just the core, and the individual service clients are available to help make the process a little easier.

## Running integration tests:
Several directories are appended with \*integration-tests. After building your project, you can run these executables to ensure everything works properly.

## Build Defines
If you dynamically link to the SDK you will need to define the USE_IMPORT_EXPORT symbol for all build targets using the SDK.
If you wish to install your own memory manager to handle allocations made by the SDK, you will need to pass the CUSTOM_MEMORY_MANAGEMENT cmake parameter (-DCUSTOM_MEMORY_MANAGEMENT) as well as define USE_AWS_MEMORY_MANAGEMENT in all build targets dependent on the SDK.

Note, if you use our export file, this will be handled automatically for you. We recommend you use our export file to handle this for you:
https://aws.amazon.com/blogs/developer/using-cmake-exports-with-the-aws-sdk-for-c/

## Initialization and Shutdown
We avoid global and static state where ever possible. However, on some platforms, dependencies need to be globally initialized. Also, we have a few global options such as
logging, memory management, http factories, and crypto factories. As a result, before using the SDK you MUST call our global initialization function. When you are finished using the SDK you should call our cleanup function.

All code using the AWS SDK and C++ should have at least the following:

```cpp
#include <aws/core/Aws.h>
...

Aws::SDKOptions options;
Aws::InitAPI(options);

//use the sdk

Aws::ShutdownAPI(options);
```

Due to the way memory managers work, many of the configuration options take closures instead of pointers directly in order to ensure that the memory manager
is installed prior to any memory allocations occuring.

Here are a few recipes:

* Just use defaults:
```cpp
Aws::SDKOptions options;
Aws::InitAPI(options);
.....
Aws::ShutdownAPI(options);
```

* Install custom memory manager:
```cpp
MyMemoryManager memoryManager;

Aws::SDKOptions options;
options.memoryManagementOptions.memoryManager = &memoryManager;
Aws::InitAPI(options);
.....
Aws::ShutdownAPI(options);
```

* Override default http client factory:
```cpp
Aws::SDKOptions options;
options.httpOptions.httpClientFactory_create_fn = [](){ return Aws::MakeShared<MyCustomHttpClientFactory>("ALLOC_TAG", arg1); };
Aws::InitAPI(options);
.....
Aws::ShutdownAPI(options);
```
## Logging
The AWS SDK for C++ includes logging support that you can configure. When initializing the logging system, you can control the filter level and the logging target (file with a name that has a configurable prefix or a stream). The log file generated by the prefix option rolls over once per hour to allow for archiving or deleting log files.

You can provide your own logger. However, it is incredibly simple to use the default logger we've already provided:

In your main function:

```cpp
SDKOptions options;
options.loggingOptions.logLevel = Aws::Utils::Logging::LogLevel::Info;
Aws::InitAPI(options);
//do SDK stuff;
Aws::ShutdownAPI(options);
```