﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/chime/Chime_EXPORTS.h>
#include <aws/chime/model/TranscribeLanguageCode.h>
#include <aws/chime/model/TranscribeVocabularyFilterMethod.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/chime/model/TranscribeRegion.h>
#include <aws/chime/model/TranscribePartialResultsStability.h>
#include <aws/chime/model/TranscribeContentIdentificationType.h>
#include <aws/chime/model/TranscribeContentRedactionType.h>
#include <utility>

namespace Aws
{
namespace Utils
{
namespace Json
{
  class JsonValue;
  class JsonView;
} // namespace Json
} // namespace Utils
namespace Chime
{
namespace Model
{

  /**
   * <p>Settings specific for Amazon Transcribe as the live transcription engine.</p>
   * <p>If you specify an invalid combination of parameters, a
   * <code>TranscriptFailed</code> event will be sent with the contents of the
   * <code>BadRequestException</code> generated by Amazon Transcribe. For more
   * information on each parameter and which combinations are valid, refer to the <a
   * href="https://docs.aws.amazon.com/transcribe/latest/APIReference/API_streaming_StartStreamTranscription.html">StartStreamTranscription</a>
   * API in the <i>Amazon Transcribe Developer Guide</i>.</p><p><h3>See Also:</h3>  
   * <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/EngineTranscribeSettings">AWS
   * API Reference</a></p>
   */
  class EngineTranscribeSettings
  {
  public:
    AWS_CHIME_API EngineTranscribeSettings();
    AWS_CHIME_API EngineTranscribeSettings(Aws::Utils::Json::JsonView jsonValue);
    AWS_CHIME_API EngineTranscribeSettings& operator=(Aws::Utils::Json::JsonView jsonValue);
    AWS_CHIME_API Aws::Utils::Json::JsonValue Jsonize() const;


    ///@{
    /**
     * <p>Specify the language code that represents the language spoken.</p> <p>If
     * you're unsure of the language spoken in your audio, consider using
     * <code>IdentifyLanguage</code> to enable automatic language identification.</p>
     */
    inline const TranscribeLanguageCode& GetLanguageCode() const{ return m_languageCode; }
    inline bool LanguageCodeHasBeenSet() const { return m_languageCodeHasBeenSet; }
    inline void SetLanguageCode(const TranscribeLanguageCode& value) { m_languageCodeHasBeenSet = true; m_languageCode = value; }
    inline void SetLanguageCode(TranscribeLanguageCode&& value) { m_languageCodeHasBeenSet = true; m_languageCode = std::move(value); }
    inline EngineTranscribeSettings& WithLanguageCode(const TranscribeLanguageCode& value) { SetLanguageCode(value); return *this;}
    inline EngineTranscribeSettings& WithLanguageCode(TranscribeLanguageCode&& value) { SetLanguageCode(std::move(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify how you want your vocabulary filter applied to your transcript.</p>
     * <p> To replace words with <code>***</code>, choose <code>mask</code>.</p> <p>To
     * delete words, choose <code>remove</code>.</p> <p>To flag words without changing
     * them, choose <code>tag</code>.</p>
     */
    inline const TranscribeVocabularyFilterMethod& GetVocabularyFilterMethod() const{ return m_vocabularyFilterMethod; }
    inline bool VocabularyFilterMethodHasBeenSet() const { return m_vocabularyFilterMethodHasBeenSet; }
    inline void SetVocabularyFilterMethod(const TranscribeVocabularyFilterMethod& value) { m_vocabularyFilterMethodHasBeenSet = true; m_vocabularyFilterMethod = value; }
    inline void SetVocabularyFilterMethod(TranscribeVocabularyFilterMethod&& value) { m_vocabularyFilterMethodHasBeenSet = true; m_vocabularyFilterMethod = std::move(value); }
    inline EngineTranscribeSettings& WithVocabularyFilterMethod(const TranscribeVocabularyFilterMethod& value) { SetVocabularyFilterMethod(value); return *this;}
    inline EngineTranscribeSettings& WithVocabularyFilterMethod(TranscribeVocabularyFilterMethod&& value) { SetVocabularyFilterMethod(std::move(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify the name of the custom vocabulary filter that you want to use when
     * processing your transcription. Note that vocabulary filter names are case
     * sensitive. </p> <p>If you use Amazon Transcribe in multiple Regions, the
     * vocabulary filter must be available in Amazon Transcribe in each Region.</p>
     * <p>If you include <code>IdentifyLanguage</code> and want to use one or more
     * vocabulary filters with your transcription, use the
     * <code>VocabularyFilterNames</code> parameter instead.</p>
     */
    inline const Aws::String& GetVocabularyFilterName() const{ return m_vocabularyFilterName; }
    inline bool VocabularyFilterNameHasBeenSet() const { return m_vocabularyFilterNameHasBeenSet; }
    inline void SetVocabularyFilterName(const Aws::String& value) { m_vocabularyFilterNameHasBeenSet = true; m_vocabularyFilterName = value; }
    inline void SetVocabularyFilterName(Aws::String&& value) { m_vocabularyFilterNameHasBeenSet = true; m_vocabularyFilterName = std::move(value); }
    inline void SetVocabularyFilterName(const char* value) { m_vocabularyFilterNameHasBeenSet = true; m_vocabularyFilterName.assign(value); }
    inline EngineTranscribeSettings& WithVocabularyFilterName(const Aws::String& value) { SetVocabularyFilterName(value); return *this;}
    inline EngineTranscribeSettings& WithVocabularyFilterName(Aws::String&& value) { SetVocabularyFilterName(std::move(value)); return *this;}
    inline EngineTranscribeSettings& WithVocabularyFilterName(const char* value) { SetVocabularyFilterName(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify the name of the custom vocabulary that you want to use when
     * processing your transcription. Note that vocabulary names are case
     * sensitive.</p> <p>If you use Amazon Transcribe multiple Regions, the vocabulary
     * must be available in Amazon Transcribe in each Region.</p> <p>If you include
     * <code>IdentifyLanguage</code> and want to use one or more custom vocabularies
     * with your transcription, use the <code>VocabularyNames</code> parameter
     * instead.</p>
     */
    inline const Aws::String& GetVocabularyName() const{ return m_vocabularyName; }
    inline bool VocabularyNameHasBeenSet() const { return m_vocabularyNameHasBeenSet; }
    inline void SetVocabularyName(const Aws::String& value) { m_vocabularyNameHasBeenSet = true; m_vocabularyName = value; }
    inline void SetVocabularyName(Aws::String&& value) { m_vocabularyNameHasBeenSet = true; m_vocabularyName = std::move(value); }
    inline void SetVocabularyName(const char* value) { m_vocabularyNameHasBeenSet = true; m_vocabularyName.assign(value); }
    inline EngineTranscribeSettings& WithVocabularyName(const Aws::String& value) { SetVocabularyName(value); return *this;}
    inline EngineTranscribeSettings& WithVocabularyName(Aws::String&& value) { SetVocabularyName(std::move(value)); return *this;}
    inline EngineTranscribeSettings& WithVocabularyName(const char* value) { SetVocabularyName(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>The AWS Region in which to use Amazon Transcribe.</p> <p>If you don't specify
     * a Region, then the <code>MediaRegion</code> parameter of the <a
     * href="https://docs.aws.amazon.com/chime-sdk/latest/APIReference/API_meeting-chime_CreateMeeting.html">CreateMeeting.html</a>
     * API will be used. However, if Amazon Transcribe is not available in the
     * <code>MediaRegion</code>, then a TranscriptFailed event is sent.</p> <p>Use
     * <code>auto</code> to use Amazon Transcribe in a Region near the meeting’s
     * <code>MediaRegion</code>. For more information, refer to <a
     * href="https://docs.aws.amazon.com/chime-sdk/latest/dg/transcription-options.html#choose-region">Choosing
     * a transcription Region</a> in the <i>Amazon Chime SDK Developer Guide</i>.</p>
     */
    inline const TranscribeRegion& GetRegion() const{ return m_region; }
    inline bool RegionHasBeenSet() const { return m_regionHasBeenSet; }
    inline void SetRegion(const TranscribeRegion& value) { m_regionHasBeenSet = true; m_region = value; }
    inline void SetRegion(TranscribeRegion&& value) { m_regionHasBeenSet = true; m_region = std::move(value); }
    inline EngineTranscribeSettings& WithRegion(const TranscribeRegion& value) { SetRegion(value); return *this;}
    inline EngineTranscribeSettings& WithRegion(TranscribeRegion&& value) { SetRegion(std::move(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Enables partial result stabilization for your transcription. Partial result
     * stabilization can reduce latency in your output, but may impact accuracy.</p>
     */
    inline bool GetEnablePartialResultsStabilization() const{ return m_enablePartialResultsStabilization; }
    inline bool EnablePartialResultsStabilizationHasBeenSet() const { return m_enablePartialResultsStabilizationHasBeenSet; }
    inline void SetEnablePartialResultsStabilization(bool value) { m_enablePartialResultsStabilizationHasBeenSet = true; m_enablePartialResultsStabilization = value; }
    inline EngineTranscribeSettings& WithEnablePartialResultsStabilization(bool value) { SetEnablePartialResultsStabilization(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify the level of stability to use when you enable partial results
     * stabilization (<code>EnablePartialResultsStabilization</code>).</p> <p>Low
     * stability provides the highest accuracy. High stability transcribes faster, but
     * with slightly lower accuracy.</p>
     */
    inline const TranscribePartialResultsStability& GetPartialResultsStability() const{ return m_partialResultsStability; }
    inline bool PartialResultsStabilityHasBeenSet() const { return m_partialResultsStabilityHasBeenSet; }
    inline void SetPartialResultsStability(const TranscribePartialResultsStability& value) { m_partialResultsStabilityHasBeenSet = true; m_partialResultsStability = value; }
    inline void SetPartialResultsStability(TranscribePartialResultsStability&& value) { m_partialResultsStabilityHasBeenSet = true; m_partialResultsStability = std::move(value); }
    inline EngineTranscribeSettings& WithPartialResultsStability(const TranscribePartialResultsStability& value) { SetPartialResultsStability(value); return *this;}
    inline EngineTranscribeSettings& WithPartialResultsStability(TranscribePartialResultsStability&& value) { SetPartialResultsStability(std::move(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Labels all personally identifiable information (PII) identified in your
     * transcript. If you don't include <code>PiiEntityTypes</code>, all PII is
     * identified.</p> <p>You can’t set <code>ContentIdentificationType</code> and
     * <code>ContentRedactionType</code>.</p>
     */
    inline const TranscribeContentIdentificationType& GetContentIdentificationType() const{ return m_contentIdentificationType; }
    inline bool ContentIdentificationTypeHasBeenSet() const { return m_contentIdentificationTypeHasBeenSet; }
    inline void SetContentIdentificationType(const TranscribeContentIdentificationType& value) { m_contentIdentificationTypeHasBeenSet = true; m_contentIdentificationType = value; }
    inline void SetContentIdentificationType(TranscribeContentIdentificationType&& value) { m_contentIdentificationTypeHasBeenSet = true; m_contentIdentificationType = std::move(value); }
    inline EngineTranscribeSettings& WithContentIdentificationType(const TranscribeContentIdentificationType& value) { SetContentIdentificationType(value); return *this;}
    inline EngineTranscribeSettings& WithContentIdentificationType(TranscribeContentIdentificationType&& value) { SetContentIdentificationType(std::move(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Content redaction is performed at the segment level. If you don't include
     * <code>PiiEntityTypes</code>, all PII is redacted.</p> <p>You can’t set
     * <code>ContentIdentificationType</code> and
     * <code>ContentRedactionType</code>.</p>
     */
    inline const TranscribeContentRedactionType& GetContentRedactionType() const{ return m_contentRedactionType; }
    inline bool ContentRedactionTypeHasBeenSet() const { return m_contentRedactionTypeHasBeenSet; }
    inline void SetContentRedactionType(const TranscribeContentRedactionType& value) { m_contentRedactionTypeHasBeenSet = true; m_contentRedactionType = value; }
    inline void SetContentRedactionType(TranscribeContentRedactionType&& value) { m_contentRedactionTypeHasBeenSet = true; m_contentRedactionType = std::move(value); }
    inline EngineTranscribeSettings& WithContentRedactionType(const TranscribeContentRedactionType& value) { SetContentRedactionType(value); return *this;}
    inline EngineTranscribeSettings& WithContentRedactionType(TranscribeContentRedactionType&& value) { SetContentRedactionType(std::move(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify which types of personally identifiable information (PII) you want to
     * redact in your transcript. You can include as many types as you'd like, or you
     * can select <code>ALL</code>.</p> <p>Values must be comma-separated and can
     * include: <code>ADDRESS</code>, <code>BANK_ACCOUNT_NUMBER</code>,
     * <code>BANK_ROUTING</code>, <code>CREDIT_DEBIT_CVV</code>,
     * <code>CREDIT_DEBIT_EXPIRY</code> <code>CREDIT_DEBIT_NUMBER</code>,
     * <code>EMAIL</code>,<code>NAME</code>, <code>PHONE</code>, <code>PIN</code>,
     * <code>SSN</code>, or <code>ALL</code>.</p> <p>Note that if you include
     * <code>PiiEntityTypes</code>, you must also include
     * <code>ContentIdentificationType</code> or <code>ContentRedactionType</code>.</p>
     * <p>If you include <code>ContentRedactionType</code> or
     * <code>ContentIdentificationType</code>, but do not include
     * <code>PiiEntityTypes</code>, all PII is redacted or identified.</p>
     */
    inline const Aws::String& GetPiiEntityTypes() const{ return m_piiEntityTypes; }
    inline bool PiiEntityTypesHasBeenSet() const { return m_piiEntityTypesHasBeenSet; }
    inline void SetPiiEntityTypes(const Aws::String& value) { m_piiEntityTypesHasBeenSet = true; m_piiEntityTypes = value; }
    inline void SetPiiEntityTypes(Aws::String&& value) { m_piiEntityTypesHasBeenSet = true; m_piiEntityTypes = std::move(value); }
    inline void SetPiiEntityTypes(const char* value) { m_piiEntityTypesHasBeenSet = true; m_piiEntityTypes.assign(value); }
    inline EngineTranscribeSettings& WithPiiEntityTypes(const Aws::String& value) { SetPiiEntityTypes(value); return *this;}
    inline EngineTranscribeSettings& WithPiiEntityTypes(Aws::String&& value) { SetPiiEntityTypes(std::move(value)); return *this;}
    inline EngineTranscribeSettings& WithPiiEntityTypes(const char* value) { SetPiiEntityTypes(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify the name of the custom language model that you want to use when
     * processing your transcription. Note that language model names are case
     * sensitive.</p> <p>The language of the specified language model must match the
     * language code. If the languages don't match, the custom language model isn't
     * applied. There are no errors or warnings associated with a language
     * mismatch.</p> <p>If you use Amazon Transcribe in multiple Regions, the custom
     * language model must be available in Amazon Transcribe in each Region.</p>
     */
    inline const Aws::String& GetLanguageModelName() const{ return m_languageModelName; }
    inline bool LanguageModelNameHasBeenSet() const { return m_languageModelNameHasBeenSet; }
    inline void SetLanguageModelName(const Aws::String& value) { m_languageModelNameHasBeenSet = true; m_languageModelName = value; }
    inline void SetLanguageModelName(Aws::String&& value) { m_languageModelNameHasBeenSet = true; m_languageModelName = std::move(value); }
    inline void SetLanguageModelName(const char* value) { m_languageModelNameHasBeenSet = true; m_languageModelName.assign(value); }
    inline EngineTranscribeSettings& WithLanguageModelName(const Aws::String& value) { SetLanguageModelName(value); return *this;}
    inline EngineTranscribeSettings& WithLanguageModelName(Aws::String&& value) { SetLanguageModelName(std::move(value)); return *this;}
    inline EngineTranscribeSettings& WithLanguageModelName(const char* value) { SetLanguageModelName(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Enables automatic language identification for your transcription.</p> <p>If
     * you include <code>IdentifyLanguage</code>, you can optionally use
     * <code>LanguageOptions</code> to include a list of language codes that you think
     * may be present in your audio stream. Including language options can improve
     * transcription accuracy.</p> <p>You can also use <code>PreferredLanguage</code>
     * to include a preferred language. Doing so can help Amazon Transcribe identify
     * the language faster.</p> <p>You must include either <code>LanguageCode</code> or
     * <code>IdentifyLanguage</code>.</p> <p>Language identification can't be combined
     * with custom language models or redaction.</p>
     */
    inline bool GetIdentifyLanguage() const{ return m_identifyLanguage; }
    inline bool IdentifyLanguageHasBeenSet() const { return m_identifyLanguageHasBeenSet; }
    inline void SetIdentifyLanguage(bool value) { m_identifyLanguageHasBeenSet = true; m_identifyLanguage = value; }
    inline EngineTranscribeSettings& WithIdentifyLanguage(bool value) { SetIdentifyLanguage(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify two or more language codes that represent the languages you think may
     * be present in your media; including more than five is not recommended. If you're
     * unsure what languages are present, do not include this parameter.</p>
     * <p>Including language options can improve the accuracy of language
     * identification.</p> <p>If you include <code>LanguageOptions</code>, you must
     * also include <code>IdentifyLanguage</code>.</p>  <p>You can only
     * include one language dialect per language. For example, you cannot include
     * <code>en-US</code> and <code>en-AU</code>.</p> 
     */
    inline const Aws::String& GetLanguageOptions() const{ return m_languageOptions; }
    inline bool LanguageOptionsHasBeenSet() const { return m_languageOptionsHasBeenSet; }
    inline void SetLanguageOptions(const Aws::String& value) { m_languageOptionsHasBeenSet = true; m_languageOptions = value; }
    inline void SetLanguageOptions(Aws::String&& value) { m_languageOptionsHasBeenSet = true; m_languageOptions = std::move(value); }
    inline void SetLanguageOptions(const char* value) { m_languageOptionsHasBeenSet = true; m_languageOptions.assign(value); }
    inline EngineTranscribeSettings& WithLanguageOptions(const Aws::String& value) { SetLanguageOptions(value); return *this;}
    inline EngineTranscribeSettings& WithLanguageOptions(Aws::String&& value) { SetLanguageOptions(std::move(value)); return *this;}
    inline EngineTranscribeSettings& WithLanguageOptions(const char* value) { SetLanguageOptions(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify a preferred language from the subset of languages codes you specified
     * in <code>LanguageOptions</code>.</p> <p>You can only use this parameter if you
     * include <code>IdentifyLanguage</code> and <code>LanguageOptions</code>.</p>
     */
    inline const TranscribeLanguageCode& GetPreferredLanguage() const{ return m_preferredLanguage; }
    inline bool PreferredLanguageHasBeenSet() const { return m_preferredLanguageHasBeenSet; }
    inline void SetPreferredLanguage(const TranscribeLanguageCode& value) { m_preferredLanguageHasBeenSet = true; m_preferredLanguage = value; }
    inline void SetPreferredLanguage(TranscribeLanguageCode&& value) { m_preferredLanguageHasBeenSet = true; m_preferredLanguage = std::move(value); }
    inline EngineTranscribeSettings& WithPreferredLanguage(const TranscribeLanguageCode& value) { SetPreferredLanguage(value); return *this;}
    inline EngineTranscribeSettings& WithPreferredLanguage(TranscribeLanguageCode&& value) { SetPreferredLanguage(std::move(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify the names of the custom vocabularies that you want to use when
     * processing your transcription. Note that vocabulary names are case
     * sensitive.</p> <p>If you use Amazon Transcribe in multiple Regions, the
     * vocabulary must be available in Amazon Transcribe in each Region.</p> <p>If you
     * don't include <code>IdentifyLanguage</code> and want to use a custom vocabulary
     * with your transcription, use the <code>VocabularyName</code> parameter
     * instead.</p>
     */
    inline const Aws::String& GetVocabularyNames() const{ return m_vocabularyNames; }
    inline bool VocabularyNamesHasBeenSet() const { return m_vocabularyNamesHasBeenSet; }
    inline void SetVocabularyNames(const Aws::String& value) { m_vocabularyNamesHasBeenSet = true; m_vocabularyNames = value; }
    inline void SetVocabularyNames(Aws::String&& value) { m_vocabularyNamesHasBeenSet = true; m_vocabularyNames = std::move(value); }
    inline void SetVocabularyNames(const char* value) { m_vocabularyNamesHasBeenSet = true; m_vocabularyNames.assign(value); }
    inline EngineTranscribeSettings& WithVocabularyNames(const Aws::String& value) { SetVocabularyNames(value); return *this;}
    inline EngineTranscribeSettings& WithVocabularyNames(Aws::String&& value) { SetVocabularyNames(std::move(value)); return *this;}
    inline EngineTranscribeSettings& WithVocabularyNames(const char* value) { SetVocabularyNames(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify the names of the custom vocabulary filters that you want to use when
     * processing your transcription. Note that vocabulary filter names are case
     * sensitive.</p> <p>If you use Amazon Transcribe in multiple Regions, the
     * vocabulary filter must be available in Amazon Transcribe in each Region.</p> <p>
     * If you're <i>not</i> including <code>IdentifyLanguage</code> and want to use a
     * custom vocabulary filter with your transcription, use the
     * <code>VocabularyFilterName</code> parameter instead.</p>
     */
    inline const Aws::String& GetVocabularyFilterNames() const{ return m_vocabularyFilterNames; }
    inline bool VocabularyFilterNamesHasBeenSet() const { return m_vocabularyFilterNamesHasBeenSet; }
    inline void SetVocabularyFilterNames(const Aws::String& value) { m_vocabularyFilterNamesHasBeenSet = true; m_vocabularyFilterNames = value; }
    inline void SetVocabularyFilterNames(Aws::String&& value) { m_vocabularyFilterNamesHasBeenSet = true; m_vocabularyFilterNames = std::move(value); }
    inline void SetVocabularyFilterNames(const char* value) { m_vocabularyFilterNamesHasBeenSet = true; m_vocabularyFilterNames.assign(value); }
    inline EngineTranscribeSettings& WithVocabularyFilterNames(const Aws::String& value) { SetVocabularyFilterNames(value); return *this;}
    inline EngineTranscribeSettings& WithVocabularyFilterNames(Aws::String&& value) { SetVocabularyFilterNames(std::move(value)); return *this;}
    inline EngineTranscribeSettings& WithVocabularyFilterNames(const char* value) { SetVocabularyFilterNames(value); return *this;}
    ///@}
  private:

    TranscribeLanguageCode m_languageCode;
    bool m_languageCodeHasBeenSet = false;

    TranscribeVocabularyFilterMethod m_vocabularyFilterMethod;
    bool m_vocabularyFilterMethodHasBeenSet = false;

    Aws::String m_vocabularyFilterName;
    bool m_vocabularyFilterNameHasBeenSet = false;

    Aws::String m_vocabularyName;
    bool m_vocabularyNameHasBeenSet = false;

    TranscribeRegion m_region;
    bool m_regionHasBeenSet = false;

    bool m_enablePartialResultsStabilization;
    bool m_enablePartialResultsStabilizationHasBeenSet = false;

    TranscribePartialResultsStability m_partialResultsStability;
    bool m_partialResultsStabilityHasBeenSet = false;

    TranscribeContentIdentificationType m_contentIdentificationType;
    bool m_contentIdentificationTypeHasBeenSet = false;

    TranscribeContentRedactionType m_contentRedactionType;
    bool m_contentRedactionTypeHasBeenSet = false;

    Aws::String m_piiEntityTypes;
    bool m_piiEntityTypesHasBeenSet = false;

    Aws::String m_languageModelName;
    bool m_languageModelNameHasBeenSet = false;

    bool m_identifyLanguage;
    bool m_identifyLanguageHasBeenSet = false;

    Aws::String m_languageOptions;
    bool m_languageOptionsHasBeenSet = false;

    TranscribeLanguageCode m_preferredLanguage;
    bool m_preferredLanguageHasBeenSet = false;

    Aws::String m_vocabularyNames;
    bool m_vocabularyNamesHasBeenSet = false;

    Aws::String m_vocabularyFilterNames;
    bool m_vocabularyFilterNamesHasBeenSet = false;
  };

} // namespace Model
} // namespace Chime
} // namespace Aws
