package org.gnu.libebook;

import org.gnu.glib.GObject;
import org.gnu.glib.Handle;

/**
 * This is an object representing a Contact in Evolution.
 * An EContact contains many 'fields'. A field can be of different types
 * for example String, Photo, Address, Name, Date and boolean. There are
 * separate methods in EContact to retrive fields of different types.
 */
public class EContact extends EVCard
{
    public EContact(Handle handle)
    {
        super(handle);
    }

    protected static EContact getEContact(Handle handle)
    {
        if (handle == null)
            return null;

        EContact obj = (EContact) GObject.getGObjectFromHandle(handle);
        return obj == null ? new EContact(handle) : obj;
    }

    /**
     * This will get the value associated with a {@link EContactStringField}
     * @param field
     *              The {@link EContactStringField} you are interesting in getting.
     * @return The string value of an attribute of the EContact
     */
    public String getString(EContactStringField field)
    {
        return e_contact_get_string(getHandle(), field.getFieldEnumId());
    }

    public void setString(EContactStringField field, String value)
    {
        e_contact_set_string(getHandle(), field.getFieldEnumId(), value);
    }

    /**
     * This will get the value associated with a {@link EContactPhoto}
     * @param field
     *              The {@link EContactPhoto} you are interesting in getting.
     * @return An {@link EContactPhoto} attribute of the EContact
     */
    public EContactPhoto getPhoto(EContactPhotoField field)
    {
        return EContactPhoto.getEContactPhoto(e_contact_get_photo(getHandle(), field.getFieldEnumId()));
    }

    public void setPhoto(EContactPhotoField field, EContactPhoto value)
    {
        e_contact_set_photo(getHandle(), field.getFieldEnumId(), value.getHandle());
    }

    /**
     * This will get the value associated with a {@link EContactAddress}
     * @param field
     *              The {@link EContactAddress} you are interesting in getting.
     * @return An {@link EContactAddress} attribute of the EContact
     */
    public EContactAddress getAddress(EContactAddressField field)
    {
        return EContactAddress.getEContactAddress(e_contact_get_address(getHandle(), field.getFieldEnumId()));
    }

    public void setAddress(EContactAddressField field, EContactAddress address)
    {
        e_contact_set_address(getHandle(), field.getFieldEnumId(), address.getHandle());
    }

    /**
     * This will get the value associated with a {@link EContactName}
     * @param field
     *              The {@link EContactName} you are interesting in getting. EContactNameField.NAME is the only valid
     *              parameter here, so it begs the question why you have to pass it in at all!
     * @return An {@link EContactName} attribute of the EContact
     */
    public EContactName getName(EContactNameField field)
    {
        return EContactName.getEContactName(e_contact_get_name(getHandle(), field.getFieldEnumId()));
    }

    public void setName(EContactNameField field, EContactName name)
    {
        e_contact_set_name(getHandle(), field.getFieldEnumId(), name.getHandle());
    }

    /**
     * This will get the value associated with a {@link EContactDate}
     * @param field
     *              The {@link EContactDate} you are interesting in getting.
     * @return An {@link EContactDate} attribute of the EContact
     */
    public EContactDate getDate(EContactDateField field)
    {
        return EContactDate.getEContactDate(e_contact_get_date(getHandle(), field.getFieldEnumId()));
    }

    public void setDate(EContactDateField field, EContactDate date)
    {
        e_contact_set_date(getHandle(), field.getFieldEnumId(), date.getHandle());
    }

    /**
     * This will get the value associated with a {@link EContactBooleanField}
     * @param field
     *              The {@link EContactBooleanField} you are interesting in getting.
     * @return A boolean attribute of the EContact
     */
    public boolean getBoolean(EContactBooleanField field)
    {
        return e_contact_get_boolean(getHandle(), field.getFieldEnumId());
    }

    public void setBoolean(EContactBooleanField field, boolean contactBoolean)
    {
        e_contact_set_boolean(getHandle(), field.getFieldEnumId(), contactBoolean);
    }

    /**
     * This will deep copt the EContact
     * @return A duplicated EContact
     */
    public EContact duplicate()
    {
        return EContact.getEContact(e_contact_duplicate(getHandle()));
    }

    /**
     * Get the EContact's VCard representation
     * @return A String representing the VCard of the EContact
     */
    public String getEContactAsVCard()
    {
        return e_contact_to_vcard_string(getHandle());
    }

    /**
     * This will create a brand new EContact. It will be constructed to represent the incoming vCard
     * @param vCard
     *              A vCard
     * @return A new EContact that has all the attributes of the vCard
     */
    public static EContact createNewFromVCard(String vCard)
    {
        return getEContact(e_contact_new_from_vcard(vCard));
    }

    /**
     * Creates a blank EContact
     * @return A new blank EContact
     */
    public static EContact createNewEContact()
    {
        return getEContact(e_contact_new());
    }

    native static private Handle e_contact_new();

    native static private Handle e_contact_new_from_vcard(String vCard);

    native static private String e_contact_to_vcard_string(Handle handle);

    native static private Handle e_contact_get_address(Handle handle, int fieldId);

    native static private void e_contact_set_address(Handle handle, int fieldId, Handle value);

    native static private Handle e_contact_get_photo(Handle handle, int fieldId);

    native static private void e_contact_set_photo(Handle handle, int fieldId, Handle value);

    native static private Handle e_contact_get_date(Handle handle, int fieldId);

    native static private void e_contact_set_date(Handle handle, int fieldId, Handle value);

    native static private Handle e_contact_get_name(Handle handle, int fieldId);

    native static private void e_contact_set_name(Handle handle, int fieldId, Handle value);

    native static private boolean e_contact_get_boolean(Handle handle, int fieldId);

    native static private void e_contact_set_boolean(Handle handle, int fieldId, boolean value);

    native static private String e_contact_get_string(Handle handle, int fieldId);

    native static private void e_contact_set_string(Handle handle, int fieldId, String value);

    native static private Handle e_contact_duplicate(Handle handle);
}