package org.bouncycastle.crypto.params;

import java.math.BigInteger;

/**
 * Represents the private key parameters for the Elliptic Curve-based Certificateless
 * Signature Infrastructure (ECCSI) scheme as defined in RFC 6507.
 *
 * <p>
 * This class encapsulates the secret signing key (SSK) used in ECCSI. The SSK is generated by
 * the Key Management Service (KMS) and is a random integer modulo the order of the elliptic curve.
 * It is paired with the corresponding public key parameters, represented by an instance of
 * {@link ECCSIPublicKeyParameters}, to form the complete key material required for generating
 * and verifying ECCSI signatures without the use of traditional certificates.
 * </p>
 *
 * <p>
 * Per RFC 6507 Section 5.1:
 * <ul>
 *   <li>The SSK is generated as a random value in the appropriate range.</li>
 *   <li>It is used in conjunction with the public validation token (PVT) to perform signature
 *       operations.</li>
 *   <li>The combination of the SSK and the public key parameters enables certificateless
 *       signature generation and verification.</li>
 * </ul>
 * </p>
 *
 * @see ECCSIPublicKeyParameters
 * @see <a href="https://datatracker.ietf.org/doc/html/rfc6507">
 *      RFC 6507: Elliptic Curve-Based Certificateless Signatures for Identity-Based Encryption (ECCSI)
 *      </a>
 */
public class ECCSIPrivateKeyParameters
    extends AsymmetricKeyParameter
{
    private final BigInteger ssk;
    private final ECCSIPublicKeyParameters pub;

    /**
     * Constructs {@code ECCSIPrivateKeyParameters} with the specified secret signing key
     * and associated public key parameters.
     *
     * @param ssk the secret signing key (SSK) as a BigInteger.
     * @param pub the corresponding public key parameters, which encapsulate the public validation token.
     */
    public ECCSIPrivateKeyParameters(BigInteger ssk, ECCSIPublicKeyParameters pub)
    {
        super(true);
        this.ssk = ssk;
        this.pub = pub;
    }

    /**
     * Returns the public key parameters associated with this private key.
     *
     * @return the {@link ECCSIPublicKeyParameters} containing the public validation token (PVT).
     */
    public ECCSIPublicKeyParameters getPublicKeyParameters()
    {
        return pub;
    }

    /**
     * Returns the secret signing key (SSK) used in ECCSI.
     *
     * @return the SSK as a BigInteger.
     */
    public BigInteger getSSK()
    {
        return ssk;
    }
}
