/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.etrice.generator.base.io;

import com.google.inject.Inject;
import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.LongBuffer;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.UserDefinedFileAttributeView;
import java.util.HashSet;
import java.util.Set;
import java.util.zip.CRC32;
import org.eclipse.etrice.generator.base.io.IGeneratorFileIO;
import org.eclipse.etrice.generator.base.logging.ILogger;
import org.eclipse.etrice.generator.base.logging.NullLogger;
import org.eclipse.xtext.util.RuntimeIOException;

public class GeneratorFileIO
implements IGeneratorFileIO {
    private static final String CHECKSUM_ATTRIBUTE_NAME = "user.crc32";
    private HashSet<Path> generatedFiles;
    private Path outputPath;
    private ILogger logger;

    @Inject
    @Deprecated
    private GeneratorFileIO() {
        this.init("", new NullLogger());
    }

    public GeneratorFileIO(String outputPath, ILogger logger) {
        this.init(outputPath, logger);
    }

    public void init(String outputPath, ILogger logger) {
        this.outputPath = Paths.get(outputPath, new String[0]).normalize();
        this.logger = logger;
        this.generatedFiles = new HashSet();
    }

    public Path getOutputDirectory() {
        return this.outputPath;
    }

    @Override
    public void generateFile(String file, CharSequence content) {
        this.generateFile("generating file", file, content);
    }

    @Override
    public void generateFile(String desc, String file, CharSequence content) {
        String contentStr;
        long checksum;
        Path path = this.getPath(file);
        if (!this.isUnchanged(path, checksum = this.getContentChecksum(contentStr = content.toString()))) {
            this.logger.logInfo(desc + " " + file);
            this.writeFile(path, contentStr);
            this.setChecksumAttribute(path, checksum);
        } else {
            this.logger.logInfo(desc + " (unchanged) " + file);
        }
        this.generatedFiles.add(path);
    }

    public void cleanOutputDirectory() {
        try {
            FileCleaner visitor = new FileCleaner(this.generatedFiles, this.logger);
            Files.walkFileTree(this.outputPath, visitor);
        }
        catch (IOException e) {
            throw new RuntimeIOException((Throwable)e);
        }
    }

    private Path getPath(String file) {
        return this.outputPath.resolve(Paths.get(file, new String[0])).normalize();
    }

    private void writeFile(Path path, String content) {
        try {
            Path parent = path.getParent();
            if (parent != null) {
                Files.createDirectories(parent, new FileAttribute[0]);
            }
            BufferedWriter writer = Files.newBufferedWriter(path, StandardCharsets.UTF_8, new OpenOption[0]);
            writer.append(content);
            writer.close();
        }
        catch (IOException e) {
            throw new RuntimeIOException((Throwable)e);
        }
    }

    private boolean isUnchanged(Path path, long checksum) {
        if (Files.exists(path, new LinkOption[0])) {
            Long oldChecksum = this.getChecksumAttribute(path);
            if (oldChecksum != null && oldChecksum == checksum) {
                return true;
            }
            oldChecksum = this.getFileChecksum(path);
            if (oldChecksum != null && oldChecksum == checksum) {
                return true;
            }
        }
        return false;
    }

    private long getContentChecksum(String content) {
        CRC32 crc = new CRC32();
        crc.update(content.getBytes(StandardCharsets.UTF_8));
        return crc.getValue();
    }

    private Long getChecksumAttribute(Path path) {
        Long checksum = null;
        try {
            UserDefinedFileAttributeView view = Files.getFileAttributeView(path, UserDefinedFileAttributeView.class, new LinkOption[0]);
            if (view != null) {
                ByteBuffer buffer = ByteBuffer.allocate(8);
                view.read(CHECKSUM_ATTRIBUTE_NAME, buffer);
                buffer.flip();
                LongBuffer longBuffer = buffer.asLongBuffer();
                if (longBuffer.hasRemaining()) {
                    checksum = longBuffer.get();
                }
            }
        }
        catch (IOException | ClassCastException | IllegalArgumentException | UnsupportedOperationException e) {
            this.logger.logDebug(e.toString());
        }
        return checksum;
    }

    private void setChecksumAttribute(Path path, long checksum) {
        try {
            UserDefinedFileAttributeView view = Files.getFileAttributeView(path, UserDefinedFileAttributeView.class, new LinkOption[0]);
            if (view != null) {
                ByteBuffer buffer = ByteBuffer.allocate(8);
                buffer.putLong(checksum);
                buffer.flip();
                view.write(CHECKSUM_ATTRIBUTE_NAME, buffer);
            }
        }
        catch (IOException | ClassCastException | IllegalArgumentException | UnsupportedOperationException e) {
            this.logger.logDebug(e.toString());
        }
    }

    private Long getFileChecksum(Path path) {
        Long checksum = null;
        try {
            byte[] bytes = Files.readAllBytes(path);
            CRC32 crc = new CRC32();
            crc.update(bytes);
            checksum = crc.getValue();
        }
        catch (IOException e) {
            this.logger.logDebug(e.toString());
        }
        return checksum;
    }

    @Override
    public String getOutputPath() {
        return this.outputPath.toString();
    }

    @Override
    public void logIntro() {
        this.logger.logInfo("output directory is '" + this.getOutputPath() + "'.");
    }

    private static class FileCleaner
    implements FileVisitor<Path> {
        private Set<Path> excludedFiles;
        private ILogger logger;

        public FileCleaner(Set<Path> excludedFiles, ILogger logger) {
            this.excludedFiles = excludedFiles;
            this.logger = logger;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            if (!this.excludedFiles.contains(file)) {
                Files.delete(file);
                this.logger.logDebug("deleting file " + String.valueOf(file));
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
            return FileVisitResult.CONTINUE;
        }
    }
}

