//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2025 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.typechecker.postchk;

import static org.eclipse.escet.common.java.Lists.listc;
import static org.eclipse.escet.common.java.Maps.map;
import static org.eclipse.escet.common.java.Strings.fmt;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.RegistryFactory;
import org.eclipse.escet.cif.common.CifScopeUtils;
import org.eclipse.escet.cif.metamodel.cif.Specification;
import org.eclipse.escet.cif.metamodel.cif.annotations.AnnotatedObject;
import org.eclipse.escet.cif.metamodel.cif.annotations.Annotation;
import org.eclipse.escet.cif.metamodel.java.CifWalker;
import org.eclipse.escet.cif.typechecker.ErrMsg;
import org.eclipse.escet.cif.typechecker.annotations.AnnotationProblemReporter;
import org.eclipse.escet.cif.typechecker.annotations.AnnotationProvider;
import org.eclipse.escet.cif.typechecker.annotations.DoNothingAnnotationProvider;
import org.eclipse.escet.common.java.Assert;
import org.eclipse.escet.common.java.Strings;
import org.eclipse.escet.common.position.metamodel.position.Position;
import org.eclipse.escet.common.typechecker.SemanticProblemSeverity;
import org.osgi.framework.Bundle;
import org.osgi.framework.wiring.BundleWiring;

/**
 * CIF annotations additional type checking, for the 'post' type checking phase, using {@link AnnotationProvider
 * annotation providers}.
 */
public class CifAnnotationsPostChecker extends CifWalker {
    /** The post check environment to use. */
    private final CifPostCheckEnv env;

    /** Problem reporter for {@link AnnotationProvider annotation providers} to use to report problems. */
    private final ProblemReporter reporter = new ProblemReporter();

    /** Per combined plugin and class name, the instantiated annotation provider. Each value is unique. */
    private final Map<String, AnnotationProvider> createdProviders = map();

    /**
     * Per annotation name, the instantiated annotation provider. An annotation provider may be used for checking
     * several annotation names.
     */
    private final Map<String, AnnotationProvider> annotationProviders = map();

    /**
     * Constructor for the {@link CifAnnotationsPostChecker} class.
     *
     * @param env The post check environment to use.
     */
    public CifAnnotationsPostChecker(CifPostCheckEnv env) {
        this.env = env;
    }

    /**
     * Perform additional type checking on annotations using annotation providers.
     *
     * @param spec The specification to check. The specification must not include any component
     *     definitions/instantiations.
     */
    public void check(Specification spec) {
        // Make sure no component definition/instantiation is present.
        Assert.check(!CifScopeUtils.hasCompDefInst(spec), "Can't post check annotations on spec with comp def/inst.");

        // Find all annotations, and type check them using their corresponding annotation providers.
        walkSpecification(spec);

        // Let each instantiated annotation provider perform additional global type checking on the entire
        // specification.
        for (AnnotationProvider provider: createdProviders.values()) {
            provider.checkGlobal(spec, reporter);
        }
    }

    @Override
    protected void preprocessAnnotatedObject(AnnotatedObject annotatedObj) {
        // Type check each annotation.
        for (Annotation annotation: annotatedObj.getAnnotations()) {
            AnnotationProvider provider = ensureProvider(annotation);
            provider.checkAnnotation(annotatedObj, annotation, reporter);
        }
    }

    /**
     * Find or create an annotation provider for the given annotation.
     *
     * @param annotation The annotation that needs a provider.
     * @return The found or created annotation provider for the given annotation.
     */
    private AnnotationProvider ensureProvider(Annotation annotation) {
        // Try to get an already available annotation provider.
        String annotationName = annotation.getName();
        AnnotationProvider provider = annotationProviders.get(annotationName);
        if (provider != null) {
            return provider;
        }

        // Try to find an extension point for the annotation.
        IConfigurationElement extension = getExtensionPoint(annotationName, annotation.getPosition());
        if (extension == null) {
            // Could not find exactly one valid extension point. Don't try again for the next annotation with the same
            // name.
            provider = new DoNothingAnnotationProvider();
            annotationProviders.put(annotationName, provider);
            return provider;
        }

        // Try to find an already existing annotation provider that can handle the annotation.
        String extensionKey = getExtensionPluginName(extension) + "/" + getExtensionClassName(extension);
        provider = createdProviders.get(extensionKey);
        if (provider != null) {
            annotationProviders.put(annotationName, provider);
            return provider;
        }

        // Try to create an annotation provider from the extension point.
        provider = tryCreateAnnotationProvider(extension, annotation.getPosition());
        if (provider != null) {
            createdProviders.put(extensionKey, provider);
            annotationProviders.put(annotationName, provider);
            return provider;
        }

        // Failed to create a provider. Don't try again for the next annotation with the same name.
        provider = new DoNothingAnnotationProvider();
        annotationProviders.put(annotationName, provider);
        return provider;
    }

    /**
     * Get the extension point for annotations with the given name.
     *
     * @param annotationName The annotation name.
     * @param position The position of the annotation.
     * @return The extension point, or {@code null} if none are found, multiple are found, or a found extension point
     *     doesn't have the required information.
     */
    private IConfigurationElement getExtensionPoint(String annotationName, Position position) {
        // Use extension registry to find registered providers.
        IExtensionRegistry registry = RegistryFactory.getRegistry();
        String extensionPointId = "org.eclipse.escet.cif.annotations";
        IConfigurationElement[] extensions = registry.getConfigurationElementsFor(extensionPointId);

        // Get provider extension point for the requested annotation name.
        List<IConfigurationElement> goodExtensions = listc(1);
        boolean anyIssue = false;
        for (IConfigurationElement extension: extensions) {
            // Skip non-provider extensions.
            if (!"provider".equals(extension.getName())) {
                continue;
            }

            // Skip providers for other annotations.
            if (!annotationName.equals(getExtensionAnnotationName(extension))) {
                continue;
            }

            // Check that the found annotation provider extension has the required information.
            if (!checkAnnotationProviderExtension(extension, position)) {
                anyIssue = true;
                continue;
            }

            // We may be storing an extension here that cannot be constructed. That is fine, since having more than one
            // extension (no matter whether they can be constructed or not) is never allowed.
            goodExtensions.add(extension);
        }

        // If problems were reported earlier, it is unclear what the problem is exactly. Avoid drawing conclusions in
        // that case.
        if (anyIssue) {
            return null;
        }

        // No issues so far. If there is one extension, return it.
        if (goodExtensions.size() == 1) {
            return goodExtensions.get(0);
        }

        // No issues so far and not exactly one provider. Report the found problem.
        if (goodExtensions.isEmpty()) {
            env.addProblem(ErrMsg.ANNO_UNREGISTERED_NAME, position, annotationName);
        } else if (goodExtensions.size() > 1) {
            String names = goodExtensions.stream().map(ext -> fmt("\"%s\"", getExtensionClassName(ext)))
                    .sorted(Strings.SORTER).collect(Collectors.joining(", "));
            env.addProblem(ErrMsg.ANNO_PROVIDER_ERROR, position, annotationName, fmt(
                    "multiple annotation providers are registered for the annotation in the current environment: %s.",
                    names));
        }
        return null;
    }

    /**
     * Check that the annotation provider extension has all the required information.
     *
     * <p>
     * If the check is successful, the {@link #getExtensionContributorName}, {@link #getExtensionAnnotationName},
     * {@link #getExtensionClassName}, and {@link #getExtensionPluginName} methods will not fail to return the
     * requested content of the checked extension.
     * </p>
     *
     * @param extension Extension point to check.
     * @param position Position of the annotation with the annotation name.
     * @return Whether all needed data is available in the extension point.
     */
    private boolean checkAnnotationProviderExtension(IConfigurationElement extension, Position position) {
        // Get the annotation and contributor names. The annotation name has been checked already, the contributor name
        // never fails.
        String annotationName = getExtensionAnnotationName(extension);
        String contributorName = getExtensionContributorName(extension);

        // Get OSGi bundle name.
        String pluginName = getExtensionPluginName(extension);
        if (pluginName == null) {
            env.addProblem(ErrMsg.ANNO_PROVIDER_ERROR, position, annotationName,
                    fmt("annotation provider does not specify a plugin (contributed by \"%s\").", contributorName));
            return false;
        }

        // Get class name.
        String className = getExtensionClassName(extension);
        if (className == null) {
            env.addProblem(ErrMsg.ANNO_PROVIDER_ERROR, position, annotationName,
                    fmt("annotation provider does not specify a class (contributed by \"%s\").", contributorName));
            return false;
        }

        // No problems found.
        return true;
    }

    /**
     * Try to construct an annotation provider with the given information.
     *
     * @param extension Extension point to use. Must have been checked with {@link #checkAnnotationProviderExtension}
     *     already.
     * @param position Position of the annotation name in the CIF specification.
     * @return The constructed provider if all goes well. Otherwise an error is reported and {@code null} is returned.
     */
    private AnnotationProvider tryCreateAnnotationProvider(IConfigurationElement extension, Position position) {
        // Extract the extension point names from the extension.
        String contributorName = getExtensionContributorName(extension);
        String annotationName = getExtensionAnnotationName(extension);
        String pluginName = getExtensionPluginName(extension);
        String className = getExtensionClassName(extension);

        // The requested names should be available (the contributor name never fails).
        Assert.notNull(annotationName);
        Assert.notNull(pluginName);
        Assert.notNull(className);

        // Get OSGi bundle.
        Bundle bundle = Platform.getBundle(pluginName);
        if (bundle == null) {
            env.addProblem(ErrMsg.ANNO_PROVIDER_ERROR, position, annotationName,
                    fmt("annotation provider plugin \"%s\" not found (contributed by \"%s\").", pluginName,
                            contributorName));
            return null;
        }

        // Check bundle state.
        int state = bundle.getState();
        boolean stateOk = state == Bundle.RESOLVED || state == Bundle.STARTING || state == Bundle.ACTIVE;
        if (!stateOk) {
            env.addProblem(ErrMsg.ANNO_PROVIDER_ERROR, position, annotationName,
                    fmt("annotation provider plugin \"%s\" is in a wrong state (state %d, contributed by \"%s\").",
                            pluginName, state, contributorName));
            return null;
        }

        // Get class loader from bundle.
        BundleWiring bundleWiring = bundle.adapt(BundleWiring.class);
        if (bundleWiring == null) {
            env.addProblem(ErrMsg.ANNO_PROVIDER_ERROR, position, annotationName,
                    fmt("annotation provider plugin \"%s\" has no bundle wiring (contributed by \"%s\").",
                            pluginName, contributorName));
            return null;
        }

        ClassLoader classLoader = bundleWiring.getClassLoader();
        if (classLoader == null) {
            env.addProblem(ErrMsg.ANNO_PROVIDER_ERROR, position, annotationName,
                    fmt("annotation provider plugin \"%s\" has no class loader (contributed by \"%s\").",
                            pluginName, contributorName));
            return null;
        }

        // Get class.
        Class<?> cls;
        try {
            cls = classLoader.loadClass(className);
        } catch (ClassNotFoundException e) {
            env.addProblem(ErrMsg.ANNO_PROVIDER_ERROR, position, annotationName,
                    fmt("annotation provider plugin \"%s\" is missing annotation provider class \"%s\" "
                            + "(contributed by \"%s\").", pluginName, className, contributorName));
            return null;
        }

        // Try to construct the provider with the annotation name parameter.
        AnnotationProvider provider = null;
        try {
            provider = (AnnotationProvider)cls.getDeclaredConstructor(String.class).newInstance(annotationName);
        } catch (ReflectiveOperationException e) {
            // Constructor with string parameter failed.
        }

        // If a provider was constructed, return it.
        if (provider != null) {
            return provider;
        }

        // Try to construct the provider without the annotation name parameter.
        try {
            provider = (AnnotationProvider)cls.getDeclaredConstructor().newInstance();
        } catch (ReflectiveOperationException e) {
            env.addProblem(ErrMsg.ANNO_PROVIDER_ERROR, position, annotationName,
                    fmt("annotation provider plugin \"%s\" has an annotation provider class \"%s\" "
                            + "that could not be instantiated with or without annotation name parameter "
                            + "(contributed by \"%s\").", pluginName, className, contributorName));
        }

        // Return the result.
        return provider;
    }

    /**
     * Get the contributor of the annotation provider extension point.
     *
     * @param extension The annotation provider extension to inspect.
     * @return The contributor of the extension.
     */
    private String getExtensionContributorName(IConfigurationElement extension) {
        return extension.getContributor().getName();
    }

    /**
     * Get the name of the annotation of the annotation provider extension point.
     *
     * @param extension The annotation provider extension to inspect.
     * @return The name of the annotation in the extension if it exists. Otherwise, {@code null} is returned.
     */
    private String getExtensionAnnotationName(IConfigurationElement extension) {
        return extension.getAttribute("annotationName");
    }

    /**
     * Get the name of the plugin of the annotation provider extension point.
     *
     * @param extension The annotation provider extension to inspect.
     * @return The name of the plugin in the extension if it exists. Otherwise, {@code null} is returned.
     */
    private String getExtensionPluginName(IConfigurationElement extension) {
        return extension.getAttribute("plugin");
    }

    /**
     * Get the name of the class of the annotation provider extension point.
     *
     * @param extension The annotation provider extension to inspect.
     * @return The name of the class in the extension if it exists. Otherwise, {@code null} is returned.
     */
    private String getExtensionClassName(IConfigurationElement extension) {
        return extension.getAttribute("class");
    }

    /** Problem reporter for {@link AnnotationProvider annotation providers} to use to report problems. */
    private class ProblemReporter implements AnnotationProblemReporter {
        @Override
        public void reportProblem(String annotationName, String message, Position position,
                SemanticProblemSeverity severity)
        {
            ErrMsg msg = switch (severity) {
                case ERROR -> ErrMsg.ANNO_SPECIFIC_ERR;
                case WARNING -> ErrMsg.ANNO_SPECIFIC_WARN;
                default -> throw new AssertionError("Unknown severity: " + severity);
            };
            env.addProblem(msg, position, annotationName, message);
            // Non-fatal problem.
        }
    }
}
