﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/voice-id/VoiceIDRequest.h>
#include <aws/voice-id/VoiceID_EXPORTS.h>
#include <aws/voice-id/model/SpeakerEnrollmentJobStatus.h>

#include <utility>

namespace Aws {
namespace VoiceID {
namespace Model {

/**
 */
class ListSpeakerEnrollmentJobsRequest : public VoiceIDRequest {
 public:
  AWS_VOICEID_API ListSpeakerEnrollmentJobsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListSpeakerEnrollmentJobs"; }

  AWS_VOICEID_API Aws::String SerializePayload() const override;

  AWS_VOICEID_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The identifier of the domain that contains the speaker enrollment jobs.</p>
   */
  inline const Aws::String& GetDomainId() const { return m_domainId; }
  inline bool DomainIdHasBeenSet() const { return m_domainIdHasBeenSet; }
  template <typename DomainIdT = Aws::String>
  void SetDomainId(DomainIdT&& value) {
    m_domainIdHasBeenSet = true;
    m_domainId = std::forward<DomainIdT>(value);
  }
  template <typename DomainIdT = Aws::String>
  ListSpeakerEnrollmentJobsRequest& WithDomainId(DomainIdT&& value) {
    SetDomainId(std::forward<DomainIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Provides the status of your speaker enrollment Job.</p>
   */
  inline SpeakerEnrollmentJobStatus GetJobStatus() const { return m_jobStatus; }
  inline bool JobStatusHasBeenSet() const { return m_jobStatusHasBeenSet; }
  inline void SetJobStatus(SpeakerEnrollmentJobStatus value) {
    m_jobStatusHasBeenSet = true;
    m_jobStatus = value;
  }
  inline ListSpeakerEnrollmentJobsRequest& WithJobStatus(SpeakerEnrollmentJobStatus value) {
    SetJobStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results that are returned per call. You can use
   * <code>NextToken</code> to obtain more pages of results. The default is 100; the
   * maximum allowed page size is also 100.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListSpeakerEnrollmentJobsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If <code>NextToken</code> is returned, there are more results available. The
   * value of <code>NextToken</code> is a unique pagination token for each page. Make
   * the call again using the returned token to retrieve the next page. Keep all
   * other arguments unchanged. Each pagination token expires after 24 hours.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListSpeakerEnrollmentJobsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_domainId;

  SpeakerEnrollmentJobStatus m_jobStatus{SpeakerEnrollmentJobStatus::NOT_SET};

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_domainIdHasBeenSet = false;
  bool m_jobStatusHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace VoiceID
}  // namespace Aws
