﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/quicksight/QuickSightRequest.h>
#include <aws/quicksight/QuickSight_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace QuickSight {
namespace Model {

/**
 */
class DeleteThemeRequest : public QuickSightRequest {
 public:
  AWS_QUICKSIGHT_API DeleteThemeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteTheme"; }

  AWS_QUICKSIGHT_API Aws::String SerializePayload() const override;

  AWS_QUICKSIGHT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The ID of the Amazon Web Services account that contains the theme that you're
   * deleting.</p>
   */
  inline const Aws::String& GetAwsAccountId() const { return m_awsAccountId; }
  inline bool AwsAccountIdHasBeenSet() const { return m_awsAccountIdHasBeenSet; }
  template <typename AwsAccountIdT = Aws::String>
  void SetAwsAccountId(AwsAccountIdT&& value) {
    m_awsAccountIdHasBeenSet = true;
    m_awsAccountId = std::forward<AwsAccountIdT>(value);
  }
  template <typename AwsAccountIdT = Aws::String>
  DeleteThemeRequest& WithAwsAccountId(AwsAccountIdT&& value) {
    SetAwsAccountId(std::forward<AwsAccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An ID for the theme that you want to delete.</p>
   */
  inline const Aws::String& GetThemeId() const { return m_themeId; }
  inline bool ThemeIdHasBeenSet() const { return m_themeIdHasBeenSet; }
  template <typename ThemeIdT = Aws::String>
  void SetThemeId(ThemeIdT&& value) {
    m_themeIdHasBeenSet = true;
    m_themeId = std::forward<ThemeIdT>(value);
  }
  template <typename ThemeIdT = Aws::String>
  DeleteThemeRequest& WithThemeId(ThemeIdT&& value) {
    SetThemeId(std::forward<ThemeIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the theme that you want to delete. </p> <p> <b>Note:</b> If
   * you don't provide a version number, you're using this call to
   * <code>DeleteTheme</code> to delete all versions of the theme.</p>
   */
  inline long long GetVersionNumber() const { return m_versionNumber; }
  inline bool VersionNumberHasBeenSet() const { return m_versionNumberHasBeenSet; }
  inline void SetVersionNumber(long long value) {
    m_versionNumberHasBeenSet = true;
    m_versionNumber = value;
  }
  inline DeleteThemeRequest& WithVersionNumber(long long value) {
    SetVersionNumber(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_awsAccountId;

  Aws::String m_themeId;

  long long m_versionNumber{0};
  bool m_awsAccountIdHasBeenSet = false;
  bool m_themeIdHasBeenSet = false;
  bool m_versionNumberHasBeenSet = false;
};

}  // namespace Model
}  // namespace QuickSight
}  // namespace Aws
