﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/elasticache/ElastiCacheRequest.h>
#include <aws/elasticache/ElastiCache_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ElastiCache {
namespace Model {

/**
 * <p>Represents the input of a <code>DescribeCacheEngineVersions</code>
 * operation.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticache-2015-02-02/DescribeCacheEngineVersionsMessage">AWS
 * API Reference</a></p>
 */
class DescribeCacheEngineVersionsRequest : public ElastiCacheRequest {
 public:
  AWS_ELASTICACHE_API DescribeCacheEngineVersionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeCacheEngineVersions"; }

  AWS_ELASTICACHE_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICACHE_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The cache engine to return. Valid values: <code>memcached</code> |
   * <code>redis</code> </p>
   */
  inline const Aws::String& GetEngine() const { return m_engine; }
  inline bool EngineHasBeenSet() const { return m_engineHasBeenSet; }
  template <typename EngineT = Aws::String>
  void SetEngine(EngineT&& value) {
    m_engineHasBeenSet = true;
    m_engine = std::forward<EngineT>(value);
  }
  template <typename EngineT = Aws::String>
  DescribeCacheEngineVersionsRequest& WithEngine(EngineT&& value) {
    SetEngine(std::forward<EngineT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The cache engine version to return.</p> <p>Example: <code>1.4.14</code> </p>
   */
  inline const Aws::String& GetEngineVersion() const { return m_engineVersion; }
  inline bool EngineVersionHasBeenSet() const { return m_engineVersionHasBeenSet; }
  template <typename EngineVersionT = Aws::String>
  void SetEngineVersion(EngineVersionT&& value) {
    m_engineVersionHasBeenSet = true;
    m_engineVersion = std::forward<EngineVersionT>(value);
  }
  template <typename EngineVersionT = Aws::String>
  DescribeCacheEngineVersionsRequest& WithEngineVersion(EngineVersionT&& value) {
    SetEngineVersion(std::forward<EngineVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of a specific cache parameter group family to return details
   * for.</p> <p>Valid values are: <code>memcached1.4</code> |
   * <code>memcached1.5</code> | <code>memcached1.6</code> | <code>redis2.6</code> |
   * <code>redis2.8</code> | <code>redis3.2</code> | <code>redis4.0</code> |
   * <code>redis5.0</code> | <code>redis6.x</code> | <code>redis6.2</code> |
   * <code>redis7</code> | <code>valkey7</code> </p> <p>Constraints:</p> <ul> <li>
   * <p>Must be 1 to 255 alphanumeric characters</p> </li> <li> <p>First character
   * must be a letter</p> </li> <li> <p>Cannot end with a hyphen or contain two
   * consecutive hyphens</p> </li> </ul>
   */
  inline const Aws::String& GetCacheParameterGroupFamily() const { return m_cacheParameterGroupFamily; }
  inline bool CacheParameterGroupFamilyHasBeenSet() const { return m_cacheParameterGroupFamilyHasBeenSet; }
  template <typename CacheParameterGroupFamilyT = Aws::String>
  void SetCacheParameterGroupFamily(CacheParameterGroupFamilyT&& value) {
    m_cacheParameterGroupFamilyHasBeenSet = true;
    m_cacheParameterGroupFamily = std::forward<CacheParameterGroupFamilyT>(value);
  }
  template <typename CacheParameterGroupFamilyT = Aws::String>
  DescribeCacheEngineVersionsRequest& WithCacheParameterGroupFamily(CacheParameterGroupFamilyT&& value) {
    SetCacheParameterGroupFamily(std::forward<CacheParameterGroupFamilyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of records to include in the response. If more records
   * exist than the specified <code>MaxRecords</code> value, a marker is included in
   * the response so that the remaining results can be retrieved.</p> <p>Default:
   * 100</p> <p>Constraints: minimum 20; maximum 100.</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeCacheEngineVersionsRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional marker returned from a prior request. Use this marker for
   * pagination of results from this operation. If this parameter is specified, the
   * response includes only records beyond the marker, up to the value specified by
   * <code>MaxRecords</code>.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeCacheEngineVersionsRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If <code>true</code>, specifies that only the default version of the
   * specified engine or engine and major version combination is to be returned.</p>
   */
  inline bool GetDefaultOnly() const { return m_defaultOnly; }
  inline bool DefaultOnlyHasBeenSet() const { return m_defaultOnlyHasBeenSet; }
  inline void SetDefaultOnly(bool value) {
    m_defaultOnlyHasBeenSet = true;
    m_defaultOnly = value;
  }
  inline DescribeCacheEngineVersionsRequest& WithDefaultOnly(bool value) {
    SetDefaultOnly(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_engine;

  Aws::String m_engineVersion;

  Aws::String m_cacheParameterGroupFamily;

  int m_maxRecords{0};

  Aws::String m_marker;

  bool m_defaultOnly{false};
  bool m_engineHasBeenSet = false;
  bool m_engineVersionHasBeenSet = false;
  bool m_cacheParameterGroupFamilyHasBeenSet = false;
  bool m_maxRecordsHasBeenSet = false;
  bool m_markerHasBeenSet = false;
  bool m_defaultOnlyHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElastiCache
}  // namespace Aws
