/*
 * ConfigFileHandler.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1999-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, rinaldo, suhler.
 *
 * Version:  2.2
 * Created by suhler on 99/06/28
 * Last modified by suhler on 04/08/30 09:02:07
 */

package sunlabs.brazil.handler;

import sunlabs.brazil.session.SessionManager;
import sunlabs.brazil.server.Handler;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.util.Glob;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.Properties;

/**
 * Handler for manipulating per-user session state that can be
 * represented as ASCII name/value pairs.  The state for each session
 * is stored in a file, which is expected to be in java properties format.
 *
 * If "prefix" is matched, the contents of the (usually cached) config file
 * for the current session is added to the request properties.
 *
 * If the url matches the "set" property, the contents of the config
 * file are changed based on the supplied query parameters (either GET
 * of POST).  If no config file exists for the session, one is created
 * from a default properties file.  Only properties already in the
 * config file may be changed using the "set" method.
 *
 * If a "%" is specified in the file name, it is replaced by the
 * SessionID property, if any, or "common" if sessions aren't used.
 * This should be replaced with something more general, so we can have
 * more arbitrary mappings between request and the session info.
 * <p>
 * The following request properties are used:
 * <dl class=props>
 * <dt>prefix	<dd> The URL prefix required for all documents
 * <dt>set	<dd> The url pattern to match setting properties.
 *		     Currently, it must also match "prefix".
 * <dt>noContent<dd> a url, matching the "set" pattern that causes
 *		     a "204 no content" to be returned  to the client
 *			(experimental).
 * <dt>name	<dd> The name of the config file. the first "%" is replaced
 *		     by the current SessionID.
 * <dt>default	<dd> The default properties file to "seed" session properties
 * <dt>glob	<dd> Properties that match this "glob" pattern may be set
 *		     using the "set" pattern.  If this property is specified,
 *		     the "default" property is optional.
 * <dt>root	<dd> The document root (no properties prefix required). If the
 *		     "name" or "default" properties don't start with a "/", 
 *		     this is used as the current directory.
 *		   
 * </dl>
 * If "%" is specified in the file name, a new session file is 
 * created only if 1) a property is changed from the default, and 2)
 * A cookie was received by the browser.
 * <p>
 * See also: {@link sunlabs.brazil.template.SetTemplate}
 * which is preferrable in most cases, providing a templated based
 * (instead of URL based) 
 * mechanism for maintaining persistent properties.
 *
 * @author		Stephen Uhler
 * @version		2.2, 04/08/30
 */

public class ConfigFileHandler implements Handler {
    static final String SET = "set";   // glob matching url setting props
    static final String PREFIX = "prefix";   // prefix for adding to props
    static final String NAME = "name";     // name of the config file
    static final String DEFAULT = "default";    // the default config file
    static final String ROOT = "root";    // root of config file
    static final String GLOB = "glob";  

    String propsPrefix;
    String name;	// name of the config file
    String urlPrefix;	// prefix for any request we handle
    String set;	// glob pattern to match url needed to modify props
    String nc;	// glob pattern to match url needed for no content
    String match; // glob pattern to match settable props not in deflt file
    String root;	// where to find the root of the config file
    Properties defaultProperties;	// where to find the defaults
    boolean copy = false;	// undocumented backwart compat flag

    /**
     * Make sure default properties exist before starting this handler, 
     * or that "match" is specified".
     */

    public boolean
    init(Server server, String prefix) {
	propsPrefix = prefix;
	urlPrefix = server.props.getProperty(prefix + PREFIX, "/");
	set = server.props.getProperty(prefix + SET, urlPrefix + prefix);
	nc = server.props.getProperty(prefix + "noContent", "X");
	match = server.props.getProperty(prefix + GLOB);
	name = server.props.getProperty(prefix + NAME, prefix + ".cfg");
	copy = (server.props.getProperty(prefix + "copy") != null);
	root = server.props.getProperty(ROOT, ".");
	String defaultName = server.props.getProperty(prefix + DEFAULT);
	File defaultFile = null;
	if (defaultName != null) {
	    if (defaultName.startsWith("/")) {
		defaultFile = new File(defaultName);
	    } else {
		defaultFile = new File(root, defaultName);
	    }
	}
	defaultProperties = new Properties();
	try {
	    FileInputStream in = new FileInputStream(defaultFile);
	    defaultProperties.load(in);
	    in.close();
	} catch (IOException e) {
	    if (match == null) {
		server.log(Server.LOG_WARNING, prefix, "No default file: " +
		defaultFile + " ConfigFileHandler NOT installed!");	
		return false;
	    }
	}
	server.log(Server.LOG_DIAGNOSTIC, prefix, "\n  set=" + set +
		"\n  name=" + name + "\n  default=" + defaultFile);
	return true;
    }

    /**
     * Extract the session state into the request object, optionally
     * modifying the properties.  If the properties are modified,
     * they are stored in a file for safe keeping.
     */

    public boolean
    respond(Request request) throws IOException {
	if (!request.url.startsWith(urlPrefix)) {
	    return false;
	}

	/*
	 * Get the existing properties.
	 */

	root = request.props.getProperty(ROOT, root);
	String id = request.props.getProperty("SessionID", "common");
	Properties p = (Properties) SessionManager.getSession(id,
		propsPrefix, Properties.class);

	/*
	 * If no properties exist, read in the session
	 * properties, otherwise use the default properties
	 */

	if (p.isEmpty()) {
	    File file = getPropsFile(name, id);
	    request.log(Server.LOG_DIAGNOSTIC, propsPrefix,
		    "No properties, looking in: " + file);
	    try {
		FileInputStream in = new FileInputStream(file);
		p.load(in);
		in.close();
	    } catch (IOException e) {
		request.log(Server.LOG_DIAGNOSTIC, propsPrefix,
		    "Can't find config file: " +
		    file + " using default properties");	
		Enumeration enumer = defaultProperties.propertyNames();
		while(enumer.hasMoreElements()) {
		    String key = (String) enumer.nextElement();
		    p.put(key, defaultProperties.getProperty(key));
		}
	    }
	}

	/*
	 * If "SET" then update the properties file
	 */

	if (Glob.match(set, request.url)) {
	    Dictionary update = request.getQueryData(null);
	    // System.out.println("Setting from: " + update);
	    Enumeration enumer = update.keys();
	    boolean changed = false;
	    synchronized (p) {
		while(enumer.hasMoreElements()) {
		    String key = (String) enumer.nextElement();
		    if (p.containsKey(key) || isMatch(key)) {
			p.put(key, update.get(key));
			changed = true; // XXX not quite
		    } else {
			request.log(Server.LOG_DIAGNOSTIC, propsPrefix,
			    "Can't set key " + key +
			    ", name doesn't exist in default file" +
			    " or doesn;t match: " + match);
		    }
		}
		if (!changed) {
		    request.log(Server.LOG_DIAGNOSTIC, propsPrefix,
			    "Set called, nothing changed!!");
		}

		/*
		 * If the client has cookies enabled, try to save the
		 * properties.  This depends upon a side effect of the
		 * "sessionHandler" to let us know if the browser
		 * actually sent us our cookie.
		 */

		if (changed && (name.indexOf("%") < 0 ||
			request.props.containsKey("UrlID") ||
			request.props.containsKey("gotCookie"))) {
		    File file = getPropsFile(name, id);
		    try {
			FileOutputStream out = new FileOutputStream(file);
			p.save(out, request.serverUrl() + request.url +
			    " (from ConfigFileHandler: " + propsPrefix + ")");
			out.close();
			request.log(Server.LOG_DIAGNOSTIC, propsPrefix,
			    "Saving configuration properties to " + file);
		    } catch (IOException e) {
			request.log(Server.LOG_WARNING, propsPrefix,
				"Can't save properties to: " + file);
		    }
		}
	    }

	    /*
	     * If this is a post, trash the input and change to a get.
	     * XXX This is probably incorrect.
	     */

	    if (request.method.equals("POST")) {
		request.method = "GET";
		request.postData = null;
		request.headers.put("Content-Length", "0");
	    }	
	}

	if (copy) {
	    Enumeration enumer = p.keys();
	    while(enumer.hasMoreElements()) {
		String key = (String) enumer.nextElement();
		request.props.put(key, p.getProperty(key));
	    }
	} else {
	    request.addSharedProps(p);
//	    request.log(Server.LOG_DIAGNOSTIC, propsPrefix, "Chaining " +
//		    p + " onto request.props");
	}
	return false;
    }

    /**
     * True if key "s" matches glob pattern in "match".  If
     * match is null, then no match.
     */

    private boolean
    isMatch(String s) {
	return (match != null && Glob.match(match, s));
    }

    /**
     * Find the config file name
     */

    private File
    getPropsFile(String name, String id) {
	int index;
	if ((index = name.indexOf("%")) >= 0) {
	    name = name.substring(0,index) + id + name.substring(index+1);
	}
	File file;
	if (name.startsWith("/")) {
	    file = new File(name);
	} else {
	    file = new File(root, name);
	}
	return file;
    }
}
