/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.maven.plugin.compiler;

import javax.tools.DiagnosticListener;
import javax.tools.JavaCompiler;
import javax.tools.JavaFileObject;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.Writer;
import java.lang.module.ModuleDescriptor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.maven.api.JavaPathType;
import org.apache.maven.api.PathType;
import org.apache.maven.api.ProjectScope;

import static org.apache.maven.plugin.compiler.AbstractCompilerMojo.SUPPORT_LEGACY;

/**
 * A task which configures and executes the Java compiler for the test classes.
 * This executor contains additional configurations compared to the base class.
 *
 * @author Martin Desruisseaux
 */
class ToolExecutorForTest extends ToolExecutor {
    /**
     * The output directory of the main classes.
     * This directory will be added to the class-path or module-path.
     *
     * @see TestCompilerMojo#mainOutputDirectory
     */
    protected final Path mainOutputDirectory;

    /**
     * Path to the {@code module-info.class} file of the main code, or {@code null} if that file does not exist.
     */
    private final Path mainModulePath;

    /**
     * Whether to place the main classes on the module path when {@code module-info} is present.
     * The default and recommended value is {@code true}. The user may force to {@code false},
     * in which case the main classes are placed on the class path, but this is deprecated.
     * This flag may be removed in a future version if we remove support of this practice.
     *
     * @deprecated Use {@code "claspath-jar"} dependency type instead, and avoid {@code module-info.java} in tests.
     *
     * @see TestCompilerMojo#useModulePath
     */
    @Deprecated(since = "4.0.0")
    private final boolean useModulePath;

    /**
     * Whether a {@code module-info.java} file is defined in the test sources.
     * In such case, it has precedence over the {@code module-info.java} in main sources.
     * This is defined for compatibility with Maven 3, but not recommended.
     *
     * @deprecated Avoid {@code module-info.java} in tests.
     */
    @Deprecated(since = "4.0.0")
    private final boolean hasTestModuleInfo;

    /**
     * Whether the tests are declared in their own module. If {@code true},
     * then the {@code module-info.java} file of the test declares a name
     * different than the {@code module-info.java} file of the main code.
     */
    private boolean testInItsOwnModule;

    /**
     * Whether the {@code module-info} of the tests overwrites the main {@code module-info}.
     * This is a deprecated practice, but is accepted if {@link #SUPPORT_LEGACY} is true.
     */
    private boolean overwriteMainModuleInfo;

    /**
     * Name of the main module to compile, or {@code null} if not yet determined.
     * If the project is not modular, then this field contains an empty string.
     *
     * TODO: use "*" as a sentinel value for modular source hierarchy.
     *
     * @see #getMainModuleName()
     */
    private String moduleName;

    /**
     * Whether {@link #addModuleOptions(Options)} has already been invoked.
     * The options shall be completed only once, otherwise conflicts may occur.
     */
    private boolean addedModuleOptions;

    /**
     * If non-null, the {@code module} part to remove in {@code target/test-classes/module/package}.
     * This {@code module} directory is generated by {@code javac} for some compiler options.
     * We keep that directory when the project is configured with the new {@code <source>} element,
     * but have to remove it for compatibility reason if the project is compiled in the old way.
     *
     * @deprecated Exists only for compatibility with the Maven 3 way to do a modular project.
     * Is likely to cause confusion, for example with incremental builds.
     * New projects should use the {@code <source>} elements instead.
     */
    @Deprecated(since = "4.0.0")
    private String directoryLevelToRemove;

    /**
     * Creates a new task by taking a snapshot of the current configuration of the given <abbr>MOJO</abbr>.
     * This constructor creates the {@linkplain #outputDirectory output directory} if it does not already exist.
     *
     * @param mojo the <abbr>MOJO</abbr> from which to take a snapshot
     * @param listener where to send compilation warnings, or {@code null} for the Maven logger
     * @throws MojoException if this constructor identifies an invalid parameter in the <abbr>MOJO</abbr>
     * @throws IOException if an error occurred while creating the output directory or scanning the source directories
     */
    @SuppressWarnings("deprecation")
    ToolExecutorForTest(TestCompilerMojo mojo, DiagnosticListener<? super JavaFileObject> listener) throws IOException {
        super(mojo, listener);
        mainOutputDirectory = mojo.mainOutputDirectory;
        mainModulePath = mojo.mainModulePath;
        useModulePath = mojo.useModulePath;
        hasTestModuleInfo = mojo.hasTestModuleInfo;
        /*
         * If we are compiling the test classes of a modular project, add the `--patch-modules` options.
         * In this case, the option values are directories of main class files of the patched module.
         */
        final var patchedModules = new LinkedHashMap<String, Set<Path>>();
        for (SourceDirectory dir : sourceDirectories) {
            String moduleToPatch = dir.moduleName;
            if (moduleToPatch == null) {
                moduleToPatch = getMainModuleName();
                if (moduleToPatch.isEmpty()) {
                    continue; // No module-info found.
                }
                if (SUPPORT_LEGACY) {
                    String testModuleName = mojo.getTestModuleName(sourceDirectories);
                    if (testModuleName != null) {
                        overwriteMainModuleInfo = testModuleName.equals(getMainModuleName());
                        if (!overwriteMainModuleInfo) {
                            testInItsOwnModule = true;
                            continue; // The test classes are in their own module.
                        }
                    }
                }
                directoryLevelToRemove = moduleToPatch;
            }
            patchedModules.put(moduleToPatch, new LinkedHashSet<>()); // Signal that this module exists in the test.
        }
        /*
         * The values of `patchedModules` are empty lists. Now, add the real paths to
         * main class for each module that exists in both the main code and the test.
         */
        mojo.getSourceRoots(ProjectScope.MAIN).forEach((root) -> {
            root.module().ifPresent((moduleToPatch) -> {
                Set<Path> paths = patchedModules.get(moduleToPatch);
                if (paths != null) {
                    Path path = root.targetPath().orElseGet(() -> Path.of(moduleToPatch));
                    path = mainOutputDirectory.resolve(path);
                    paths.add(path);
                }
            });
        });
        patchedModules.values().removeIf(Set::isEmpty);
        patchedModules.forEach((moduleToPatch, paths) -> {
            dependencies(JavaPathType.patchModule(moduleToPatch)).addAll(paths);
        });
        /*
         * If there is no module to patch, we probably have a non-modular project.
         * In such case, we need to put the main output directory on the classpath.
         * It may also be a modular project not declared in the `<source>` element.
         */
        if (patchedModules.isEmpty() && Files.exists(mainOutputDirectory)) {
            PathType pathType = JavaPathType.CLASSES;
            if (hasModuleDeclaration) {
                pathType = JavaPathType.MODULES;
                if (!testInItsOwnModule) {
                    String moduleToPatch = getMainModuleName();
                    if (!moduleToPatch.isEmpty()) {
                        pathType = JavaPathType.patchModule(moduleToPatch);
                        directoryLevelToRemove = moduleToPatch;
                    }
                }
            }
            prependDependency(pathType, mainOutputDirectory);
        }
    }

    /**
     * {@return the module name of the main code, or an empty string if none}
     * This method reads the module descriptor when first needed and caches the result.
     * This used if the user did not specified an explicit {@code <module>} element in the sources.
     *
     * @throws IOException if the module descriptor cannot be read.
     */
    private String getMainModuleName() throws IOException {
        if (moduleName == null) {
            if (mainModulePath != null) {
                try (InputStream in = Files.newInputStream(mainModulePath)) {
                    moduleName = ModuleDescriptor.read(in).name();
                }
            } else {
                moduleName = "";
            }
        }
        return moduleName;
    }

    /**
     * If the given module name is empty, tries to infer a default module name.
     */
    @Override
    final String inferModuleNameIfMissing(String moduleName) throws IOException {
        return (!testInItsOwnModule && moduleName.isEmpty()) ? getMainModuleName() : moduleName;
    }

    /**
     * Completes the given configuration with module options the first time that this method is invoked.
     * If at least one {@value ModuleInfoPatch#FILENAME} file is found in a root directory of test sources,
     * then these files are parsed and the options that they declare are added to the given configuration.
     * Otherwise, if {@link #hasModuleDeclaration} is {@code true}, then this method generates the
     * {@code --add-modules} and {@code --add-reads} options for dependencies that are not in the main compilation.
     * If this method is invoked more than once, all invocations after the first one have no effect.
     *
     * @param configuration where to add the options
     * @throws IOException if the module information of a dependency or the module-info patch cannot be read
     */
    @SuppressWarnings({"checkstyle:MissingSwitchDefault", "fallthrough"})
    private void addModuleOptions(final Options configuration) throws IOException {
        if (addedModuleOptions) {
            return;
        }
        addedModuleOptions = true;
        ModuleInfoPatch info = null;
        ModuleInfoPatch defaultInfo = null;
        final var patches = new LinkedHashMap<String, ModuleInfoPatch>();
        for (SourceDirectory source : sourceDirectories) {
            Path file = source.root.resolve(ModuleInfoPatch.FILENAME);
            String module;
            if (Files.notExists(file)) {
                if (SUPPORT_LEGACY && useModulePath && hasTestModuleInfo && hasModuleDeclaration) {
                    /*
                     * Do not add any `--add-reads` parameters. The developers should put
                     * everything needed in the `module-info`, including test dependencies.
                     */
                    continue;
                }
                /*
                 * No `patch-module-info` file. Generate a default module patch instance for the
                 * `--add-modules TEST-MODULE-PATH` and `--add-reads TEST-MODULE-PATH` options.
                 * We generate that patch only for the first module. If there is more modules
                 * without `patch-module-info`, we will copy the `defaultInfo` instance.
                 */
                module = source.moduleName;
                if (module == null) {
                    module = getMainModuleName();
                    if (module.isEmpty()) {
                        continue;
                    }
                }
                if (defaultInfo != null) {
                    patches.putIfAbsent(module, null); // Remember that we will need to compute a value later.
                    continue;
                }
                defaultInfo = new ModuleInfoPatch(module, info);
                defaultInfo.setToDefaults();
                info = defaultInfo;
            } else {
                info = new ModuleInfoPatch(getMainModuleName(), info);
                try (BufferedReader reader = Files.newBufferedReader(file)) {
                    info.load(reader);
                }
                module = info.getModuleName();
            }
            if (patches.put(module, info) != null) {
                throw new ModuleInfoPatchException("\"module-info-patch " + module + "\" is defined more than once.");
            }
        }
        /*
         * Replace all occurrences of `TEST-MODULE-PATH` by the actual dependency paths.
         * Add `--add-modules` and `--add-reads` options with default values equivalent to
         * `TEST-MODULE-PATH` for every module that do not have a `module-info-patch` file.
         */
        for (Map.Entry<String, ModuleInfoPatch> entry : patches.entrySet()) {
            info = entry.getValue();
            if (info != null) {
                info.replaceProjectModules(sourceDirectories);
                info.replaceTestModulePath(dependencyResolution);
            } else {
                // `defaultInfo` cannot be null if this `info` value is null.
                entry.setValue(defaultInfo.patchWithSameReads(entry.getKey()));
            }
        }
        /*
         * Write the runtime dependencies in the `META-INF/maven/module-info-patch.args` file.
         * Note that we unconditionally write in the root output directory, not in the module directory,
         * because a single option file applies to all modules.
         */
        if (!patches.isEmpty()) {
            Path directory = // TODO: replace by Path.resolve(String, String...) with JDK22.
                    Files.createDirectories(outputDirectory.resolve("META-INF").resolve("maven"));
            try (BufferedWriter out = Files.newBufferedWriter(directory.resolve("module-info-patch.args"))) {
                for (ModuleInfoPatch m : patches.values()) {
                    m.writeTo(configuration, out);
                }
            }
        }
    }

    /**
     * @hidden
     */
    @Override
    public boolean applyIncrementalBuild(AbstractCompilerMojo mojo, Options configuration) throws IOException {
        addModuleOptions(configuration); // Effective only once.
        return super.applyIncrementalBuild(mojo, configuration);
    }

    /**
     * @hidden
     */
    @Override
    public boolean compile(JavaCompiler compiler, Options configuration, Writer otherOutput) throws IOException {
        addModuleOptions(configuration); // Effective only once.
        try (var r = ModuleDirectoryRemover.create(outputDirectory, directoryLevelToRemove)) {
            return super.compile(compiler, configuration, otherOutput);
        }
    }

    /**
     * Separates the compilation of {@code module-info} from other classes. This is needed when the
     * {@code module-info} of the test classes overwrite the {@code module-info} of the main classes.
     * In the latter case, we need to compile the test {@code module-info} first in order to substitute
     * the main module-info by the test one before to compile the remaining test classes.
     */
    @Override
    final CompilationTaskSources[] toCompilationTasks(final SourcesForRelease unit) {
        if (!(SUPPORT_LEGACY && useModulePath && hasTestModuleInfo && overwriteMainModuleInfo)) {
            return super.toCompilationTasks(unit);
        }
        CompilationTaskSources moduleInfo = null;
        final List<Path> files = unit.files;
        for (int i = files.size(); --i >= 0; ) {
            if (SourceDirectory.isModuleInfoSource(files.get(i))) {
                moduleInfo = new CompilationTaskSources(List.of(files.remove(i)));
                if (files.isEmpty()) {
                    return new CompilationTaskSources[] {moduleInfo};
                }
                break;
            }
        }
        if (files.isEmpty()) {
            return new CompilationTaskSources[0];
        }
        var task = new CompilationTaskSources(files) {
            /**
             * Substitutes the main {@code module-info.class} by the test's one, compiles test classes,
             * then restores the original {@code module-info.class}. The test {@code module-info.class}
             * must have been compiled separately before this method is invoked.
             */
            @Override
            boolean compile(JavaCompiler.CompilationTask task) throws IOException {
                try (unit) {
                    unit.substituteModuleInfos(mainOutputDirectory, outputDirectory);
                    return super.compile(task);
                }
            }
        };
        if (moduleInfo != null) {
            return new CompilationTaskSources[] {moduleInfo, task};
        } else {
            return new CompilationTaskSources[] {task};
        }
    }
}
