/*
 * Decompiled with CFR 0.152.
 */
package org.apache.helix.manager.zk;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableMap;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.apache.helix.AccessOption;
import org.apache.helix.BucketDataAccessor;
import org.apache.helix.HelixException;
import org.apache.helix.HelixProperty;
import org.apache.helix.manager.zk.ZkBaseDataAccessor;
import org.apache.helix.msdcommon.exception.InvalidRoutingDataException;
import org.apache.helix.zookeeper.api.client.HelixZkClient;
import org.apache.helix.zookeeper.api.client.RealmAwareZkClient;
import org.apache.helix.zookeeper.datamodel.ZNRecord;
import org.apache.helix.zookeeper.datamodel.serializer.ByteArraySerializer;
import org.apache.helix.zookeeper.datamodel.serializer.ZNRecordJacksonSerializer;
import org.apache.helix.zookeeper.impl.client.FederatedZkClient;
import org.apache.helix.zookeeper.impl.factory.DedicatedZkClientFactory;
import org.apache.helix.zookeeper.util.GZipCompressionUtil;
import org.apache.helix.zookeeper.zkclient.DataUpdater;
import org.apache.helix.zookeeper.zkclient.exception.ZkNoNodeException;
import org.apache.helix.zookeeper.zkclient.serialize.ZkSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZkBucketDataAccessor
implements BucketDataAccessor,
AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(ZkBucketDataAccessor.class);
    private static final int DEFAULT_BUCKET_SIZE = 51200;
    private static final long DEFAULT_VERSION_TTL = TimeUnit.MINUTES.toMillis(1L);
    private static final String BUCKET_SIZE_KEY = "BUCKET_SIZE";
    private static final String DATA_SIZE_KEY = "DATA_SIZE";
    private static final String METADATA_KEY = "METADATA";
    private static final String LAST_SUCCESSFUL_WRITE_KEY = "LAST_SUCCESSFUL_WRITE";
    private static final String LAST_WRITE_KEY = "LAST_WRITE";
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private static final ScheduledExecutorService GC_THREAD = Executors.newSingleThreadScheduledExecutor(runnable -> {
        Thread thread = new Thread(runnable, "ZkBucketDataAccessorGcThread");
        thread.setDaemon(true);
        return thread;
    });
    private final int _bucketSize;
    private final long _versionTTLms;
    private final ZkSerializer _zkSerializer;
    private final RealmAwareZkClient _zkClient;
    private final ZkBaseDataAccessor<byte[]> _zkBaseDataAccessor;
    private final Map<String, ScheduledFuture> _gcTaskFutureMap = new ConcurrentHashMap<String, ScheduledFuture>();
    private boolean _usesExternalZkClient = false;

    public ZkBucketDataAccessor(String zkAddr, int bucketSize, long versionTTLms) {
        this(ZkBucketDataAccessor.createRealmAwareZkClient(zkAddr), bucketSize, versionTTLms, false);
    }

    public ZkBucketDataAccessor(RealmAwareZkClient zkClient) {
        this(zkClient, 51200, DEFAULT_VERSION_TTL, true);
    }

    public ZkBucketDataAccessor(RealmAwareZkClient zkClient, int bucketSize, long versionTTLms) {
        this(zkClient, bucketSize, versionTTLms, true);
    }

    private ZkBucketDataAccessor(RealmAwareZkClient zkClient, int bucketSize, long versionTTLms, boolean usesExternalZkClient) {
        this._zkClient = zkClient;
        this._zkBaseDataAccessor = new ZkBaseDataAccessor(this._zkClient);
        this._zkSerializer = new ZNRecordJacksonSerializer();
        this._bucketSize = bucketSize;
        this._versionTTLms = versionTTLms;
        this._usesExternalZkClient = usesExternalZkClient;
    }

    public ZkBucketDataAccessor(String zkAddr) {
        this(zkAddr, 51200, DEFAULT_VERSION_TTL);
    }

    private static RealmAwareZkClient createRealmAwareZkClient(String zkAddr) {
        RealmAwareZkClient zkClient;
        if (Boolean.getBoolean("helix.multiZkEnabled") || zkAddr == null) {
            LOG.warn("ZkBucketDataAccessor: either multi-zk enabled or zkAddr is null - starting ZkBucketDataAccessor in multi-zk mode!");
            try {
                RealmAwareZkClient.RealmAwareZkConnectionConfig connectionConfig = new RealmAwareZkClient.RealmAwareZkConnectionConfig.Builder().build();
                RealmAwareZkClient.RealmAwareZkClientConfig clientConfig = new RealmAwareZkClient.RealmAwareZkClientConfig();
                zkClient = new FederatedZkClient(connectionConfig, clientConfig);
            }
            catch (IllegalArgumentException | InvalidRoutingDataException e) {
                throw new HelixException("Not able to connect on realm-aware mode", e);
            }
        } else {
            zkClient = DedicatedZkClientFactory.getInstance().buildZkClient(new HelixZkClient.ZkConnectionConfig(zkAddr));
        }
        zkClient.setZkSerializer(new ByteArraySerializer());
        return zkClient;
    }

    @Override
    public <T extends HelixProperty> boolean compressedBucketWrite(String rootPath, T value) throws IOException {
        boolean[] success;
        DataUpdater<byte[]> lastWriteVersionUpdater = dataInZk -> {
            if (dataInZk == null || ((byte[])dataInZk).length == 0) {
                return "0".getBytes();
            }
            String lastWriteVersionStr = new String((byte[])dataInZk);
            long lastWriteVersion = Long.parseLong(lastWriteVersionStr);
            return String.valueOf(++lastWriteVersion).getBytes();
        };
        ZkBaseDataAccessor.AccessResult result = this._zkBaseDataAccessor.doUpdate(rootPath + "/LAST_WRITE", lastWriteVersionUpdater, AccessOption.PERSISTENT);
        if (result._retCode != ZkBaseDataAccessor.RetCode.OK) {
            throw new HelixException(String.format("Failed to write the write version at path: %s!", rootPath));
        }
        byte[] binaryVersion = (byte[])result._updatedValue;
        String versionStr = new String(binaryVersion);
        long version = Long.parseLong(versionStr);
        String versionedDataPath = rootPath + "/" + versionStr;
        byte[] serializedRecord = this._zkSerializer.serialize(value.getRecord());
        byte[] compressedRecord = GZipCompressionUtil.compress(serializedRecord);
        int numBuckets = (compressedRecord.length + this._bucketSize - 1) / this._bucketSize;
        ArrayList<String> paths = new ArrayList<String>();
        ArrayList<byte[]> buckets = new ArrayList<byte[]>();
        int ptr = 0;
        for (int counter = 0; counter < numBuckets; ++counter) {
            paths.add(versionedDataPath + "/" + counter);
            if (counter == numBuckets - 1) {
                buckets.add(Arrays.copyOfRange(compressedRecord, ptr, ptr + compressedRecord.length % this._bucketSize));
            } else {
                buckets.add(Arrays.copyOfRange(compressedRecord, ptr, ptr + this._bucketSize));
            }
            ptr += this._bucketSize;
        }
        ImmutableMap metadata = ImmutableMap.of((Object)BUCKET_SIZE_KEY, (Object)Integer.toString(this._bucketSize), (Object)DATA_SIZE_KEY, (Object)Integer.toString(compressedRecord.length));
        byte[] binaryMetadata = OBJECT_MAPPER.writeValueAsBytes((Object)metadata);
        paths.add(versionedDataPath + "/METADATA");
        buckets.add(binaryMetadata);
        for (boolean s : success = this._zkBaseDataAccessor.setChildren(paths, buckets, AccessOption.PERSISTENT)) {
            if (s) continue;
            throw new HelixException(String.format("Failed to write the data buckets for path: %s", rootPath));
        }
        DataUpdater<byte[]> lastSuccessfulWriteVersionUpdater = dataInZk -> {
            if (dataInZk == null || ((byte[])dataInZk).length == 0) {
                return versionStr.getBytes();
            }
            String lastWriteVersionStr = new String((byte[])dataInZk);
            long lastWriteVersion = Long.parseLong(lastWriteVersionStr);
            if (lastWriteVersion < version) {
                return versionStr.getBytes();
            }
            return null;
        };
        if (!this._zkBaseDataAccessor.update(rootPath + "/LAST_SUCCESSFUL_WRITE", lastSuccessfulWriteVersionUpdater, AccessOption.PERSISTENT)) {
            throw new HelixException(String.format("Failed to write the last successful write metadata at path: %s!", rootPath));
        }
        this.scheduleStaleVersionGC(rootPath);
        return true;
    }

    @Override
    public <T extends HelixProperty> HelixProperty compressedBucketRead(String path, Class<T> helixPropertySubType) {
        return (HelixProperty)helixPropertySubType.cast(this.compressedBucketRead(path));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void compressedBucketDelete(String path) {
        if (!this._zkBaseDataAccessor.remove(path, AccessOption.PERSISTENT)) {
            throw new HelixException(String.format("Failed to delete the bucket data! Path: %s", path));
        }
        ZkBucketDataAccessor zkBucketDataAccessor = this;
        synchronized (zkBucketDataAccessor) {
            this._gcTaskFutureMap.remove(path);
        }
    }

    @Override
    public void disconnect() {
        if (!this._usesExternalZkClient && this._zkClient != null && !this._zkClient.isClosed()) {
            this._zkClient.close();
        }
    }

    private HelixProperty compressedBucketRead(String path) {
        byte[] serializedRecord;
        Map metadata;
        String versionToRead = this.getLastSuccessfulWriteVersion(path);
        byte[] binaryMetadata = this._zkBaseDataAccessor.get(path + "/" + versionToRead + "/METADATA", null, AccessOption.PERSISTENT);
        if (binaryMetadata == null) {
            throw new ZkNoNodeException(String.format("Metadata ZNode does not exist for path: %s", path));
        }
        try {
            metadata = (Map)OBJECT_MAPPER.readValue(binaryMetadata, Map.class);
        }
        catch (IOException e) {
            throw new HelixException(String.format("Failed to deserialize path metadata: %s!", path), e);
        }
        Object bucketSizeObj = metadata.get(BUCKET_SIZE_KEY);
        Object dataSizeObj = metadata.get(DATA_SIZE_KEY);
        if (bucketSizeObj == null) {
            throw new HelixException(String.format("Metadata ZNRecord does not have %s! Path: %s", BUCKET_SIZE_KEY, path));
        }
        if (dataSizeObj == null) {
            throw new HelixException(String.format("Metadata ZNRecord does not have %s! Path: %s", DATA_SIZE_KEY, path));
        }
        int bucketSize = Integer.parseInt((String)bucketSizeObj);
        int dataSize = Integer.parseInt((String)dataSizeObj);
        int numBuckets = (dataSize + this._bucketSize - 1) / this._bucketSize;
        byte[] compressedRecord = new byte[dataSize];
        String dataPath = path + "/" + versionToRead;
        ArrayList<String> paths = new ArrayList<String>();
        for (int i = 0; i < numBuckets; ++i) {
            paths.add(dataPath + "/" + i);
        }
        List<byte[]> buckets = this._zkBaseDataAccessor.get(paths, null, AccessOption.PERSISTENT, true);
        int copyPtr = 0;
        for (int i = 0; i < numBuckets; ++i) {
            if (i == numBuckets - 1) {
                System.arraycopy(buckets.get(i), 0, compressedRecord, copyPtr, dataSize % bucketSize);
                continue;
            }
            System.arraycopy(buckets.get(i), 0, compressedRecord, copyPtr, bucketSize);
            copyPtr += bucketSize;
        }
        ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(compressedRecord);
        try {
            serializedRecord = GZipCompressionUtil.uncompress(byteArrayInputStream);
        }
        catch (IOException e) {
            throw new HelixException(String.format("Failed to decompress path: %s!", path), e);
        }
        ZNRecord originalRecord = (ZNRecord)this._zkSerializer.deserialize(serializedRecord);
        return new HelixProperty(originalRecord);
    }

    @Override
    public void close() {
        this.disconnect();
    }

    public void finalize() {
        this._zkBaseDataAccessor.close();
        this.close();
    }

    private synchronized void scheduleStaleVersionGC(String rootPath) {
        if (this._gcTaskFutureMap.containsKey(rootPath)) {
            return;
        }
        this._gcTaskFutureMap.put(rootPath, GC_THREAD.schedule(() -> {
            try {
                this._gcTaskFutureMap.remove(rootPath);
                this.deleteStaleVersions(rootPath);
            }
            catch (Exception ex) {
                LOG.error("Failed to delete the stale versions.", (Throwable)ex);
            }
        }, this._versionTTLms, TimeUnit.MILLISECONDS));
    }

    private void deleteStaleVersions(String rootPath) {
        String currentVersionStr = this.getLastSuccessfulWriteVersion(rootPath);
        List<String> children = this._zkBaseDataAccessor.getChildNames(rootPath, AccessOption.PERSISTENT);
        if (children == null || children.isEmpty()) {
            return;
        }
        List<String> pathsToDelete = this.getPathsToDelete(rootPath, this.filterChildrenNames(children, Long.parseLong(currentVersionStr)));
        for (String pathToDelete : pathsToDelete) {
            this._zkBaseDataAccessor.remove(pathToDelete, AccessOption.PERSISTENT);
        }
    }

    private List<String> filterChildrenNames(List<String> childrenNodes, long currentVersion) {
        ArrayList<String> childrenToRemove = new ArrayList<String>();
        for (String child : childrenNodes) {
            long childVer;
            if (child.equals(LAST_SUCCESSFUL_WRITE_KEY) || child.equals(LAST_WRITE_KEY)) continue;
            try {
                childVer = Long.parseLong(child);
            }
            catch (NumberFormatException ex) {
                LOG.warn("Found an invalid ZNode: {}", (Object)child);
                continue;
            }
            if (childVer >= currentVersion) continue;
            childrenToRemove.add(child);
        }
        return childrenToRemove;
    }

    private List<String> getPathsToDelete(String path, List<String> staleVersions) {
        ArrayList<String> pathsToDelete = new ArrayList<String>();
        staleVersions.forEach(ver -> pathsToDelete.add(path + "/" + ver));
        return pathsToDelete;
    }

    private String getLastSuccessfulWriteVersion(String path) {
        byte[] binaryVersionToRead = this._zkBaseDataAccessor.get(path + "/LAST_SUCCESSFUL_WRITE", null, AccessOption.PERSISTENT);
        if (binaryVersionToRead == null) {
            throw new ZkNoNodeException(String.format("Last successful write ZNode does not exist for path: %s", path));
        }
        return new String(binaryVersionToRead);
    }
}

