/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.configuration;

import java.io.File;
import java.time.Duration;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.common.time.Time;
import org.apache.flink.configuration.CheckpointingOptions;
import org.apache.flink.configuration.ConfigOption;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.CoreOptions;
import org.apache.flink.configuration.GlobalConfiguration;
import org.apache.flink.configuration.JobManagerOptions;
import org.apache.flink.configuration.MemorySize;
import org.apache.flink.configuration.MetricOptions;
import org.apache.flink.configuration.ResourceManagerOptions;
import org.apache.flink.configuration.StructuredOptionsSplitter;
import org.apache.flink.configuration.TaskManagerOptions;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.TimeUtils;

public class ConfigurationUtils {
    private static final String[] EMPTY = new String[0];

    public static Optional<Time> getSystemResourceMetricsProbingInterval(Configuration configuration) {
        if (!configuration.getBoolean(MetricOptions.SYSTEM_RESOURCE_METRICS)) {
            return Optional.empty();
        }
        return Optional.of(Time.milliseconds(configuration.getLong(MetricOptions.SYSTEM_RESOURCE_METRICS_PROBING_INTERVAL)));
    }

    @Nonnull
    public static String[] parseTempDirectories(Configuration configuration) {
        return ConfigurationUtils.splitPaths(configuration.getString(CoreOptions.TMP_DIRS));
    }

    @Nonnull
    public static File getRandomTempDirectory(Configuration configuration) {
        String[] tmpDirectories = ConfigurationUtils.parseTempDirectories(configuration);
        Preconditions.checkState(tmpDirectories.length > 0, String.format("No temporary directory has been specified for %s", CoreOptions.TMP_DIRS.key()));
        int randomIndex = ThreadLocalRandom.current().nextInt(tmpDirectories.length);
        return new File(tmpDirectories[randomIndex]);
    }

    @Nonnull
    public static String[] parseLocalStateDirectories(Configuration configuration) {
        String configValue = configuration.getString(CheckpointingOptions.LOCAL_RECOVERY_TASK_MANAGER_STATE_ROOT_DIRS, "");
        return ConfigurationUtils.splitPaths(configValue);
    }

    public static Map<String, String> parseMap(String stringSerializedMap) {
        return StructuredOptionsSplitter.splitEscaped(stringSerializedMap, ',').stream().map(p -> StructuredOptionsSplitter.splitEscaped(p, ':')).collect(Collectors.toMap(arr -> (String)arr.get(0), arr -> (String)arr.get(1)));
    }

    public static Time getStandaloneClusterStartupPeriodTime(Configuration configuration) {
        long standaloneClusterStartupPeriodTime = configuration.getLong(ResourceManagerOptions.STANDALONE_CLUSTER_STARTUP_PERIOD_TIME);
        Time timeout = standaloneClusterStartupPeriodTime >= 0L ? Time.milliseconds(standaloneClusterStartupPeriodTime) : Time.milliseconds(configuration.getLong(JobManagerOptions.SLOT_REQUEST_TIMEOUT));
        return timeout;
    }

    @Nonnull
    public static Configuration createConfiguration(Properties properties) {
        Configuration configuration = new Configuration();
        Set<String> propertyNames = properties.stringPropertyNames();
        for (String propertyName : propertyNames) {
            configuration.setString(propertyName, properties.getProperty(propertyName));
        }
        return configuration;
    }

    @Nonnull
    public static Map<String, String> hideSensitiveValues(Map<String, String> keyValuePairs) {
        HashMap<String, String> result = new HashMap<String, String>();
        for (Map.Entry<String, String> keyValuePair : keyValuePairs.entrySet()) {
            if (GlobalConfiguration.isSensitive(keyValuePair.getKey())) {
                result.put(keyValuePair.getKey(), "******");
                continue;
            }
            result.put(keyValuePair.getKey(), keyValuePair.getValue());
        }
        return result;
    }

    @Nonnull
    public static String[] splitPaths(@Nonnull String separatedPaths) {
        return separatedPaths.length() > 0 ? separatedPaths.split(",|" + File.pathSeparator) : EMPTY;
    }

    public static String assembleDynamicConfigsStr(Map<String, String> config) {
        return config.entrySet().stream().map(e -> String.format("-D %s=%s", e.getKey(), e.getValue())).collect(Collectors.joining(" "));
    }

    @VisibleForTesting
    public static Map<String, String> parseTmResourceDynamicConfigs(String dynamicConfigsStr) {
        HashMap<String, String> configs = new HashMap<String, String>();
        String[] configStrs = dynamicConfigsStr.split(" ");
        Preconditions.checkArgument(configStrs.length % 2 == 0, "Dynamic option string contained odd number of arguments: #arguments=%s, (%s)", configStrs.length, dynamicConfigsStr);
        for (int i = 0; i < configStrs.length; ++i) {
            String configStr = configStrs[i];
            if (i % 2 == 0) {
                Preconditions.checkArgument(configStr.equals("-D"));
                continue;
            }
            String[] configKV = configStr.split("=");
            Preconditions.checkArgument(configKV.length == 2);
            configs.put(configKV[0], configKV[1]);
        }
        ConfigurationUtils.checkConfigContains(configs, TaskManagerOptions.CPU_CORES.key());
        ConfigurationUtils.checkConfigContains(configs, TaskManagerOptions.FRAMEWORK_HEAP_MEMORY.key());
        ConfigurationUtils.checkConfigContains(configs, TaskManagerOptions.FRAMEWORK_OFF_HEAP_MEMORY.key());
        ConfigurationUtils.checkConfigContains(configs, TaskManagerOptions.TASK_HEAP_MEMORY.key());
        ConfigurationUtils.checkConfigContains(configs, TaskManagerOptions.TASK_OFF_HEAP_MEMORY.key());
        ConfigurationUtils.checkConfigContains(configs, TaskManagerOptions.NETWORK_MEMORY_MAX.key());
        ConfigurationUtils.checkConfigContains(configs, TaskManagerOptions.NETWORK_MEMORY_MIN.key());
        ConfigurationUtils.checkConfigContains(configs, TaskManagerOptions.MANAGED_MEMORY_SIZE.key());
        ConfigurationUtils.checkConfigContains(configs, TaskManagerOptions.JVM_METASPACE.key());
        ConfigurationUtils.checkConfigContains(configs, TaskManagerOptions.JVM_OVERHEAD_MIN.key());
        ConfigurationUtils.checkConfigContains(configs, TaskManagerOptions.JVM_OVERHEAD_MAX.key());
        ConfigurationUtils.checkConfigContains(configs, TaskManagerOptions.NUM_TASK_SLOTS.key());
        return configs;
    }

    private static void checkConfigContains(Map<String, String> configs, String key) {
        Preconditions.checkArgument(configs.containsKey(key), "Key %s is missing present from dynamic configs.", key);
    }

    @VisibleForTesting
    public static Map<String, String> parseJvmArgString(String jvmParamsStr) {
        String xmx = "-Xmx";
        String xms = "-Xms";
        String maxDirect = "-XX:MaxDirectMemorySize=";
        String maxMetadata = "-XX:MaxMetaspaceSize=";
        HashMap<String, String> configs = new HashMap<String, String>();
        for (String paramStr : jvmParamsStr.split(" ")) {
            if (paramStr.startsWith("-Xmx")) {
                configs.put("-Xmx", paramStr.substring("-Xmx".length()));
                continue;
            }
            if (paramStr.startsWith("-Xms")) {
                configs.put("-Xms", paramStr.substring("-Xms".length()));
                continue;
            }
            if (paramStr.startsWith("-XX:MaxDirectMemorySize=")) {
                configs.put("-XX:MaxDirectMemorySize=", paramStr.substring("-XX:MaxDirectMemorySize=".length()));
                continue;
            }
            if (!paramStr.startsWith("-XX:MaxMetaspaceSize=")) continue;
            configs.put("-XX:MaxMetaspaceSize=", paramStr.substring("-XX:MaxMetaspaceSize=".length()));
        }
        Preconditions.checkArgument(configs.containsKey("-Xmx"));
        Preconditions.checkArgument(configs.containsKey("-Xms"));
        Preconditions.checkArgument(configs.containsKey("-XX:MaxMetaspaceSize="));
        return configs;
    }

    public static Map<String, String> getPrefixedKeyValuePairs(String prefix, Configuration configuration) {
        HashMap<String, String> result = new HashMap<String, String>();
        for (Map.Entry<String, String> entry : configuration.toMap().entrySet()) {
            if (!entry.getKey().startsWith(prefix) || entry.getKey().length() <= prefix.length()) continue;
            String key = entry.getKey().substring(prefix.length());
            result.put(key, entry.getValue());
        }
        return result;
    }

    public static <T> T convertValue(Object rawValue, Class<?> clazz) {
        if (Integer.class.equals(clazz)) {
            return (T)ConfigurationUtils.convertToInt(rawValue);
        }
        if (Long.class.equals(clazz)) {
            return (T)ConfigurationUtils.convertToLong(rawValue);
        }
        if (Boolean.class.equals(clazz)) {
            return (T)ConfigurationUtils.convertToBoolean(rawValue);
        }
        if (Float.class.equals(clazz)) {
            return (T)ConfigurationUtils.convertToFloat(rawValue);
        }
        if (Double.class.equals(clazz)) {
            return (T)ConfigurationUtils.convertToDouble(rawValue);
        }
        if (String.class.equals(clazz)) {
            return (T)ConfigurationUtils.convertToString(rawValue);
        }
        if (clazz.isEnum()) {
            return (T)ConfigurationUtils.convertToEnum(rawValue, clazz);
        }
        if (clazz == Duration.class) {
            return (T)ConfigurationUtils.convertToDuration(rawValue);
        }
        if (clazz == MemorySize.class) {
            return (T)ConfigurationUtils.convertToMemorySize(rawValue);
        }
        if (clazz == Map.class) {
            return (T)ConfigurationUtils.convertToProperties(rawValue);
        }
        throw new IllegalArgumentException("Unsupported type: " + clazz);
    }

    static <T> T convertToList(Object rawValue, Class<?> atomicClass) {
        if (rawValue instanceof List) {
            return (T)rawValue;
        }
        return (T)StructuredOptionsSplitter.splitEscaped(rawValue.toString(), ';').stream().map(s -> ConfigurationUtils.convertValue(s, atomicClass)).collect(Collectors.toList());
    }

    static Map<String, String> convertToProperties(Object o) {
        if (o instanceof Map) {
            return (Map)o;
        }
        List<String> listOfRawProperties = StructuredOptionsSplitter.splitEscaped(o.toString(), ',');
        return listOfRawProperties.stream().map(s -> StructuredOptionsSplitter.splitEscaped(s, ':')).peek(pair -> {
            if (pair.size() != 2) {
                throw new IllegalArgumentException("Could not parse pair in the map " + pair);
            }
        }).collect(Collectors.toMap(a -> (String)a.get(0), a -> (String)a.get(1)));
    }

    static <E extends Enum<?>> E convertToEnum(Object o, Class<E> clazz) {
        if (o.getClass().equals(clazz)) {
            return (E)((Enum)o);
        }
        return (E)Arrays.stream(clazz.getEnumConstants()).filter(e -> e.toString().toUpperCase(Locale.ROOT).equals(o.toString().toUpperCase(Locale.ROOT))).findAny().orElseThrow(() -> new IllegalArgumentException(String.format("Could not parse value for enum %s. Expected one of: [%s]", clazz, Arrays.toString(clazz.getEnumConstants()))));
    }

    static Duration convertToDuration(Object o) {
        if (o.getClass() == Duration.class) {
            return (Duration)o;
        }
        return TimeUtils.parseDuration(o.toString());
    }

    static MemorySize convertToMemorySize(Object o) {
        if (o.getClass() == MemorySize.class) {
            return (MemorySize)o;
        }
        return MemorySize.parse(o.toString());
    }

    static String convertToString(Object o) {
        if (o.getClass() == String.class) {
            return (String)o;
        }
        if (o.getClass() == Duration.class) {
            Duration duration = (Duration)o;
            return TimeUtils.formatWithHighestUnit(duration);
        }
        if (o instanceof List) {
            return ((List)o).stream().map(e -> StructuredOptionsSplitter.escapeWithSingleQuote(ConfigurationUtils.convertToString(e), ";")).collect(Collectors.joining(";"));
        }
        if (o instanceof Map) {
            return ((Map)o).entrySet().stream().map(e -> {
                String escapedKey = StructuredOptionsSplitter.escapeWithSingleQuote(e.getKey().toString(), ":");
                String escapedValue = StructuredOptionsSplitter.escapeWithSingleQuote(e.getValue().toString(), ":");
                return StructuredOptionsSplitter.escapeWithSingleQuote(escapedKey + ":" + escapedValue, ",");
            }).collect(Collectors.joining(","));
        }
        return o.toString();
    }

    static Integer convertToInt(Object o) {
        if (o.getClass() == Integer.class) {
            return (Integer)o;
        }
        if (o.getClass() == Long.class) {
            long value = (Long)o;
            if (value <= Integer.MAX_VALUE && value >= Integer.MIN_VALUE) {
                return (int)value;
            }
            throw new IllegalArgumentException(String.format("Configuration value %s overflows/underflows the integer type.", value));
        }
        return Integer.parseInt(o.toString());
    }

    static Long convertToLong(Object o) {
        if (o.getClass() == Long.class) {
            return (Long)o;
        }
        if (o.getClass() == Integer.class) {
            return ((Integer)o).longValue();
        }
        return Long.parseLong(o.toString());
    }

    static Boolean convertToBoolean(Object o) {
        if (o.getClass() == Boolean.class) {
            return (Boolean)o;
        }
        switch (o.toString().toUpperCase()) {
            case "TRUE": {
                return true;
            }
            case "FALSE": {
                return false;
            }
        }
        throw new IllegalArgumentException(String.format("Unrecognized option for boolean: %s. Expected either true or false(case insensitive)", o));
    }

    static Float convertToFloat(Object o) {
        if (o.getClass() == Float.class) {
            return (Float)o;
        }
        if (o.getClass() == Double.class) {
            double value = (Double)o;
            if (value == 0.0 || value >= (double)1.4E-45f && value <= 3.4028234663852886E38 || value >= -3.4028234663852886E38 && value <= (double)-1.4E-45f) {
                return Float.valueOf((float)value);
            }
            throw new IllegalArgumentException(String.format("Configuration value %s overflows/underflows the float type.", value));
        }
        return Float.valueOf(Float.parseFloat(o.toString()));
    }

    static Double convertToDouble(Object o) {
        if (o.getClass() == Double.class) {
            return (Double)o;
        }
        if (o.getClass() == Float.class) {
            return ((Float)o).doubleValue();
        }
        return Double.parseDouble(o.toString());
    }

    public static boolean canBePrefixMap(ConfigOption<?> configOption) {
        return configOption.getClazz() == Map.class && !configOption.isList();
    }

    public static boolean filterPrefixMapKey(String key, String candidate) {
        String prefixKey = key + ".";
        return candidate.startsWith(prefixKey);
    }

    static Map<String, String> convertToPropertiesPrefixed(Map<String, Object> confData, String key) {
        String prefixKey = key + ".";
        return confData.keySet().stream().filter(k -> k.startsWith(prefixKey)).collect(Collectors.toMap(k -> k.substring(prefixKey.length()), k -> ConfigurationUtils.convertToString(confData.get(k))));
    }

    static boolean containsPrefixMap(Map<String, Object> confData, String key) {
        return confData.keySet().stream().anyMatch(candidate -> ConfigurationUtils.filterPrefixMapKey(key, candidate));
    }

    static boolean removePrefixMap(Map<String, Object> confData, String key) {
        List<String> prefixKeys = confData.keySet().stream().filter(candidate -> ConfigurationUtils.filterPrefixMapKey(key, candidate)).collect(Collectors.toList());
        prefixKeys.forEach(confData::remove);
        return !prefixKeys.isEmpty();
    }

    private ConfigurationUtils() {
    }
}

