/******************************************************************************
 * (c) Copyright 2002,2003, 1060 Research Ltd
 *
 * This Software is licensed to You, the licensee, for use under the terms of
 * the 1060 Public License v1.0. Please read and agree to the 1060 Public
 * License v1.0 [www.1060research.com/license] before using or redistributing
 * this software.
 *
 * In summary the 1060 Public license has the following conditions.
 * A. You may use the Software free of charge provided you agree to the terms
 * laid out in the 1060 Public License v1.0
 * B. You are only permitted to use the Software with components or applications
 * that provide you with OSI Certified Open Source Code [www.opensource.org], or
 * for which licensing has been approved by 1060 Research Limited.
 * You may write your own software for execution by this Software provided any
 * distribution of your software with this Software complies with terms set out
 * in section 2 of the 1060 Public License v1.0
 * C. You may redistribute the Software provided you comply with the terms of
 * the 1060 Public License v1.0 and that no warranty is implied or given.
 * D. If you find you are unable to comply with this license you may seek to
 * obtain an alternative license from 1060 Research Limited by contacting
 * license@1060research.com or by visiting www.1060research.com
 *
 * NO WARRANTY:  THIS SOFTWARE IS NOT COVERED BY ANY WARRANTY. SEE 1060 PUBLIC
 * LICENSE V1.0 FOR DETAILS
 *
 * THIS COPYRIGHT NOTICE IS *NOT* THE 1060 PUBLIC LICENSE v1.0. PLEASE READ
 * THE DISTRIBUTED 1060_Public_License.txt OR www.1060research.com/license
 *
 * File:          $RCSfile: FilesystemCache.java,v $
 * Version:       $Name:  $ $Revision: 1.4 $
 * Last Modified: $Date: 2005/03/14 12:38:36 $
 *****************************************************************************/
package org.ten60.netkernel.cache.filesystem;

import com.ten60.netkernel.cache.ICachelet;
import com.ten60.netkernel.container.Container;
import com.ten60.netkernel.module.ModuleDefinition;
import com.ten60.netkernel.scheduler.Scheduler;
import com.ten60.netkernel.urii.*;
import com.ten60.netkernel.urii.aspect.IAspectBinaryStream;
import com.ten60.netkernel.urrequest.URRequest;
import com.ten60.netkernel.urrequest.URResult;
import com.ten60.netkernel.util.SysLogger;
import com.ten60.netkernel.util.NetKernelException;

import org.ten60.netkernel.xml.util.XMLUtils;
import org.ten60.netkernel.xml.xda.*;
import org.ten60.netkernel.layer1.representation.MonoRepresentationImpl;
import org.ten60.netkernel.layer1.representation.ByteArrayAspect;

import org.ten60.netkernel.cache.*;

import java.io.*;
import java.net.*;
import java.util.*;
import org.w3c.dom.*;


/**
 * Cache to filesystem
 * @author  tab
 */
public final class FilesystemCache implements ICachelet
{
	private int mWorkThreshold=1000;
	private int mMaximumSize=1000;
	private int mPruneSize=250;
	private File mBaseDir;
	private Container mContainer;
	private Scheduler mScheduler;
	
	private Map mCacheTable;
	
	/** Creates a new instance of NullCache */
	public FilesystemCache()
	{
	}
	
	public void init(Container aContainer, ModuleDefinition aModule)
	{	mContainer=aContainer;
		try
		{	URL configURL=aModule.getResource("/etc/FilesystemCacheConfig.xml");
			configURL.openConnection();
			InputStream configIS=configURL.openStream();
			Document configDoc=XMLUtils.parse(new InputStreamReader(configIS));
			DOMXDA config = new DOMXDA(configDoc,false);
			URI filesystemPath = URI.create(config.getText("filesystemPath",true));
			URI baseDir = URI.create(aContainer.getBasePathURI()).resolve(filesystemPath);
			mBaseDir = new File(baseDir);
			mBaseDir.mkdirs();
			
			mWorkThreshold = Integer.parseInt(config.getText("representationWorkThreshold",true));
			mMaximumSize = Integer.parseInt(config.getText("cacheSize",true));
			mPruneSize = Integer.parseInt(config.getText("pruneSize",true));
		}
		catch (Throwable e)
		{	SysLogger.log1(SysLogger.WARNING, this, "Error during FilesystemCache initialisation: %1",e.toString());
			URI filesystemPath = URI.create("cache/");
			URI baseDir = URI.create(aContainer.getBasePathURI()).resolve(filesystemPath);
			mBaseDir = new File(baseDir);
			mBaseDir.mkdirs();
		}
                finally
                {   reset();
                }
	}
	
	private void reset()
	{	File[] files=mBaseDir.listFiles();
		for (int i=0; i<files.length; i++)
		{	files[i].delete();
		}
		mCacheTable = new HashMap(mMaximumSize);
	}
	
	public void put(URResult aResult)
	{	URRequest request = aResult.getRequest();
		int type = request.getType();
		if (type==URRequest.RQT_SOURCE)
		{	IURMeta meta = aResult.getResource().getMeta();
			if (!(meta.isIntermediate() || meta.isExpired() || request.getArgs().size()>0) && meta.getCreationCost()>=mWorkThreshold)
			{	
				// check if we need to prune first
				if (mCacheTable.size()>=mMaximumSize)
				{	prune();
				}
				
				CacheKey key = new CacheKey(request,meta.isContextSensitive());
				//remove existing
				synchronized(mCacheTable)
				{	FileCacheEntry entry = (FileCacheEntry)mCacheTable.remove(key);
					if (entry!=null)
					{	File f = new File(mBaseDir,entry.getFilename());
						f.delete();
					}
				}
				//add new result
				File file = generateUniqueFilename();
				try
				{	IAspectBinaryStream bin = ensureBinary(aResult);
					OutputStream os = new BufferedOutputStream(new FileOutputStream(file));
					bin.write(os);
					int length = (int)file.length();
					FileCacheEntry fce = new FileCacheEntry(file.getName(),meta,key,length);
					synchronized(mCacheTable)
					{	mCacheTable.put(key,fce);
					}
				}
				catch (NetKernelException e)
				{	/* failed to transrept- cannot cache this */
				}
				catch (Exception e)
				{	e.printStackTrace();
					SysLogger.log1(SysLogger.WARNING, this, "Error during FilesystemCache put: %1",e.toString());
				}
			}
		}
		else if (type==URRequest.RQT_SINK || type==URRequest.RQT_DELETE)
		{	// an entry has potentially been invalidated
			CacheKey key = new CacheKey(request,false);
			synchronized(mCacheTable)
			{	FileCacheEntry entry = (FileCacheEntry)mCacheTable.remove(key);
				if (entry!=null)
				{	File f = new File(mBaseDir,entry.getFilename());
					f.delete();
				}
			}
		}
	}
	
	
	private File generateUniqueFilename()
	{	File result;
		do
		{	String name = Long.toHexString(Double.doubleToLongBits(Math.random())&0xFFFFFF);
			result = new File(mBaseDir,name);
			if (!result.exists())
			{	break;
			}
		} while (true);
		return result;
	}
	
	private IAspectBinaryStream ensureBinary(URResult aResult) throws NetKernelException
	{	IURRepresentation rep = aResult.getResource();
		IAspectBinaryStream result;
		if (!rep.hasAspect(IAspectBinaryStream.class))
		{	URRequest parent = aResult.getRequest();
			URRequest request = new URRequest(parent.getURI(), null, parent.getSession(), parent.getContext(), URRequest.RQT_TRANSREPRESENT, parent.getCWU(), parent, IAspectBinaryStream.class);
			request.addArg(URRequest.URI_SYSTEM, aResult.getResource());
			mScheduler = (Scheduler)mContainer.getComponent(Scheduler.URI);
			rep = mScheduler.requestSynch(request).getResource();
		}
		result = (IAspectBinaryStream)rep.getAspect(IAspectBinaryStream.class);
		return result;
	}
	
	public IURRepresentation get(URRequest aRequest)
	{	IURRepresentation result=null;
		FileCacheEntry entry;
		CacheKey key = new CacheKey(aRequest,true);
		synchronized(mCacheTable)
		{	entry = (FileCacheEntry)mCacheTable.get(key);
		}
		if (entry!=null)
		{	IURMeta meta = entry.getMeta();
			File file = new File(mBaseDir,entry.getFilename());
			if (!meta.isExpired())
			{	// retrieve entry - copy into memory array
				entry.touch();
                try
                {   ByteArrayOutputStream baos = new ByteArrayOutputStream((int)file.length());
                    org.ten60.netkernel.layer1.util.Utils.pipe(new FileInputStream(file),baos);
                    ByteArrayAspect aspect=new ByteArrayAspect(baos);
                    result = new MonoRepresentationImpl(meta,aspect);
                } catch (IOException e)
                {   SysLogger.log2(SysLogger.WARNING, this, "Failed to retrieve %1 from cache: %2", aRequest.getURI().toString(), e.toString());
                }
			}
			else
			{	// delete it if expired
				synchronized(mCacheTable)
				{	mCacheTable.remove(key);
					file.delete();
				}
			}
		}
		return result;
	}
	
	public void prune()
	{	synchronized(mCacheTable)
		{	
			ArrayList ts=new ArrayList(mCacheTable.size());
			for (Iterator i =mCacheTable.values().iterator(); i.hasNext(); )
			{	FileCacheEntry entry=(FileCacheEntry)i.next();
				IURMeta meta = entry.getMeta();
				if(meta.isExpired())
				{	// remove now
					i.remove();
					File file = new File(mBaseDir,entry.getFilename());
					file.delete();
				}
				else
				{	// add to list of cull candidates
					ts.add(entry);
				}
			}
			// sort by touch time
			Comparator comparator = new Comparator()
			{	public int compare(Object aO1, Object aO2)
				{	FileCacheEntry o1 = (FileCacheEntry)aO1;
					FileCacheEntry o2 = (FileCacheEntry)aO2;
					return (int)((o2.getTouchtime()-o1.getTouchtime()));
				}
			};
			Collections.sort(ts, comparator);
			
			// delete oldest entries
			int i2 = mMaximumSize-mPruneSize;
			for (int i=ts.size()-1; i>=i2; i--)
			{	FileCacheEntry entry = (FileCacheEntry)ts.get(i);
				mCacheTable.remove(entry.getKey());
				File file = new File(mBaseDir,entry.getFilename());
				file.delete();
			}
		}
	}
	
	public void write(Writer aWriter) throws IOException
	{	synchronized(mCacheTable)
		{	long now = System.currentTimeMillis();
			for (Iterator i = mCacheTable.entrySet().iterator(); i.hasNext(); )
			{	aWriter.write("<file>");
				Map.Entry entry = (Map.Entry)i.next();
				CacheKey key = (CacheKey)entry.getKey();
				FileCacheEntry cr=(FileCacheEntry)entry.getValue();
				aWriter.write("<key>");
				aWriter.write(XMLUtils.escape(key.getURI().toString()));
				aWriter.write("</key>");
				aWriter.write("<module>");
				aWriter.write(XMLUtils.escape(key.getModule().toString()));
				aWriter.write("</module>");
				aWriter.write("<age>");
				aWriter.write(Long.toString(now-cr.getTouchtime()));
				aWriter.write("</age>");
				aWriter.write("<cost>");
				aWriter.write(Integer.toString(cr.getMeta().getCreationCost()));
				aWriter.write("</cost>");
				aWriter.write("<size>");
				aWriter.write(Integer.toString(cr.getLength()));
				aWriter.write("</size>");
				aWriter.write("</file>");
			}
		}
	}

	public ICachelet getBackingCache()
	{	return null;
	}	
	
}